///
/// Pandora connection parameter implementation.
///	@file		pandoraparameters.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-03-17
///	@copyright	Copyright (c) 2015-2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cassert>

#include "utility.h"
#include "pandora.h"

using namespace std;

namespace Pandora {
    /// Names for persistence keys
    namespace Key {
        const char *const Username = "username";
        const char *const Password = "password";
        const char *const Proxy = "proxy";
        const char *const ControlProxy = "controlProxy";
        const char *const CacheMinimum = "cacheMinimum";
        const char *const CacheMaximum = "cacheMaximum";
    }  // namespace Key

    ConnectionParameters::ConnectionParameters (void) : SourceParameters (Ownership::Type::SHARED){};

    /** Create new connection parameters, retrieving values from user settings.
        @param src A user settings dictionary to retrieve settings from.
        @throw invalid_argument if values in the settings dictionary are invalid. */
    ConnectionParameters::ConnectionParameters (const UserData::StringDictionary &src) : SourceParameters (src) {
        username = src.get (Key::Username, "");
        if (username.empty())
            throw invalid_argument ("username not recoverable");
        password = lamer_cipher (username, src.get (Key::Password, ""));
        if (password.empty())
            throw invalid_argument ("user password missing");
        proxy = src.get (Key::Proxy, "");
        control_proxy = src.get (Key::ControlProxy, "");

        cache_minimum = stoi (src.get (Key::CacheMinimum, "4000"));
        if (cache_minimum < 800 || cache_minimum >= 1000000)
            throw invalid_argument ("Invalid cacheMinimum");
        
        cache_maximum = stoi (src.get (Key::CacheMaximum, "4800"));
        if (cache_maximum < 1000 || cache_maximum >= 1000000)
            throw invalid_argument ("Invalid cacheMaximum");

        if (cache_minimum + 100 > cache_maximum)
            throw invalid_argument ("Invalid cache min/max settings");
    }

    /// Copy connection parameters into a user settings dictionary.
    bool ConnectionParameters::persist (UserData::StringDictionary &dest) const {
        dest.add (Key::Username, username);
        dest.add (Key::Password, lamer_cipher (username, password));
        dest.add (Key::Proxy, proxy);
        dest.add (Key::ControlProxy, control_proxy);
        dest.add (Key::CacheMinimum, to_string (cache_minimum));
        dest.add (Key::CacheMaximum, to_string (cache_maximum));
        return SourceParameters::persist (dest);
    }
}  // namespace Pandora
