///
/// Pandora messages.
/// @file       mediaunits/pandora/pandoramessages.cpp - pianod project
/// @author     Perette Barella
/// @date       2020-03-24
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <string>
#include <memory>

#include <parsnip.h>

#include "musictypes.h"
#include "pandoratypes.h"

namespace Pandora {
    class Communication;
    class Song;
    class Artist;
    class Station;

    extern const std::string Type_Track;
    extern const std::string Type_Album;
    extern const std::string Type_Artist;
    extern const std::string Type_Station;
    extern const std::string Type_StationFactory;

    /** Pandora request class.  This provides a framework for assembling
        requests and decoding responses. */
    class Request {
        friend Communication;

    private:
        /// Where there this request will be routed to within the REST API.
        const std::string endpoint;

    protected:
        Source *source;

        Request (Source *src, const char *ep);

        ThingieList extractAssortedThingsFromDictionary (const Parsnip::Data &message) const;

    public:
        virtual bool debug() const;

        /** Convert request into a message as a Parsnip Data object.
            @return Data containing serializable form of request. */
        virtual Parsnip::Data retrieveRequestMessage() const = 0;

        /** Decode a response message, already converted to Parsnip Data,
            into some internal representation.
            @param message The message to be decoded. */
        virtual void extractResponse (const Parsnip::Data &message) const = 0;
    };
    
    /// Pandora request class.  A request whose response is empty/unimportant.
    class Notification : public Request {
    protected:
        Notification (Source *src, const char *ep);
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: Retrieve version number.  We use this as a no-op.
    class RetrieveVersionRequest : public Request {
        mutable int result;

    public:
        RetrieveVersionRequest (Source *src);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const int getResult() const {
            return result;
        }
    };

    /// Pandora message: Retrieve station list
    class RequestStationList : public Request {
        mutable std::unique_ptr<PlaylistList> results;

    public:
        RequestStationList (Source *src);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const PlaylistList &getResponse() const {
            assert (results);
            return *results;
        }
    };

    /// Pandora message: create a station
    class RequestCreateStation : public Request {
        std::string name;
        std::string create_from_id;
        mutable Station *result{nullptr};

    public:
        RequestCreateStation (Source *src, const std::string &name, const std::string &creator_pandora_id);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline Station *getResponse() const {
            assert (result);
            return result;
        }
    };

    /// Pandora message: delete a station
    class RequestRemoveStation : public Request {
        std::string station_id;

    public:
        RequestRemoveStation (Source *src, Station *station);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: transform a shared station into a personal station
    class RequestTransformStation : public Request {
        std::string station_id;
        mutable Station *updated_info{nullptr};

    public:
        RequestTransformStation (Source *src, Station *station);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const Station *getUpdatedStation() const {
            assert (updated_info);
            return updated_info;
        }
    };

    /// Pandora message: rename a station
    class RequestRenameStation : public Request {
        std::string station_id;
        std::string name;

    public:
        RequestRenameStation (Source *src, Station *station, const std::string &new_name);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: Retrieve station seeds
    class RequestStationSeeds : public Request {
    protected:
        std::string station_id;
        mutable std::unique_ptr<ThingieList> seeds;
        MusicThingie *extractAssortedSeed (const Parsnip::Data &seed_info) const;
        RequestStationSeeds (Source *src, const char *endpoint, Station *station);

    public:
        RequestStationSeeds (Source *src, Station *station);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const ThingieList &getSeeds() const {
            assert (seeds);
            return *seeds;
        }
    };

    /// Pandora message: Retrieve station details (seeds and a few other goodies)
    class RequestStationDetails : public RequestStationSeeds {
        mutable int positive_feedback{-1};
        mutable int negative_feedback{-1};

    public:
        RequestStationDetails (Source *src, Station *station);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const bool positiveFeedback() const {
            assert (positive_feedback >= 0);
            return positive_feedback;
        }
        inline const bool negativeFeedback() const {
            assert (negative_feedback >= 0);
            return negative_feedback;
        }
    };

    /// Pandora message: Add or remove station seeds
    class RequestAlterStationSeed : public Request {
        std::string station_id;
        std::string seed_id;

    public:
        RequestAlterStationSeed (Source *src, Station *station, const std::string &id, bool add);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: Retrieve station feedback (ratings)
    class RequestStationFeedback : public Request {
        std::string station_id;
        bool positive;
        mutable std::unique_ptr<SongList> results;

    public:
        RequestStationFeedback (Source *src, Station *station, bool positive);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const SongList &getResponse() const {
            assert (results);
            return *results;
        }
    };

    /// Pandora message: Provide feedback for a song on a station.
    class RequestAddTiredSong : public Request {
        const std::string track_token;

    public:
        RequestAddTiredSong (Source *src, const std::string &token);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: Provide feedback for a song that has played on a station.
    class RequestAddFeedback : public Request {
        const std::string track_token;
        const bool positive;
        mutable std::unique_ptr<std::string> feedback_id;

    public:
        RequestAddFeedback (Source *src, const std::string &token, bool like);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const std::string &getResponse() const {
            assert (feedback_id);
            return *feedback_id;
        }
    };

    /// Pandora message: Remove feedback for a song on a station.
    class RequestDeleteFeedback : public Request {
        const std::string feedback_id;
        const bool positive;

    public:
        RequestDeleteFeedback (Source *src, const std::string &feedback, bool like);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
    };

    /// Pandora message: Retrieve list of music to play from a station.
    class RequestQueueTracks : public Request {
        const std::string station_id;
        const PlayableSong *last_play;
        mutable std::unique_ptr<SongList> results;

    public:
        RequestQueueTracks (Source *src, const Station *station, const PlayableSong *last);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const SongList &getResponse() {
            assert (results);
            return *results;
        }
    };

    /// Pandora message: Retrieve list of music to play from a station.
    class RequestTrackReplay : public Request {
        const PlayableSong *request;
        const PlayableSong *last_played;
        mutable PlayableSong *result;
    public:
        RequestTrackReplay (Source *src, const PlayableSong *song, const PlayableSong *current);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline PlayableSong *getResponse() {
            assert (result);
            return result;
        }
    };

    /// Pandora message: Notify servers of playback start
    class PlaybackStartNotification : public Notification {
        std::string track_token;

    public:
        PlaybackStartNotification (Source *src, const PlayableSong *song);
        virtual Parsnip::Data retrieveRequestMessage() const override;
    };

    /// Pandora message: Notify servers of pausing playback
    class PlaybackPauseNotification : public Notification {
    public:
        PlaybackPauseNotification (Source *src, const PlayableSong * = nullptr);
        virtual Parsnip::Data retrieveRequestMessage() const override;
    };

    /// Pandora message: Notify servers of playback resumption
    class PlaybackResumedNotification : public Notification {
    public:
        PlaybackResumedNotification (Source *src, const PlayableSong * = nullptr);
        virtual Parsnip::Data retrieveRequestMessage() const override;
    };

    // Pandora message: Search request.
    class SearchRequest : public Request {
        std::string query;
        std::string desired_type;
        mutable std::unique_ptr<std::vector<std::string> > results;

    public:
        static const std::string Type_ANY;
        SearchRequest (Source *src, const std::string &q, const std::string &type = Type_ANY);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const std::vector<std::string> &getResponse() {
            assert (results);
            return *results;
        }
    };

    // Pandora message: retrieve details about songs, artists, albums, whatever.
    class RetrieveAnnotations : public Request {
        const std::vector<std::string> retrieval_items;

    protected:
        mutable std::unique_ptr<ThingieList> results;

    public:
        RetrieveAnnotations (Source *src, const std::vector<std::string> &items);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline const ThingieList &getAnnotations() {
            assert (results);
            return *results;
        }
    };

    // Pandora message: retrieve details about a single song, artist, album, whatever.
    class RetrieveSingleAnnotation : public RetrieveAnnotations {
        const std::vector<std::string> retrieval_items;

    public:
        RetrieveSingleAnnotation (Source *src, const std::string &item);
        inline MusicThingie *getAnnotation() {
            assert (results);
            assert (results->size() == 1);
            return (*results).front();
        }
    };


    // Pandora message: retrieve advertisements.
    class RetrieveAdverts : public Request {
        PlayableSong *last_song;
        Station *station;
        mutable std::unique_ptr <SongList> adverts;
    public:
        RetrieveAdverts (Source *src, PlayableSong *last_play, Station *sta);
        virtual Parsnip::Data retrieveRequestMessage() const override;
        virtual void extractResponse (const Parsnip::Data &message) const override;
        inline SongList &getAdverts () {
            assert (adverts);
            return *adverts;
        }
    };
}  // namespace Pandora
