///
/// Library/cache supporting Pandora player.
///	@file		mediaunits/pandora/pandoralibrary.cpp - pianod
///	@author		Perette Barella
///	@date		2020-04-03
///	@copyright	Copyright 2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cassert>

#include "pandoratypes.h"
#include "pandoracomm.h"
#include "pandoramessages.h"
#include "pandora.h"

namespace Pandora {
    Library::Library (Source *src, const ThingiePoolParameters &params) : ThingiePool (params), source (src) {
    }

    /** Convert a Pandora ID to a pianod ID.
        @param pandora_id the ID to convert.
        @return The Pandora ID with the source serial number and MusicThingie type letter prepended.
        @throws invalid_argument if the value is neither a song nor artist ID. */
    static std::string pandora_to_pianod_id (const Source *source, std::string id) {
        assert (id.size() >= 3);
        MusicThingie::Type type_letter;
        std::string type = id.substr (0, 3);
        if (type == "TR:") {
            type_letter = MusicThingie::Type::Song;
        } else if (type == "AR:") {
            type_letter = MusicThingie::Type::Artist;
        } else if (type == "AL:") {
            type_letter = MusicThingie::Type::Album;
        } else if (type == "ST:" || type == "PL:" || type == "SF:") {
            type_letter = MusicThingie::Type::Playlist;
        } else {
            throw std::invalid_argument (type);
        }
        return std::to_string (source->serialNumber()) + char (type_letter) + id;
    }

    /** Unabridge a song or the library.  Given a song:
        - If it's not in the library, add it.
        - If already in the library, unabridge both.
        @return The song, unabridged if possible. */
    Song *Library::unabridge (Song *song) {
        MusicThingie *thing = get (song->songPandoraId());
        if (thing) {
            assert (dynamic_cast<Song *> (thing));
            song->unabridge (static_cast<Song *> (thing));
        } else {
            ThingiePool::add (song);
            source->markDirty();
        }
        return song;
    };

    /** Retrieve a more complete song:
        - If it's not in the library, return what we're given.
        - If in the library, unabridge the library copy, and return that.
        @return The song or a compatible one, unabridged if possible. */
    const Song *Library::unabridged (const Song *song) {
        MusicThingie *thing = get (song->songPandoraId());
        if (thing) {
            assert (dynamic_cast<Song *> (thing));
            Song *other_song = static_cast<Song *> (thing);
            other_song->unabridge (const_cast<Song *> (song), false);
            return other_song;
        }
        return song;
    };

    /** Unabridge music things.  Given a music thingie:
        - If it's not in the library, add it.
        - If it's in the library and not a song, do nothing.
        - If it is in the library, and it's a song, unabridge both.
        @return The music thingie, unabridged if possible. */
    MusicThingie *Library::unabridge (MusicThingie *thing) {
        MusicThingie *other = get (thing->id());
        if (other) {
            Song *song = static_cast<Song *> (thing->asSong());
            if (song) {
                song->unabridge (static_cast<Song *> (other));
            }
        } else {
            ThingiePool::add (thing);
            source->markDirty();
        };
        return thing;
    }

    /// Unabridge a list of songs.
    void Library::unabridge (const SongList &songs) {
        for (auto song : songs) {
            unabridge (song);
        }
        source->markDirty();
    }

    /// Unabridge a list of music things.
    void Library::unabridge (const ThingieList &things) {
        for (auto thing : things) {
            unabridge (thing);
        }
    }

    /** Update a song in the library.
        Unabridge the new item from any info in the library.
        Then add or update the library with the new one. */
    Song *Library::update (Song *song) {
        MusicThingie *other = get (song->id());
        if (other) {
            assert (dynamic_cast<Song *> (other));
            song->unabridge (static_cast<Song *> (other), false);
        }
        ThingiePool::add (song);
        return song;
    }

    void Library::update (const SongList &songs) {
        for (auto song : songs) {
            assert (dynamic_cast<Song *> (song));
            update (static_cast<Song *> (song));
        }
        source->markDirty();
    }

    /** Retrieve details for something.  If it is in the library,
        those are returned; if not, details are retrieved from Pandora.
        @param pandora_id The ID of the thing to retrieve.
        @param The thing requested, or nullptr if it could not be acquired. */
    MusicThingie *Library::fulfill (const std::string &pandora_id) {
        std::string pianod_id = pandora_to_pianod_id (source, pandora_id);
        MusicThingie *got = get (pianod_id);
        if (!got) {
            RetrieveSingleAnnotation annote (source, pandora_id);
            Status status = source->executeRequest (annote);
            if (status != Status::Ok) {
                flog (LOG_WHERE (LOG_ERROR), "Could not retrieve annotation");
                return nullptr;
            }
            got = annote.getAnnotation();
            ThingiePool::add (got);
            source->markDirty();
        }
        return got;
    }

    /** Retrieve details for several things.  Those already in the library
        are returned as-is.  Those missing are retrieved from Pandora.
        @param list A list of IDs of the things to retrieve.
        @param The list of requested things, less any that could not be acquired. */
    ThingieList Library::fulfill (const std::vector<std::string> &list) {
        // Make a list of all the missing thingies
        std::vector<std::string> need_retrieval;
        for (const auto &pandora_id : list) {
            try {
                MusicThingie *thing = get (pandora_to_pianod_id (source, pandora_id));
                if (!thing) {
                    need_retrieval.push_back (pandora_id);
                }
            } catch (std::invalid_argument &ex) {
                flog (LOG_WHERE (LOG_ERROR), "Unknown ID type: ", pandora_id);
            }
        }
        // Perform a single annotation request for all missing items
        if (!need_retrieval.empty()) {
            RetrieveAnnotations retriever (source, need_retrieval);
            Status status = source->executeRequest (retriever);
            if (status == Status::Ok) {
                ThingiePool::add (retriever.getAnnotations());
            } else {
                flog (LOG_WHERE (LOG_ERROR), "Could not retrieve annotations");
            }
        }

        // Make a completed list.  If we're still missing something, ignore it.
        ThingieList things;
        for (const auto &pandora_id : list) {
            MusicThingie *thing = get (pandora_to_pianod_id (source, pandora_id));
            if (thing) {
                things.push_back (thing);
            }
        }

        return things;
    }

    SongList Library::getPlayableSongs (std::function<bool (PlayableSong *)> matcher, const Filter &filter) const {
        SongList results;
        for (auto &item : *this) {
            if (item.second.item->matches (filter)) {
                PlayableSong *song = dynamic_cast <PlayableSong *> (item.second.item.get());
                if (song && matcher (song)) {
                    results.push_back (song);
                }
            }
        }
        return results;
    }
    
    SongList Library::getPlayableSongs (Station *station, const Filter &filter) const {
        std::function <bool (PlayableSong *)> matcher = [station] (PlayableSong *song)->bool {
            return song->playlist() == station;
        };
        return getPlayableSongs (matcher, filter);
    }

    SongList Library::getPlayableSongs (Artist *artist) const {
        std::function <bool (PlayableSong *)> matcher = [artist] (PlayableSong *song)->bool {
            return artist->artistPandoraId() == song->artistPandoraId();
        };
        return getPlayableSongs (matcher, Filter::All);
    }
    SongList Library::getPlayableSongs (Album *album) const {
        std::function <bool (PlayableSong *)> matcher = [album] (PlayableSong *song)->bool {
            if (!album->albumId().empty() && !song->albumId().empty()) {
                return album->albumId() == song->albumId();
            }
            return (album->artistId() == song->artistPandoraId() &&
                    album->albumTitle() == song->albumTitle());
        };
        return getPlayableSongs (matcher, Filter::All);
    }

    Parsnip::Data Library::persist() const {
        Parsnip::Data data{Parsnip::Data::List};

        for (const auto &item : *this) {
            Song *song = static_cast<Song *> (item.second.item.get()->asSong());
            if (song) {
                data.push_back (song->persist());
            }
        }
        return (data);
    }

    void Library::restore (const Parsnip::Data &data) {
        for (const auto &song : data) {
            if (song.contains (Key::TrackToken)) {
                ThingiePool::add (new PlayableSong (source, song, true));
            } else {
                ThingiePool::add (new Song (source, song));
            }
        }
    }

}  // namespace Pandora
