///
/// Pandora communication.
/// @file       mediaunits/pandora/pandoracomm.h - pianod project
/// @author     Perette Barella
/// @date       2020-03-23
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <string>

#include <ctime>

#include <parsnip.h>

#include "logging.h"
#include "httpclient.h"

#include "pandoramessages.h"

namespace Pandora {
    /// Communication completion statuses
    enum class Status {
        Ok,
        CorruptMessage,
        MessageFormatUnknown,
        AllocationError,
        InternalError,
        CommunicationError,
        TooFrequentErrors,
        BadRequest = 400,
        Unauthorized = 401,
        StreamingViolation = 429,
        BadGateway = 502
    };

    const std::string status_strerror (Status status);

    class UserFeatures {
    public:
        int inactivity_timeout{300};
        int daily_skip_limit{60};
        int station_skip_limit{6};
        int station_count{0};
        bool adverts{true};
        bool replays{false};
        bool hifi_audio_encoding{false};
    };

    /** Pandora communication class.  Provided a request, assembles a JSON request
        and issues it to Pandora, then decodes the response.  This class manages
        the CSRF (cross-site request forgery) and authorization tokens, and
        reauthenticates if/when necessary. */
    class Communication {
    private:
        enum State {
            Uninitialized,  /// No CRSF token yet.
            Initialized,    /// Have CRSF token, but not yet authenticated.
            Authenticated   /// Ready to go...
        };

        /// Location of the API.
        static const std::string EndpointUrl;

        std::string username;
        std::string password;
        std::string proxy;
        std::string csrf_token;        ///< Cross-site request forgery token.
        std::string auth_token;        ///< Authorization token, retrieved during login.
        int sequential_failures{0};    ///< Number of failed transactions without successes.
        time_t lockout_until{0};       ///< Time until which communications are blocked/automatically fail.
        time_t session_expiration{0};  ///< Time at which session expires from inactivity.

        UserFeatures features;  ///< User capabilities, retrieved from login response.

        State state{State::Uninitialized};

        HttpClient http_client;

        Status performRequest (const Request &request);
        Status retrieveCSRFtoken();
        Status authenticate();

    public:
        Communication (const std::string &name, const std::string &pass, const std::string &prox = "");
        Status execute (Request &request, bool retry_if_auth_required = true);
        Status sendSimpleNotification (const std::string &url);
        
        Parsnip::Data persist () const;
        void restore (const Parsnip::Data &data);
        
        inline time_t retryTime() const {
            return lockout_until;
        }
        inline const UserFeatures &getFeatures() const {
            return features;
        }
        inline const time_t offlineUntil () const {
            return lockout_until;
        }
        inline const time_t sessionExpires () const {
            return session_expiration;
        }
    };

}  // namespace Pandora
