///
/// Pandora communication library.
/// @file       mediaunits/pandora/pandoratypes.h - pianod project
/// @author     Perette Barella
/// @date       2020-03-23
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <string>

#include <cctype>

#include <curl/curl.h>

#include "logging.h"

#include "pandoramessages.h"
#include "pandoracomm.h"

namespace Pandora {
    /// Pandora REST API location
    const std::string Communication::EndpointUrl = "https://www.pandora.com/api/";

    namespace Key {
        static const char *CSRFToken = "csrfToken";
    }

    /** Return a string corresponding to a Pandora communication status.
        @param status The communication status.
        @return The corresponding string. */
    const std::string status_strerror (Status status) {
        switch (status) {
            case Status::Ok:
                return "Ok";
            case Status::CorruptMessage:
                return "Corrupt message";
            case Status::MessageFormatUnknown:
                return "Unknown message format";
            case Status::AllocationError:
                return "Allocation error";
            case Status::CommunicationError:
                return "Communication error";
            case Status::TooFrequentErrors:
                return "Too many errors (temporary lockout)";
            case Status::BadRequest:
                return "HTTP 400/Bad request";
            case Status::Unauthorized:
                return "HTTP 401/Unauthorized";
            case Status::StreamingViolation:
                return "HTTP 429/Streaming Violation";
            case Status::BadGateway:
                return "HTTP 502/Bad Gateway";
            default:
                return "Error #" + std::to_string (int (status));
        }
    }

    /// Pandora login request
    class AuthorizationRequest : public Request {
        const std::string username;
        const std::string password;
        mutable std::string authorization_token;
        mutable UserFeatures features;

    public:
        AuthorizationRequest (const std::string &user, const std::string &pass)
        : Request (nullptr, "v1/auth/login"), username (user), password (pass){};

        virtual Parsnip::Data retrieveRequestMessage() const override {
            return Parsnip::Data{Parsnip::Data::Dictionary,
                                       "username",
                                       username,
                                       "password",
                                       password,
                                       "keepLoggedIn",
                                       true};
        }
        virtual void extractResponse (const Parsnip::Data &message) const override {
            authorization_token = message["authToken"].asString();

            features.station_count = message["stationCount"].asInteger();

            auto &config = message["config"];
            features.inactivity_timeout = config["inactivityTimeout"].asInteger();
            features.daily_skip_limit = config["dailySkipLimit"].asInteger();
            features.station_skip_limit = config["stationSkipLimit"].asInteger();

            for (const auto &flag : config["flags"]) {
                const std::string &f = flag.asString();
                if (f == "noAds") {
                    features.adverts = false;
                } else if (f == "replaysEnabled") {
                    features.replays = true;
                } else if (f == "highQualityStreamingAvailable") {
                    features.hifi_audio_encoding = true;
                }
            }
        }
        inline const std::string &getResponse() {
            return authorization_token;
        }
        inline const UserFeatures &getFeatures() {
            return features;
        }
    };

    /// Construct a new communicator given the user's name and password, and an optional proxy server.
    Communication::Communication (const std::string &name, const std::string &pass, const std::string &prox)
    : username (name), password (pass), proxy (prox) {
    }

    /// Retrieve the Pandora home page and extract the CSRF token.
    Status Communication::retrieveCSRFtoken() {
        HttpClient::Request request;
        request.type = HttpClient::RequestType::Head;
        request.URL = "https://www.pandora.com";
        request.proxy = proxy;
        const HttpClient::Response response = http_client.performHttpRequest (request);
        if (response.curl_code == CURLE_OK && response.http_status == 200) {
            auto find = response.cookies.find ("csrftoken");
            if (find != response.headers.end()) {
                csrf_token = find->second;
                flog (LOG_WHERE (LOG_PANDORA), "Pandora CSRF token retrieved.");
                return Status::Ok;
            } else {
                flog (LOG_WHERE (LOG_ERROR), "No CSRF token.");
                response.dump();
            }
            return Status::MessageFormatUnknown;
        }
        flog (LOG_WHERE (LOG_ERROR), "HTTP request failed:");
        response.dump();
        return Status::CommunicationError;
    }

    /** Perform an API request.
        @param request The request to perform.
        @return Status::Ok or an error value.
        @throws Exceptions thrown by message decoders. */
    Status Communication::performRequest (const Request &request) {
        try {
            HttpClient::Request req;
            req.type = HttpClient::RequestType::Post;
            req.URL = EndpointUrl + request.endpoint;
            req.proxy = proxy;
            req.headers["X-CsrfToken"] = csrf_token;
            req.headers["Content-Type"] = "application/json;charset=utf-8";
            req.cookies["csrftoken"] = csrf_token;
            if (!auth_token.empty()) {
                req.headers["X-AuthToken"] = auth_token;
            }
            req.debug = request.debug();
            const auto request_message = request.retrieveRequestMessage();
            req.body = request_message.toJson();

            // Set up for some logging controls
            bool detail = (logging_enabled (LOG_PANDORA) && logging_enabled (LOG_PROTOCOL)) || req.debug;
            LOG_TYPE PANDORA_HTTP = LOG_TYPE (detail ? 0 : (LOG_PANDORA | LOG_PROTOCOL));

            flog (LOG_WHERE (PANDORA_HTTP), "Pandora transaction to ", request.endpoint);
            if (detail) {
                request_message.dumpJson ("Request");
            }

            try {
                const HttpClient::Response response = http_client.performHttpRequest (req);
                if (response.http_status < 100 || response.http_status >= 300) {
                    flog (LOG_WHERE (LOG_ERROR), "Failed HTTP request: ", request.endpoint);
                    req.dump();
                    response.dump();
                    return Status (response.http_status);
                }
                Parsnip::Data response_message;
                try {
                    response_message = Parsnip::parse_json (response.body);
                    if (detail) {
                        response_message.dumpJson ("Response");
                    }
                    request.extractResponse (response_message);
                    return Status::Ok;
                } catch (const Parsnip::Exception &err) {
                    flog (LOG_WHERE (LOG_ERROR), "Unexpected HTTP response: ", err.what());
                    if (!detail) {
                        request_message.dumpJson ("Request");
                        response_message.dumpJson ("Response");
                    }
                    return Status::MessageFormatUnknown;
                }
            } catch (const HttpClient::Exception &ex) {
                flog (LOG_WHERE (LOG_ERROR), ex.what());
                req.dump();
                if (!detail) {
                    request_message.dumpJson ("Request");
                }
                return Status::CommunicationError;
            }
        } catch (const std::bad_alloc &err) {
            flog (LOG_WHERE (LOG_ERROR), "Allocation error");
            return Status::AllocationError;
        }
        assert (!"Unreachable");
    }

    /// Authenticate with Pandora.
    Status Communication::authenticate() {
        AuthorizationRequest request (username, password);
        Status status = performRequest (request);
        if (status == Status::Ok) {
            auth_token = request.getResponse();
            features = request.getFeatures();
        }
        return status;
    }

    /** Execute an HTTP request.  Acquire CSRF token and authenticate if necessary.
        - If an error indicates authentication has expired, log in again and retry request.
        - For other errors, back off for a period to prevent overloading Pandora servers.
        @param request The request to perform.
        @internal @param retry_if_auth_required Internal use (for managing recursion).
        @return Status::Ok, or an error. */
    Status Communication::execute (Request &request, bool retry_if_auth_required) {
        time_t now = time (nullptr);
        if (lockout_until && (lockout_until > now)) {
            return Status::TooFrequentErrors;
        }
        if (state == State::Authenticated && now >= session_expiration) {
            state = State::Initialized;
            auth_token.clear();
        }
        Status stat;
        try {
            switch (state) {
                case State::Uninitialized:
                    stat = retrieveCSRFtoken();
                    if (stat != Status::Ok) {
                        flog (LOG_WHERE (LOG_PANDORA), "Pandora CSRF token retrieval failed.");
                        break;
                    }
                    state = State::Initialized;
                    // Fall through
                case State::Initialized:
                    stat = authenticate();
                    if (stat != Status::Ok) {
                        flog (LOG_WHERE (LOG_PANDORA), "Pandora authentication failed.");
                        break;
                    }
                    state = State::Authenticated;
                    // Fall through
                case State::Authenticated:
                    stat = performRequest (request);
                    if (stat == Status::Unauthorized) {
                        state = State::Initialized;
                        auth_token.clear();
                        if (retry_if_auth_required) {
                            return execute (request, false);
                        }
                    }
                    // Fall through
            }
        } catch (const HttpClient::Exception &ex) {
            flog (LOG_ERROR, "HttpClient (", request.endpoint, "): ", ex.what());
            stat = Status::CommunicationError;
        }
        if (stat != Status::Ok) {
            // Gradually back off if we start getting errors.
            if (sequential_failures < 8) {
                sequential_failures++;
            }
            if (sequential_failures > 2) {
                int duration = (1 << (sequential_failures - 1));
                flog (LOG_WHERE (LOG_ERROR),
                      "Multiple successive failures, disabling communication for ",
                      duration,
                      " seconds");
                lockout_until = time (nullptr) + duration;
            }
        } else {
            sequential_failures = 0;
            lockout_until = 0;
            session_expiration = now + features.inactivity_timeout;
        }
        return stat;
    }

    /** Send a simple notification by hitting a URL. */
    Status Communication::sendSimpleNotification (const std::string &url) {
        try {
            HttpClient::Request req;
            req.type = HttpClient::RequestType::Get;
            req.URL = url;
            req.proxy = proxy;
            // If the site isn't Pandora, don't give them cookies or tokens.
            int site_end = 0;
            for (int slash_count = 3; slash_count > 0 && site_end < url.size(); site_end++) {
                if (url [site_end] == '/') {
                    slash_count --;
                }
            }
            if (site_end > 12) {
                std::string site = url.substr (site_end - 12, 12);
                for (auto &ch : site) {
                    ch = tolower (ch);
                }
                if (site == "pandora.com/") {
                    req.cookies["csrftoken"] = csrf_token;
                    if (!auth_token.empty()) {
                        req.headers["X-AuthToken"] = auth_token;
                    }
                }
            }
            
            // This method is used for tracking notifications.  Many aren't
            // even pandora.com.  So screw auth and CSRF tokens, there's enough
            // tracking junk in their URLs.

            // Set up for some logging controls
            LOG_TYPE PANDORA_HTTP = LOG_TYPE (LOG_PANDORA | LOG_PROTOCOL);
            flog (LOG_WHERE (PANDORA_HTTP), "Pandora notification to ", url);

            try {
                const HttpClient::Response response = http_client.performHttpRequest (req);
                if (response.http_status < 100 || response.http_status >= 300) {
                    flog (LOG_WHERE (LOG_ERROR), "Failed HTTP notification: ", url);
                    req.dump();
                    return Status (response.http_status);
                }
                return Status::Ok;
            } catch (const HttpClient::Exception &ex) {
                flog (LOG_WHERE (LOG_ERROR), ex.what());
                req.dump();
                return Status::CommunicationError;
            }
        } catch (const std::bad_alloc &err) {
            flog (LOG_WHERE (LOG_ERROR), "Allocation error");
            return Status::AllocationError;
        }
        assert (!"Unreachable");
    }

    /** Persist communication settings.
        @return A dictionary with any settings the communicator wants saved. */
    Parsnip::Data Communication::persist() const {
        return Parsnip::Data{Parsnip::Data::Dictionary, Key::CSRFToken, csrf_token};
    }

    /** Restore communication settings.
        @param data A dictionary of previous settings from which to restore. */
    void Communication::restore (const Parsnip::Data &data) {
        csrf_token = data [Key::CSRFToken].asString();
        if (!csrf_token.empty()) {
            state = State::Initialized;
        }
    }

}  // namespace Pandora
