///
/// Class for source parameters and their parsing.
///	@file		mediaparameters.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-01-28
///	@copyright	Copyright (c) 2015-2020 Devious Fish. All rights reserved.
///

#include <config.h>
#include <football.h>

#include "fundamentals.h"
#include "datastore.h"
#include "user.h"
#include "users.h"
#include "mediaparameters.h"
#include "datastore.h"

namespace Media {
    using namespace std;

    typedef enum source_parameter_id_t {
        REMEMBER = 10000,
        RESTORE,
        WAIT,
        OWNER,
        NAME,
        ACCESS,
        PRIVATE, // Deprecated, to be removed October 2016.  Use ACCESS instead.
        SHARED, // Deprecated
        PUBLISHED, // Deprecated
        PUBLIC, // Deprecated
        SONGPROXY
    } SourceParameterId;

    LookupTable<Media::ProxyMode> ProxyWords {
        { "None",      ProxyMode::None },
        { "Donor",     ProxyMode::Donor },
        { "Recipient", ProxyMode::Recipient }
    };

    LookupTable<Media::PersistenceMode> PersistenceWords {
        { "Remember",   PersistenceMode::Remember },
        { "Restore",    PersistenceMode::Restore }
    };

    /** Parse definitions for the parameter parser */
    FB_PARSE_DEFINITION standard_options [] = {
        { REMEMBER, "remember ..." },
        { RESTORE, "restore ..." },
        { WAIT, "wait ..." },
        { OWNER, "owned by {user} ..." },
        { NAME, "name {name} ..." },
        { ACCESS, "access <mode:disowned|private|shared|published|public> ..." },
        { PRIVATE, "private ..." }, // Deprecated: use ACCESS
        { SHARED, "shared ..." }, // Deprecated: use ACCESS
        { PUBLISHED, "published ..." }, // Deprecated: use ACCESS
        { PUBLIC, "public ..." }, // Deprecated: use ACCESS
        { SONGPROXY, "song proxy <proxy:none|recipient|donor> ..." },
        { 0, NULL }
    };

    int interpretCommonSourceParameters (Football::GenericOptionParser &parser,
                                         int cmd, SourceParameters &result) {
        SourceParameterId spid  = (SourceParameterId) cmd;
        switch (spid) {
            case REMEMBER:
            case RESTORE:
                if (result.userSpecified || result.permissions == Ownership::Type::DISOWNED)
                    return E_CONFLICT;
                result.persistence = (spid == REMEMBER ? PersistenceMode::Remember :
                                      PersistenceMode::Restore);
                return FB_PARSE_SUCCESS;
            case WAIT:
                result.waitForReady = true;
                return FB_PARSE_SUCCESS;
            case OWNER:
                if (result.persistence != PersistenceMode::Temporary ||
                    result.permissions == Ownership::Type::DISOWNED)
                    return E_CONFLICT;
                result.owner = user_manager->get (parser.argv ("user"));
                result.userSpecified = true;
                return result.owner ? FB_PARSE_SUCCESS : (int) E_NOTFOUND;
            case ACCESS:
                result.permissions = OWNERSHIP [parser.argv ("mode")];
                if (result.permissions == Ownership::Type::DISOWNED) {
                    if (result.userSpecified || result.persistence != PersistenceMode::Temporary)
                        return E_CONFLICT;
                    result.owner = nullptr;
                }
                return FB_PARSE_SUCCESS;
            case PRIVATE:
            case SHARED:
            case PUBLISHED:
            case PUBLIC:
                result.permissions = OWNERSHIP [parser.argv (0)];
                return FB_PARSE_SUCCESS;
            case NAME:
            {
                result.name = parser.argv ("name");
                for (auto ch : result.name) {
                    if (!isalnum (ch)) return (int) E_INVALID;
                }
                return FB_PARSE_SUCCESS;
            }
            case SONGPROXY:
                result.song_proxy = ProxyWords [parser.argv ("proxy")];
                return FB_PARSE_SUCCESS;
        }
        return FB_PARSE_FAILURE;
    }


    bool SourceParameters::persist (UserData::StringDictionary &dest) const {
        assert (persistence != PersistenceMode::Temporary &&
                persistence != PersistenceMode::Loaded);
        dest.add ("name", name);
        dest.add ("ownership", OWNERSHIP [permissions]);
        dest.add ("song_proxy", ProxyWords [song_proxy]);
        dest.add ("persistence", PersistenceWords [persistence]);
        return !name.empty();
    }
    SourceParameters::SourceParameters (Ownership::Type perm, User *user) {
        permissions = perm;
        owner = user;
        if (user) {
            name = user->username();
        }
    }

    SourceParameters::SourceParameters (Ownership::Type perm,
                                        const std::string &sourcename) {
        assert (!sourcename.empty());
        permissions = perm;
        name = sourcename;
    };

    SourceParameters::SourceParameters (const UserData::StringDictionary &src) {
        name = src.get ("name", "");
        if (name.empty())
            throw (invalid_argument ("unnamed source"));
        if (!OWNERSHIP.tryGetValue (src.get ("ownership", "private"), &permissions))
            throw (invalid_argument ("ownership field invalid"));
        if (!ProxyWords.tryGetValue (src.get ("song_proxy", "none"), &song_proxy))
            throw (invalid_argument ("song_proxy field invalid"));
        if (persistence == PersistenceMode::Temporary)
            persistence = PersistenceMode::Loaded;
    }
    
    SourceParameters::~SourceParameters () {
    }
}


