///
/// Manages the available media source instances.
/// Methods and actions outside the usual source actions.
///	@file		mediamanager.cpp - pianod
///	@author		Perette Barella
///	@date		2014-11-28
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <string>
#include <iterator>
#include <algorithm>

#include "musictypes.h"
#include "encapmusic.h"
#include "mediaunit.h"
#include "mediaparameters.h"
#include "mediamanager.h"

#include "callback.h"
#include "callbackimpl.h"
#include "fundamentals.h"
#include "utility.h"
#include "response.h"


// Force template instantiation.
template class CallbackManager<Media::Manager, Media::Manager::Callbacks>;

namespace Media {
    using namespace std;

    Manager::Manager (void) : Source::Source (new SourceParameters (Ownership::Type::PUBLISHED, "Pianod")) {
        MusicAutoReleasePool pool;
        add (this);
        state = State::READY;

        // Create a reusable mix playlist
        // If this fails, let bad_alloc throw because we need the manager's mix playlist.
        mix_playlist = new MetaPlaylist (this, PianodPlaylist::MIX);
        mix_playlist->retain();

        // Create a reusable everything playlist
        everything_playlist = new (nothrow) MetaPlaylist (this, PianodPlaylist::EVERYTHING);
        if (everything_playlist) {
            everything_playlist->retain();
        }

        // Create a reusable transient playlist
        transient_playlist = new (nothrow) MetaPlaylist (this, PianodPlaylist::TRANSIENT);
        if (transient_playlist) {
            transient_playlist->retain();
        }
    }

    Manager::~Manager () {
        mix_playlist->release();
        if (everything_playlist) everything_playlist->release();
        if (transient_playlist) transient_playlist->release();

        iterator it = begin();
        while (it != end()) {
            iterator remove = it++;
            if (remove->second != this) {
                kept_assert (erase (remove->second));
            }
        }
    }


    /** Retrieve a source by serial number.
        @param serial The source serial number.
        @return The source, or nullptr if not found. */
    Source *Manager::get (const SerialNumber serial) const {
        const_iterator item = find (serial);
        if (item == end()) return nullptr;
        return item->second;
    }

    /** Retrieve a source by type and ID.  ID meaning varies by source.
        @param type The source type name.
        @param ident The source ID.
        @return The source, or nullptr if not found. */
    Source *Manager::get (const string &type, const string &ident) const {
        for (auto const &item : *this) {
            if (strcasecmp (item.second->kind(), type) == 0 &&
                strcasecmp (item.second->name(), ident) == 0) {
                return item.second;
            }
        }
        return nullptr;
    }

    /** Add a source.  No validation or messaging is done.
        @param source The new source.
        Caller is responsible for destruction on failure.
        @return True on success, false on error (duplicate). */
    bool Manager::add (Source *source) {
        assert (source);
        if (!get (source->kind(), source->name())) {
            static SerialNumber unitNumber = 0;
            source->serialNum = ++unitNumber;
            value_type item (source->serialNum, source);
            pair<iterator, bool> result = insert (item);
            if (result.second) {
                source->statusHandler = std::bind (&Manager::redirectingStatusHandler, this,
                                                  std::placeholders::_1, std::placeholders::_2);
                return true;
            }
            flog (LOG_WHERE (LOG_ERROR), "Failed to insert new source");
        };
        return false;
    };

    /** Interactively add a source.  Checks usability, handles events,
        announces change.
        @param source The source to add.  If there is a problem, this is freed.
        @param conn The connection where status messages will be sent.
        @param borrowed True to send borrowing rather than adding messages.
        @return false on failure, true on success. */

    bool Manager::add (Source *source, PianodConnection &conn, bool borrowed) {
        // Disowned sources have no owner, therefore fail the usual usable test.
        if (source->isOwned() && !source->isUsableBy (conn.user)) {
            conn << E_UNAUTHORIZED;
        } else if (add (source)) {
            assert (source->parameters);
            if (source->parameters->waitForReady) {
                conn.waitForEvent (WaitEvent::Type::SourceReady, source);
            } else {
                conn << S_OK;
            }
            conn.announce (borrowed ? A_SOURCE_BORROW : A_SOURCE_ADD);
            if (source->parameters->persistence != PersistenceMode::Temporary &&
                source->parameters->persistence != PersistenceMode::Loaded) {
                source->persist();
            }
            return true;
        } else {
            conn << E_DUPLICATE;
        }
        delete source;
        return false;
    }


    /** Remove a source.  If a source is busy, its is moved out of "ready" state
        and removal deferred.  The periodic() method subsequently checks if it
        is ready to be removed, and does so when possible.
        @param source The source to remove.
        @return true if successful, false if source is busy (removal is deferred). */
    bool Manager::erase (Source *source) {
        source->state = State::DEAD;
        if (!callback.queryUnanimousApproval (true, &Callbacks::canRemoveSource, source)) {
            return false;
        }
        callback.notify (&Callbacks::sourceOffline, source);
        callback.notify (&Callbacks::sourceRemoved, source);
        unordered_map::erase (source->serialNum);
        source->alert (V_SOURCES_CHANGED, "removal complete");
        delete source;
        return true;
    }


    /** Check if there are any sources in a given state.
        @return state The state to search for.
        @return true if any sources are in the state, false otherwise. */
    bool Manager::areSourcesInState (Source::State state) const {
        for (auto const &src : *this) {
            if (src.second->state == state)
                return true;
        }
        return false;
    }

    /** Get all sources (excluding the manager itself). */
    Manager::SourceList Manager::getRealSources () const {
        SourceList sources;
        sources.reserve (size());
        for (const auto &src : *this) {
            if (src.second != this) {
                sources.push_back (src.second);
            }
        }
        return sources;
    };

    /** Get sources (excluding the manager itself) in a given state. */
    Manager::SourceList Manager::getRealSources (Source::State state) const {
        SourceList sources;
        sources.reserve (size());
        for (const auto &src : *this) {
            if (src.second != this && state == src.second->state) {
                sources.push_back (src.second);
            }
        }
        return sources;
    };

    /** Ask all the sources to persist any data.
        @return True if all succeeded, false otherwise. */
    bool Manager::flush (void) {
        bool status = true;
        for (auto src : getRealSources()) {
            status = src->flush() && status;
        }
        return status;
    }


    /** Respond to a source's state change.
        - Send any appropriate messages.
        - Invoke appropriate callbacks.
        @param source The source whose state has changed. */
    void Manager::handleSourceStateChange (Source *source) {
        if (source->state == State::READY) {
            source->alert (V_SOURCES_CHANGED, "ready");
            callback.notify (&Callbacks::sourceReady, source);
        } else if (source->state == State::VALID &&
                   source->announced_state == State::READY) {
            source->alert (V_SOURCES_CHANGED, "offline");
            callback.notify (&Callbacks::sourceOffline, source);
        } else if (source->state == State::DEAD) {
            source->status ("removal initiated");
        }
    }


    /** Perform periodic activities and remove sources that deferred removal.
        This method dispatches to each registered source.
        Each source indicates when they want their next time slice.
        @return Interval until next invokation (shortest interval requested
        by any children sources). */
    float Manager::periodic (void) {
        float nextRequest = A_LONG_TIME;

        // Purge dead sources.  Range-for can't be used because of erasures.
        iterator iter = begin();
        while (iter != end()) {
            iterator remove = iter++;
            if (remove->second->state == State::DEAD) {
                assert (remove->second != this);
                remove->second->flush();
                erase (remove->second);
            }
        }

        for (auto &it : *this) {
            if (it.second != this) {
                float nextReq = it.second->periodic();
                if (nextReq < nextRequest) nextRequest = nextReq;

                // If the state has changed, dispatch the state change callbacks.
                if (it.second->announced_state != it.second->state) {
                    handleSourceStateChange(it.second);
                    it.second->announced_state = it.second->state;
                }
            }
        }
        return nextRequest;
    }

    /** Handle an alert by passing it to callbacks.
        @param status The numeric status or alert.
        @param detail Description of alert details. */
    void Manager::redirectingStatusHandler (RESPONSE_CODE status, const char *detail) {
        callback.notify (&Callbacks::statusNotification, status, detail);
    }


    /** Retrieve a playlist, song, album or artist by ID.
        @param id The item's ID.
        @return The item, or nullptr if not found or source not ready.
        @throw CommandError if the ID is invalid. */
    MusicThingie *Manager::getAnythingById (const string &id) {
        SplitId origin (id);
        return origin.source->getAnythingById (origin);
    }

    /** Retrieve a playlist, song, album or artist by ID.
        @param id The item's ID.
        @return The item, or nullptr if not found or source not ready. */
    MusicThingie *Manager::getAnythingById (const SplitId &id) {
        assert (0);
        flog (LOG_FUNCTION (LOG_WARNING), " called with split ID");
        return id.source->getAnythingById (id);
    }
}


/* Global */ Media::Manager *media_manager {nullptr};

