///
/// Command handlers for media manager and source-related actions.
///	@file		managercommand.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-01-29
///	@copyright	Copyright (c) 2015-2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cassert>

#include <football.h>

#include "fundamentals.h"
#include "sources.h"
#include "connection.h"
#include "response.h"
#include "mediaunit.h"
#include "mediamanager.h"
#include "users.h"

using namespace std;



typedef enum manager_commands_t {
    SOURCESELECT = CMD_RANGE_MEDIA_MANAGER,
    SOURCEEXEC,
    SOURCETYPELIST,
    SOURCELIST,
    SOURCEAVAILABLE,
    SOURCEMINE,
    SOURCEREMOVE,
    SOURCEFORGET,
    SOURCERENAME,
    WAITFORONEREADY,
    WAITFORMANYREADY,
    SOURCESTATISTICS,
    SOURCESTATISTICSALL
} COMMAND;

static const FB_PARSE_DEFINITION statementList[] = {
    { SOURCETYPELIST,   "source list type" },
    { SOURCELIST,       "source list [enabled]" },
    { SOURCEAVAILABLE,  "source list available" },
    { SOURCEMINE,       "source list mine" },
    { SOURCEEXEC,       "with source id {#id} [{command}] ..." },
    { SOURCEEXEC,       "with source type {type} name {name} [{command}] ..." },
    { SOURCESELECT,     "source select id {#id}" },
    { SOURCESELECT,     "source select type {type} name {name}" },
    /** Source delete changed to source disconnect 5 January 2017.  Delete to be removed 2018. */
    { SOURCEREMOVE,     "source [disconnect|delete] id {#id}" },
    { SOURCEREMOVE,     "source [disconnect|delete] type {type} name {name}" },
    { SOURCEFORGET,     "source forget type {type} name {name}" },
    { SOURCERENAME,     "source rename type {type} name {name} to {newname}" },
    { WAITFORONEREADY,  "wait for source id {#id} ready [{options}] ..." },
    { WAITFORONEREADY,  "wait for source type {type} name {name} ready [{options}] ..." },
    { WAITFORMANYREADY, "wait for source <which:any|all> [pending] ready [{options}] ..." },
    { SOURCESTATISTICS, "source statistics" },
    { SOURCESTATISTICS, "source statistics id {#id}" },
    { SOURCESTATISTICS, "source statistics type {type} name {name}" },
    { SOURCESTATISTICSALL, "source statistics all" },
    { CMD_INVALID, NULL }
};


/** Interpreter for commands that select or manage active sources.
    (Commands for enabling sources are handled by a source's interpreter.) */
class SourceCommands : public Football::Interpreter<PianodConnection, COMMAND> {
private:
    Media::Source *getSource (PianodConnection &conn);
    void dumpStatistics (PianodConnection &conn, Media::Source *source);
    virtual bool hasPermission (PianodConnection &conn, COMMAND command);
    virtual void handleCommand (PianodConnection &conn, COMMAND command);
    virtual const FB_PARSE_DEFINITION *statements (void) {
        return statementList;
    };
};

/** Get a source commands predicate (either by ID or type and name).
    @param conn The connection issuing the command.
    @return The source specified.
    @throw CommandError if the specified source does not exist. */

Media::Source *SourceCommands::getSource (PianodConnection &conn) {
    auto src = (conn ["id"] ? media_manager->get (atoi (conn ["id"])) :
                conn ["type"] ? media_manager->get (conn.argv ("type"), conn.argv ("name"))
                : conn.source());
    if (!src) {
        throw CommandError (E_NOTFOUND);
    }
    return src;
};

/// Send a statistic.
static Response statsResp (const char *name, int value) {
    return Response (I_INFO, string {name} + ": " + to_string (value));
}

/// Send statistics data for a source.
void SourceCommands::dumpStatistics (PianodConnection &conn, Media::Source *source) {
    auto stats = source->getStatistics();
    conn << Response (I_INFO, (*source)())
    << statsResp ("play attempts", stats.songs_attempted)
    << statsResp ("tracks played", stats.tracks_played)
    << statsResp ("playback failures", stats.playback_failures)
    << statsResp ("successive failures", stats.successive_failures)
    << statsResp ("songs replaced", stats.songs_replaced)
    << statsResp ("songs donated", stats.songs_donated);
}


bool SourceCommands::hasPermission (PianodConnection &conn, COMMAND command) {
    if (conn.havePrivilege (Privilege::Service))
        return true;
    switch (command) {
        case SOURCESELECT:
        case SOURCEEXEC:
        case SOURCETYPELIST:
        case SOURCEFORGET:
        case SOURCERENAME:
        case SOURCELIST:
            return conn.haveRank (Rank::Listener);
        case SOURCEAVAILABLE:
            return conn.haveRank (Rank::Standard);
        case SOURCESTATISTICS:
            return conn.source()->isReadableBy (conn.user);
        default:
            return conn.havePrivilege (Privilege::Service);
    }
};

void SourceCommands::handleCommand (PianodConnection &conn, COMMAND command) {
    switch (command) {
        case SOURCESELECT:
        {
            auto src = getSource (conn);
            if (!src->isReady()) {
                conn << Response (E_WRONG_STATE, "Source not ready or shutting down");
            } else {
                conn << S_OK;
                conn.source (src);
            }
            return;
        }
        case SOURCEEXEC:
        {
            auto src = getSource (conn);
            Media::Source::SerialNumber original = conn.source()->serialNumber();
            conn.source (src, false);
            conn.reinterpret("command");
            bool changed = (conn.source() != src);
            src = media_manager->get (original);
            if (changed) {
                // Stupid users doing unexpected insane crap...
                conn.source (src ? src : conn.source());
            } else if (src) {
                conn.source (src, false);
            } else {
                conn.announce (A_IMBECILE);
            }
            return;
        }
        case SOURCETYPELIST:
        {
            conn << S_DATA << Response (I_SOURCE, SOURCE_NAME_MANAGER);
            for (const auto name : Sources::sourceNames()) {
                conn << Response (I_SOURCE, name);
            }
            conn << S_DATA_END;
            return;
        }
        case SOURCELIST:
            for (auto const &src : *media_manager) {
                conn << S_DATA;
                conn << Response (I_ID, src.second->serialNumber());
                conn << Response (I_SOURCE, src.second->kind());
                conn << Response (I_NAME, src.second->name());
                if (src.second->isOwned()) {
                    conn << Response (I_OWNER, src.second->ownerName());
                }
            }
            conn << S_DATA_END;
            return;
        case SOURCEAVAILABLE:
        case SOURCEMINE:
        {
            if (!conn.user) {
                conn << E_LOGINREQUIRED;
                return;
            }
            auto available = user_manager->getStoredSources (command == SOURCEAVAILABLE ? UserManager::WhichSources::Listed : UserManager::WhichSources::User,
                                                     conn.user);
            for (const auto &source : available) {
                conn << S_DATA
                << Response (I_SOURCE, source.second->origin())
                << Response (I_NAME, source.second->identity());
            }
            conn << S_DATA_END;
            return;
        }
        case SOURCEFORGET:
        case SOURCERENAME:
        {
            if (!conn.user) {
                conn << E_LOGINREQUIRED;
                return;
            }
            const char *type = conn ["type"];
            const char *name = conn ["name"];
            assert (type);
            assert (name);
            UserData::DataStore *data;
            if ((data = conn.user->getData (type, name))) {
                if (command == SOURCERENAME) {
                    const char *new_name = conn ["newname"];
                    assert (new_name);
                    if (conn.user->getData (type, new_name)) {
                        throw CommandError (E_DUPLICATE);
                    }
                    UserData::StringDictionary *new_data = new UserData::StringDictionary (*static_cast <UserData::StringDictionary *> (data));
                    new_data->rename (new_name);
                    (*new_data) ["name"] = new_name;
                    if (!conn.user->attachData (new_data)) {
                        delete new_data;
                        throw CommandError (E_NAK);
                    }
                }
                conn.user->removeData (type, name);
                conn << S_OK;
            } else {
                conn << E_NOTFOUND;
            }
            return;
        }
        case SOURCEREMOVE:
        {
            auto src = getSource (conn);
            if (src->serialNumber() == 1) {
                conn << Response (E_UNAUTHORIZED, "Media Manager cannot be removed");
            } else {
                conn << (media_manager->erase (src) ? S_OK : S_PENDING);
                conn.announce (A_SOURCE_REMOVE);
            }
            return;
        }
        case WAITFORMANYREADY:
            if (conn ["pending"] && !media_manager->areSourcesPending())
                conn << E_WRONG_STATE;
            else
                conn.waitForEventWithOptions (WaitEvent::Type::SourceReady,
                                              conn.argvEquals ("which", "all") ? nullptr : media_manager);
            return;
        case WAITFORONEREADY:
        {
            auto src = getSource (conn);
            if (src->isReady())
                conn << E_WRONG_STATE;
            else
                conn.waitForEventWithOptions (WaitEvent::Type::SourceReady, src);
            return;
        }
        case SOURCESTATISTICS:
            conn << S_DATA;
            dumpStatistics (conn, getSource (conn));
            conn << S_DATA_END;
            return;
        case SOURCESTATISTICSALL:
            conn << S_DATA;
            for (auto &src : *media_manager) {
                dumpStatistics (conn, src.second);
            }
            conn << S_DATA_END;
            return;
        default:
            conn << E_NOT_IMPLEMENTED;
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}

static SourceCommands interpreter;

void register_media_manager_commands (PianodService *service) {
    service->addCommands(&interpreter);
}

