///
/// Media source for filesystem/local audio files.
///	@file		filesystemsource.cpp - pianod
///	@author		Perette Barella
///	@date		2015-02-25
///	@copyright	Copyright 2015-2017 Devious Fish.  All rights reserved.
///

#include <config.h>

#include "fundamentals.h"
#include "sources.h"
#include "filesystem.h"
#include "encapmusic.h"
#include "musiclibrary.h"
#include "musiclibraryparameters.h"
#include "mediaunit.h"
#include "mediaparameters.h"

namespace Filesystem {
    /*
     *                  Identity
     */
    using namespace Media;
    using namespace std;

    Source::Source (const Parameters &params)
    : library (this, params.path, params.scan_frequency),
    Media::Source (new Parameters (params)) {
        MusicAutoReleasePool pool;

        // Create a reusable mix playlist
        mix_playlist = new (nothrow) MetaPlaylist (this,
                                                  PianodPlaylist::PlaylistType::MIX);
        if (mix_playlist)
            mix_playlist->retain();

        // Create a reusable everything playlist
        everything_playlist = new (nothrow) MetaPlaylist (this,
                                                         PianodPlaylist::PlaylistType::EVERYTHING);
        if (everything_playlist)
            everything_playlist->retain();

        if ((params.scan_frequency == MusicLibrary::SCAN_ON_LOAD ||
             params.scan_frequency == MusicLibrary::SCAN_EVERY_LOAD) &&
            library.searches.empty()) {
            library.startScan();
        }
    }

    Source::~Source () {
        if (mix_playlist) mix_playlist->release();
        if (everything_playlist) everything_playlist->release();
    }

    const char *Source::kind (void) const {
        return SOURCE_NAME_FILESYSTEM;
    };

    
    /*
     *                      Capabilities
     */

    bool Source::canExpandToAllSongs (void) const {
        return true;
    };

    bool Source::flush (void) {
        return (parameters->persistence == PersistenceMode::Temporary ? true :
                library.flush());
    };

    float Source::periodic (void) {
        float next_request = library.periodic();
        if (library.searches.empty()) {
            if (library.valid && state < State::READY)
                state = State::READY;
            else if (!library.valid && state == State::READY)
                state = State::VALID;
            auto params = static_cast <const Parameters *> (parameters);
            if (params->scan_frequency == MusicLibrary::SCAN_DAILY &&
                library.last_scan + 86400 /* One day */ < time (nullptr)) {
                try {
                    library.startScan();
                } catch (...) {
                    flog (LOG_WHERE (LOG_WARNING), params->path,
                          "Cannot start periodic filesystem scan");
                }
            }
        }
        return next_request;
    }

    /*
     *                      Playlist Methods
     */

    PlaylistList Source::getPlaylists (const Filter &filter) {
        PlaylistList list;
        for (auto &item : library.playlists) {
            if (filter.matches (item.second)) {
                list.push_back (item.second);
            }
        }
        return list;
    };

    MusicThingie *Source::getAnythingById (const Media::SplitId &id) {
        return library.getById (id.type, id.innerId);
    };
    
    PianodPlaylist *Source::getMixPlaylist (void) {
        return mix_playlist;
    }

    PianodPlaylist *Source::getEverythingPlaylist (void) {
        return everything_playlist;
    }

    PianodPlaylist *Source::createPlaylist (const char *name,
                                          MusicThingie::Type type,
                                          MusicThingie *from) {
        assert (name);
        return library.createPlaylist (name, type, from);
    };

    PianodPlaylist *Source::createPlaylist (const char *name, const Filter &filter) {
        assert (name);
        return library.createPlaylist(name, filter);
    }

    PianodPlaylist *Source::getTransientPlaylist (const Filter &criteria) {
        return library.formTransientPlaylist (criteria);
    }

    // Miscellaneous

    SongList Source::getRandomSongs (PianodPlaylist *playlist, const UserList &users,
                                     SelectionMethod selectionMethod) {
        auto params = static_cast <const Parameters *> (parameters);
        return library.getRandomSongs(playlist, users, selectionMethod, *params);
    }

    ThingieList Source::getSuggestions (const Filter &filter, SearchRange where) {
        return ThingieList (library.getSuggestions (filter, where));
    }

    // Typecast thing to an equivalent Filesystem suggestion
    MusicThingie *Source::getSuggestion (MusicThingie *thing,
                                         MusicThingie::Type type,
                                         SearchRange where) {
        assert (thing->source() != this);
        return Media::Source::getSuggestion (thing, type, where, true);
    }

    void Source::playbackProblem (void) {
        library.checkValidity ();
    };

    void Source::rescan (bool reset) {
        if (reset)
            library.clear ();
        library.startScan ();
    }

} // </namespace>
