///
/// Command handlers for filesystem source.
///	@file		filesystemcommand.cpp - pianod
///	@author		Perette Barella
///	@date		2015-02-25
///	@copyright	Copyright 2015-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cassert>

#include <football.h>

#include "fundamentals.h"
#include "sources.h"
#include "connection.h"
#include "user.h"
#include "users.h"
#include "response.h"
#include "mediamanager.h"
#include "mediaparameters.h"
#include "musiclibraryparameters.h"

#include "filesystem.h"

using namespace std;

typedef enum filesystem_command_t {
    FSADDSOURCE = CMD_RANGE_FILESYSTEM,
    FSEXISTING,
    FSRESCAN
} FSCOMMAND;

static const FB_PARSE_DEFINITION statementList[] = {
    { FSADDSOURCE,      "filesystem add {path} [{parameters}] ..." },
    { FSEXISTING,       "filesystem use {name} [{parameters}] ..." },
    { FSRESCAN,         "filesystem rescan [fresh]" },
    { CMD_INVALID,      NULL }
};

/** Parser for filesystem media source commands. */
class FilesystemCommands : public Football::Interpreter<PianodConnection, FSCOMMAND> {
private:
    virtual bool hasPermission (PianodConnection &conn, FSCOMMAND command);
    virtual void handleCommand (PianodConnection &conn, FSCOMMAND command);
    virtual const FB_PARSE_DEFINITION *statements (void) {
        return statementList;
    };
};

bool FilesystemCommands::hasPermission (PianodConnection &conn, FSCOMMAND) {
    return conn.havePrivilege (Privilege::Service);
};

void FilesystemCommands::handleCommand (PianodConnection &conn, FSCOMMAND command) {
    switch (command) {
        case FSADDSOURCE:
        {
            if (!conn.user) {
                conn << E_LOGINREQUIRED;
                return;
            }
            Filesystem::Parameters p (Ownership::Type::PUBLISHED, conn.user);
            p.path = conn.argv ("path");
            if (MusicLibrary::parameterParser.interpret(conn.argvFrom ("parameters"),
                                          p, &conn) == FB_PARSE_SUCCESS) {
                p.sanitize();
                Media::Source *m = new Filesystem::Source (p);
                media_manager->add (m, conn);
            }
            return;
        }
        case FSEXISTING:
        {
            UserData::StringDictionary *persisted;
            User *owner;
            RESPONSE_CODE status = user_manager->findStoredSource (SOURCE_NAME_FILESYSTEM,
                                                                   conn.argv ("name"),
                                                                   conn.user,
                                                                   &persisted,
                                                                   &owner);
            if (status != S_OK) {
                conn << status;
                return;
            }
            try {
                Filesystem::Parameters params (*persisted);
                params.owner = owner;
                if (MusicLibrary::parameterParser.interpret (conn.argvFrom ("parameters"), params, &conn) == FB_PARSE_SUCCESS) {
                    params.sanitize();
                    Media::Source *m = new Filesystem::Source (params);
                    media_manager->add (m, conn, owner != conn.user);
                }
            } catch (const invalid_argument &e) {
                conn << Response (E_CREDENTIALS, e.what());
            }
            return;
        }
        case FSRESCAN:
        {
            auto fs = dynamic_cast<Filesystem::Source *> (conn.source());
            if (!fs) {
                conn << E_WRONGTYPE;
            } else if (!fs->isEditableBy(conn.user)) {
                conn << E_UNAUTHORIZED;
            } else {
                fs->rescan (conn ["fresh"]);
                conn << S_OK;
            }
            return;
        }
        default:
            conn << E_NOT_IMPLEMENTED;
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}

static FilesystemCommands interpreter;

void register_filesystem_commands (PianodService *service) {
    service->addCommands(&interpreter);
}

void restore_filesystem_source (UserData::StringDictionary *persisted, User *user) {
    Media::Source *m = nullptr;
    try {
        Filesystem::Parameters params (*persisted);
        params.owner = user;
        m = new Filesystem::Source (params);
        if (!media_manager->add (m))
            delete m;
    } catch (...) {
        delete m;
        throw;
    }
    return;
}
