///
/// Football parser unit test.
/// Validates parser operation by:
/// - Creating valid parser and making sure the results they return are correct.
/// - Creating invalid parsers and making sure the parser generator balks.
/// @file       fb_parsertest.c - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2014-11-24
/// @copyright  Copyright 2014-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <stdio.h>
#include <stdbool.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>
#include "fb_public.h"
#include "fb_service.h"


#define PARSER_CREATE_FAIL (-200)
#define PARSER_CREATE_SUCCESS (-201)

#define ARGV_CREATE_FAILURE (-400)


/* Check values >= 0 mean check argnames */
/* These two currently share the same parameter, so duplicate ID is correct. */
#define CHECK_ERROR_FIELD (-1)
#define CHECK_NEXT_FIELD (-1)
#define CHECK_END (-2)





static int compare_ints (int expect, int actual) {
    if (expect == actual) {
        printf ("ok: %d\n", expect);
        return true;
    }
    printf ("mismatch: Expected %d, got %d\n", expect, actual);
    return false;
}




static bool compare_strings (char *expect, char *actual) {
    if (expect == NULL && actual == NULL) {
        printf ("ok: nullptr\n");
        return true;
    }
    if (expect != NULL && actual != NULL) {
        if (strcmp (expect, actual) == 0) {
            printf ("ok: '%s'\n", expect);
            return true;
        }
    }
    printf ("mismatch: '%s'\n"
            "Expected: '%s'\n", actual, expect);
    return false;
}


static bool create_argv_test (void) {
    char **argv;
    char **argr;
    printf ("==============================================\n"
            "Performing test %s\n"
            "----------------------------------------------\n", __func__);
    int argc = fb_create_argv("   test of \"this\" parser, foot\"ball, \"and quot\"ing\" for it!", &argv, &argr);
    bool ok = true;
    ok = compare_ints (8, argc) && ok;
    ok = compare_strings ("test", argv [0]) && ok;
    ok = compare_strings ("test of \"this\" parser, foot\"ball, \"and quot\"ing\" for it!", argr [0]) && ok;
    ok = compare_strings ("of", argv [1]) && ok;
    ok = compare_strings("of \"this\" parser, foot\"ball, \"and quot\"ing\" for it!", argr [1]) &ok;
    ok = compare_strings ("this", argv [2]) && ok;
    ok = compare_strings("\"this\" parser, foot\"ball, \"and quot\"ing\" for it!", argr [2]) & ok;
    ok = compare_strings ("parser,", argv [3]) && ok;
    ok = compare_strings("parser, foot\"ball, \"and quot\"ing\" for it!", argr [3]) & ok;
    ok = compare_strings ("foot\"ball,", argv [4]) && ok;
    ok = compare_strings("foot\"ball, \"and quot\"ing\" for it!", argr [4]) & ok;
    ok = compare_strings ("and quot\"ing", argv [5]) && ok;
    ok = compare_strings("\"and quot\"ing\" for it!", argr [5]) & ok;
    ok = compare_strings ("for", argv [6]) && ok;
    ok = compare_strings("for it!", argr [6]) & ok;
    ok = compare_strings ("it!", argv [7]) && ok;
    ok = compare_strings("it!", argr [7]) & ok;
    ok = compare_strings (NULL, argv [8]) && ok;
    ok = compare_strings(NULL, argr [8]) & ok;
    fb_destroy_argv (argv);

    printf ("Subtest: easy peasy\n");
    argc = fb_create_argv (" \t\r\n easy peasy   ", &argv, &argr);
    ok = compare_ints (2, argc) && ok;
    ok = compare_strings ("easy", argv [0]) && ok;
    ok = compare_strings ("easy peasy   ", argr [0]) && ok;
    ok = compare_strings ("peasy", argv [1]) && ok;
    ok = compare_strings("peasy   ", argr [1]) &ok;
    ok = compare_strings (NULL, argv [2]) && ok;
    ok = compare_strings(NULL, argr [2]) & ok;
    fb_destroy_argv (argv);

    printf ("Subtest: empty command line\n");
    argc = fb_create_argv ("", &argv, &argr);
    ok = compare_ints (0, argc) && ok;
    ok = compare_strings (NULL, argv [0]) && ok;
    ok = compare_strings(NULL, argr [0]) & ok;
    fb_destroy_argv (argv);

    printf ("Subtest: First quoted\n");
    argc = fb_create_argv ("\"First quoted\" next", &argv, &argr);
    ok = compare_ints (2, argc) && ok;
    ok = compare_strings ("First quoted", argv [0]) && ok;
    ok = compare_strings ("\"First quoted\" next", argr [0]) && ok;
    ok = compare_strings ("next", argv [1]) && ok;
    ok = compare_strings("next", argr [1]) &ok;
    ok = compare_strings (NULL, argv [2]) && ok;
    ok = compare_strings(NULL, argr [2]) & ok;
    fb_destroy_argv (argv);

    printf ("Subtest: Whitespace and nothing else\n");
    argc = fb_create_argv (" \t\r", &argv, &argr);
    ok = compare_ints (0, argc) && ok;
    ok = compare_strings (NULL, argv [0]) && ok;
    ok = compare_strings(NULL, argr [0]) & ok;
    fb_destroy_argv (argv);

    printf ("Test %s\n", ok ? "passed" : "FAILED");
    return ok;
}


static bool test_parser (const char *test_name,
                  FB_PARSE_DEFINITION *defs,
                  char *statement,
                  int expected_result, ...) {
    printf ("==============================================\n"
            "Performing test %s\n"
            "----------------------------------------------\n", test_name);
    int test_result = PARSER_CREATE_SUCCESS;
    int success = true;
    FB_PARSER *parser = fb_create_parser ();
    if (!parser) return false;
    if (fb_parser_add_statements(parser, defs)) {
        if (statement) {
            char **argv;
            char **argr;
            int argc = fb_create_argv(statement, &argv, &argr);
            if (argc >= 0) {
                /* Need to initialize errorpoint ourself because we're
                   bypassing the public routine where it gets initialized */
                char *errorpoint = NULL;
                char **argnames = calloc(argc + 1, sizeof (char *));
                if (argnames) {
                    test_result = fb_interpret_recurse (parser, argv, argnames, &errorpoint);
                    va_list extra;
                    va_start (extra, expected_result);
                    for (int check = va_arg (extra, int); check != CHECK_END; check = va_arg (extra, int)) {
                        if (check >= 0) {
                            char *expect = va_arg (extra, char *);
                            printf ("argnames [%d] ", check);
                            compare_strings (expect, argnames [check]) || (success = false);
                        } else if (check == CHECK_ERROR_FIELD) {
                            char *expect = va_arg (extra, char *);
                            printf ("Errorpoint ");
                            compare_strings (expect, errorpoint) || (success = false);
                        }
                    }
                    va_end (extra);
                    free (argnames);
                } else {
                    test_result = ARGV_CREATE_FAILURE;
                }
                fb_destroy_argv(argv);
            } else {
                test_result = ARGV_CREATE_FAILURE;
            };
        }
        fb_parser_destroy(parser);
    } else {
        test_result = PARSER_CREATE_FAIL;
    }
    if (expected_result != test_result) {
        printf ("Expected result %d, got %d\n", expected_result, test_result);
        success = false;
    }
    printf ("Test %s\n", success ? "passed" : "FAILED");
    return success;
}

static bool basic_parser_test (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "" },
        { 2, "HELP [{search}]" },
        { 3, "FOO BAR BAT BANG BAM" },
        { 4, "USER {user} {pass}" },
        { 0, NULL }
    };

    bool success = true;
    /* Test correct statements */
    test_parser ("basic_parser_test-correct-a", parser, "", 1,
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-correct-b1", parser, "HELP", 2,
                 0, NULL,
                 1, "search",
                 CHECK_ERROR_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-correct-b2", parser, "HELP find", 2,
                 CHECK_ERROR_FIELD, NULL,
                 0, NULL,
                 1, "search",
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-correct-c", parser, "FOO BAR bat bang bam", 3,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-correct-d", parser, "USER no baka", 4,
                 0, NULL,
                 1, "user",
                 2, "pass",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);

    /* Now feed it crap */
    test_parser ("basic_parser_test-bad-a", parser, "BADCMD", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "BADCMD",
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-bad-b", parser, "HELP runon stuff", FB_PARSE_EXTRA_TERMS,
                 CHECK_ERROR_FIELD, "stuff",
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-bad-c", parser, "FOO BAR BAKA", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "BAKA",
                 CHECK_END) || (success = false);
    test_parser ("basic_parser_test-bad-d", parser, "FOO BAR BAT", FB_PARSE_INCOMPLETE,
                 CHECK_ERROR_FIELD, "BAT",
                 CHECK_END) || (success = false);
    return success;
}

static bool numeric_fillin_test (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "anything [{#number}]" },
        { 2, "real [{#number:2.71828182844-3.1415926535897932385}]" },
        { 10, "decimal [{#deci:10-12}]" },
        { 16, "auto [{#number:0x01-0x10}]" },
        { 0, NULL }
    };

    bool success = true;
    /* Borrow this parser for one last basic parsing test case */
    test_parser ("basic_parser_test-SPECIAL", parser, "", FB_PARSE_INCOMPLETE,
                 CHECK_END) || (success = false);

    /* Test that everything accepts values at the low end of the range */
    test_parser ("numeric_fillin_test-input-inside-low", parser, "anything 0", 1,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-real-inside-low", parser, "real 2.71828182844", 2,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-inside-low", parser, "decimal 10", 10,
                 0, NULL,
                 1, "deci",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-oct", parser, "auto 01", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-dec", parser, "auto 1", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-hex", parser, "auto 0x01", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);

    /* Test that everything accepts values at the high end of the range */
    test_parser ("numeric_fillin_test-input-inside-high", parser, "anything 409283492384082340349239483284", 1,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-real-inside-high", parser, "real 3.14159265", 2,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-inside-high", parser, "decimal 12", 10,
                 0, NULL,
                 1, "deci",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-oct", parser, "auto 020", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-dec", parser, "auto 16", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-hex", parser, "auto 0x0010", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);

    /* Test that everything denies values outside the low end of the range */
    test_parser ("numeric_fillin_test-input-outside-low", parser, "anything -100000300340303240", 1,
                 0, NULL,
                 1, "number",
                 CHECK_ERROR_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-real-outside-low", parser, "real 2.7182", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "2.7182",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-outside-low", parser, "decimal 0009", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0009",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-oct", parser, "auto 0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-dec", parser, "auto 0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-hex", parser, "auto 0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0",
                 CHECK_END) || (success = false);

    /* Test that everything denies values outside the high end of the range */
    test_parser ("numeric_fillin_test-input-outside-high", parser, "anything -4.3", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "-4.3", /* Unranged input does no decimals. */
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-real-outside-high", parser, "real 3.141592654", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "3.141592654",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-outside-high", parser, "decimal 0013", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0013",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-oct", parser, "auto 021", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "021",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-dec", parser, "auto 17", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "17",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-hex", parser, "auto 0x011", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "0x011",
                 CHECK_END) || (success = false);

    /* Test what happens when we throw text at the fill-ins */
    test_parser ("numeric_fillin_test-input-negative-text", parser, "anything -fred", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "-fred", /* Unranged input does no decimals. */
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-real-plain-text", parser, "real bob", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "bob",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-plain-text", parser, "decimal billy", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "billy",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-number#-with-junk", parser, "auto 09wench", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "09wench",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-negative#-with-junk", parser, "auto -43wrench", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "-43wrench",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-decimal-in-int-blank", parser, "auto 4.3", FB_PARSE_NUMERIC,
                 CHECK_ERROR_FIELD, "4.3",
                 CHECK_END) || (success = false);
    return success;
}


/* Repeat the number fill-in test with negative numbers */
static bool numeric_negative_test (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{#number:-5.0--1.0}]" },
        { 10, "decimal [{#deci:-12--10}]" },
        { 16, "auto [{#number:-0x10--0x01}]" },
        { 0, NULL }
    };

    bool success = true;
    /* Test that everything accepts values at the low end of the range */
    test_parser ("numeric_fillin_test-real-inside-low", parser, "real -5", 2,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-inside-low", parser, "decimal -10", 10,
                 0, NULL,
                 1, "deci",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-oct", parser, "auto -01", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-dec", parser, "auto -1", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-low-hex", parser, "auto -0x01", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);

    /* Test that everything accepts values at the high end of the range */
    test_parser ("numeric_fillin_test-real-inside-high", parser, "real -1", 2,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-inside-high", parser, "decimal -12", 10,
                 0, NULL,
                 1, "deci",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-oct", parser, "auto -020", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-dec", parser, "auto -16", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-inside-high-hex", parser, "auto -0x0010", 16,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);

    /* Test that everything denies values outside the low end of the range */
    test_parser ("numeric_fillin_test-real-outside-low", parser, "real -5.00001", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-5.00001",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-outside-low", parser, "decimal -0009", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0009",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-oct", parser, "auto -0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-dec", parser, "auto -0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-low-hex", parser, "auto -0", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0",
                 CHECK_END) || (success = false);

    /* Test that everything denies values outside the high end of the range */
    test_parser ("numeric_fillin_test-real-outside-high", parser, "real -.99999", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-.99999",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-decimal-outside-high", parser, "decimal -0013", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0013",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-oct", parser, "auto -021", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-021",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-dec", parser, "auto -17", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-17",
                 CHECK_END) || (success = false);
    test_parser ("numeric_fillin_test-auto-outside-high-hex", parser, "auto -0x011", FB_PARSE_RANGE,
                 CHECK_ERROR_FIELD, "-0x011",
                 CHECK_END) || (success = false);
    return success;
}

#ifdef NDEBUG
static bool optional_fillin_not_last_term (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{number}] foobar" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}


static bool alongside_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{number}]" },
        { 1, "real alongside is bad" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
static bool alongside_fillin_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real alongside is bad" },
        { 2, "real [{number}]" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}


static bool elipsis_alongside_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{number}]" },
        { 1, "real ..." },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
static bool elipsis_alongside_fillin_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real ..." },
        { 2, "real [{number}]" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}


static bool elipsis_alongafter_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real {number}" },
        { 1, "real {number} ..." },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
static bool elipsis_alongafter_fillin_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real {number} ..." },
        { 2, "real {number}" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
#endif

static bool elipsis_after_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{number}] ..." },
        { 0, NULL }
    };
    bool success = true;
    test_parser ("elipsis_after_fillin-a", parser, "real", 2,
        CHECK_NEXT_FIELD, NULL,
        CHECK_END) || (success = false);
    test_parser ("elipsis_after_fillin-b", parser, "real fish", 2,
                 0, NULL,
                 1, "number",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("elipsis_after_fillin-c", parser, "real fish swim", 2,
                 0, NULL,
                 1, "number",
                 2, "...",
                 CHECK_NEXT_FIELD, "swim",
                 CHECK_END) || (success = false);
    return success;
}

#ifdef NDEBUG
static bool elipsis_alongside_fillin_redefined (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real" },
        { 2, "real [{number}] ..." },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
static bool elipsis_alongside_fillin_redefined_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 2, "real [{number}] ..." },
        { 1, "real" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
#endif

static bool longer_alongside_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real {number} foo ..." },
        { 3, "real {number}" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_SUCCESS);
}
static bool longer_alongside_fillin_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real {number} foo ..." },
        { 2, "real {number}" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_SUCCESS);
}

#ifdef NDEBUG
static bool optional_longer_alongside_fillin (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real {number} [foo] ..." },
        { 3, "real {number}" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
static bool optional_longer_alongside_fillin_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "real {number} [foo] ..." },
        { 2, "real {number}" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
#endif

static bool named_parallel_alternation (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish <type:trout|salmon> dinner" },
        { 2, "fish <type:flounder|halibut> lunch" },
        { 3, "fish <type:bass|flounder|perch> snack" },
        { 4, "fish <type:one> breakfast" },
        { 0, NULL }
    };
    bool success = true;
    test_parser ("named_parallel_alternation-not-optional", parser, "fish dinner", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "dinner",
                 CHECK_END);
    test_parser ("named_parallel_alternation-dinner", parser, "fish trout dinner", 1,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-lunch", parser, "fish halibut lunch", 2,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-lunch", parser, "fish one breakfast", 4,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-lunch-dinner-crossed", parser, "fish halibut dinner", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "dinner",
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-flounder-lunch", parser, "fish flounder lunch", 2,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-flounder-snack", parser, "fish flounder snack", 3,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-halibut-snack", parser, "fish halibut snack", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "snack",
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-bass-lunch", parser, "fish bass lunch", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "lunch",
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-bass-snack", parser, "fish bass snack", 3,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_parallel_alternation-halibut-lunch", parser, "fish halibut lunch", 2,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    return success;
}

static bool named_parallel_alternation_reversed (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 3, "fish <type:bass|bullhead|perch> snack" },
        { 1, "fish <type:trout|salmon> dinner" },
        { 2, "fish <type:flounder|halibut> lunch" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_SUCCESS);
}

#ifdef NDEBUG
static bool mixnamed_parallel_alternation (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish <dinnertype:trout|salmon> dinner" },
        { 2, "fish <lunchtype:flounder|halibut|salmon> lunch" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_FAIL);
}
#endif





static bool separatelynamed_parallel_alternation (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish <dinnertype:trout|salmon> dinner" },
        { 2, "fish <lunchtype:flounder|halibut> lunch" },
        { 0, NULL }
    };
    return test_parser (__func__, parser, NULL, PARSER_CREATE_SUCCESS);
}



static bool named_optionals (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish [type:salmon] dinner" },
        { 2, "fish [type:flounder|halibut] lunch" },
        { 3, "fish [type:bass|flounder|perch] snack" },
        { 0, NULL }
    };
    bool success = true;
    test_parser ("named_optionals-dinner", parser, "fish dinner", 1,
                 0, NULL,
                 1, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_optionals-salmon-dinner", parser, "fish salmon dinner", 1,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_optionals-lunch", parser, "fish lunch", 2,
                 0, NULL,
                 1, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_optionals-halibut-lunch", parser, "fish halibut lunch", 2,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_optionals-flounder-lunch", parser, "fish flounder lunch", 2,
                 0, NULL,
                 1, "type",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("named_optionals-lunch-dinner-crossed", parser, "fish halibut snack", FB_PARSE_INVALID_KEYWORD,
                 CHECK_ERROR_FIELD, "snack",
                 CHECK_END) || (success = false);

    return success;
}

static bool autonamed_optionals (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish [salmon] dinner" },
        { 0, NULL }
    };
    bool success = true;
    test_parser ("autonamed_optionals-dinner", parser, "fish dinner", 1,
                 0, NULL,
                 1, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("autonamed_optionals-salmon-dinner", parser, "fish salmon dinner", 1,
                 0, NULL,
                 1, "salmon",
                 2, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    return success;
}

static bool shorter_with_optional_and_longer (void) {
    FB_PARSE_DEFINITION parser[] = {
        { 1, "fish dinner [{fishtype}]" },
        { 2, "fish dinner {fishtype} for {who}" },
        { 0, NULL }
    };
    bool success = true;
    test_parser ("shorter_with_optional_and_longer-a", parser, "fish dinner", 1,
                 0, NULL,
                 1, NULL,
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("shorter_with_optional_and_longer-b", parser, "fish dinner gefilte", 1,
                 0, NULL,
                 1, NULL,
                 2, "fishtype",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    test_parser ("shorter_with_optional_and_longer-c", parser, "fish dinner gefilte for", FB_PARSE_INCOMPLETE,
                 CHECK_ERROR_FIELD, "for",
                 CHECK_END) || (success = false);
    test_parser ("shorter_with_optional_and_longer-d", parser, "fish dinner gefilte for baka", 2,
                 0, NULL,
                 1, NULL,
                 2, "fishtype",
                 3, NULL,
                 4, "who",
                 CHECK_NEXT_FIELD, NULL,
                 CHECK_END) || (success = false);
    return success;
}


int main(void) {
    // insert code here...
    bool success = true;

    for (int i = 100; i > 0; i--) {
        create_argv_test() || (success = false);
    }
    for (int i = 100; i > 0; i--) {
        basic_parser_test () || (success = false);
    }
    numeric_fillin_test() || (success = false);
    numeric_negative_test() || (success = false);

#ifdef NDEBUG
    /* These tests test the parser builder's error handling.  */
    /* NDEBUG must be set or the parser will assert(). */
    optional_fillin_not_last_term() || (success = false);
    alongside_fillin() || (success = false);
    alongside_fillin_reversed() || (success = false);
    elipsis_alongside_fillin() || (success = false);
    elipsis_alongside_fillin_reversed() || (success = false);
    elipsis_alongafter_fillin() || (success = false);
    elipsis_alongafter_fillin_reversed() || (success = false);
    mixnamed_parallel_alternation() || (success = false);
    elipsis_alongside_fillin_redefined() || (success = false);
    elipsis_alongside_fillin_redefined_reversed() || (success = false);
    optional_longer_alongside_fillin() || (success = false);
    optional_longer_alongside_fillin_reversed() || (success = false);
#endif

    elipsis_after_fillin() || (success = false);
    longer_alongside_fillin() || (success = false);
    longer_alongside_fillin_reversed() || (success = false);
    shorter_with_optional_and_longer () || (success = false);
    named_parallel_alternation() || (success = false);
    named_parallel_alternation_reversed() || (success = false);
    separatelynamed_parallel_alternation() || (success = false);
    named_optionals () || (success = false);
    autonamed_optionals () || (success = false);

    if (success) {
        printf ("\n\nAll tests passed!\n");
    } else {
        printf ("\n\nSome tests FAILED!\n");
    }
#ifdef NDEBUG
    printf ("HOWEVER: Not NDEBUG mode, invalid parse definitions cannot be tested.\n"
            "Please recompile with NDEBUG and run again.\n");
#endif
    return !success;
}
