///
/// Football option parser template.
/// C++ classes implementing an options parser.
/// @file       fb_optionparser.cpp - Football socket abstraction layer
/// @author     Perette Barella
/// @date       2016-04-18
/// @copyright  Copyright 2014–2017 Devious Fish. All rights reserved.
///

#include <config.h>

#include "footparser.h"
#include "football.h"

namespace Football {


    GenericOptionParser::GenericOptionParser () {
        if (!parser) {
            throw std::bad_alloc();
        }
    }
    GenericOptionParser::~GenericOptionParser () {
        fb_parser_destroy (parser);
    }

    /// Add statements to the parser.
    /// @param statements An array of statement definitions to add.
    /// @return True on success, false on failure.
    bool GenericOptionParser::addStatements (const FB_PARSE_DEFINITION *statements) {
        return fb_parser_add_statements (parser, statements);
    }

    /// @internal Find the index position of a named argument.
    int GenericOptionParser::argnPosition (const char *name) {
        assert (name);
        int pos = offset;
        while (_argv [pos] != nullptr) {
            if (_argn [pos] && strcasecmp (name, _argn [pos]) == 0)
                return pos;
            pos++;
        }
        return -1;
    }
    /** @internal Create/expand argv/argn arrays to accommodate a command line.
     @return true if success, false on error. */
    bool GenericOptionParser::ensureCapacity (int size) {
        if (size < capacity) return true;
        if (_argv) delete _argv;
        if (_argn) delete _argn;
        _argv = nullptr;
        _argn = nullptr;
        capacity = size + 25;
        try {
            _argv = new char *[capacity + 1]; // Keep space for terminating null
            _argn = new char *[capacity + 1];
            return true;
        } catch (const std::bad_alloc &) {
            capacity = 0;
            return false;
        }
    }

    /** Interpret options.
     @param options A tokenized set of command line pieces, in sequential order.
     @param result Storage for option values.
     @param conn A connection associated with the parsing.
     If provided, encountered errors are reported to the connection before return.
     @return FB_PARSE_SUCCESS or a FB_PARSE_ERROR value.
     @see FB_PARSE_ERROR */
    int GenericOptionParser::interpret (const std::vector<std::string> &options, void *result, Connection *conn) {
        if (options.size() == 0) return FB_PARSE_SUCCESS;
        if (!ensureCapacity ((int) options.size())) return FB_PARSE_FAILURE;
        for (size_t i = 0; i < options.size(); i++) {
            _argv [i] = (char *) options [i].c_str();
            _argn [i] = nullptr;
        }
        _argv [options.size()] = nullptr;
        _argn [options.size()] = nullptr;
        offset = 0;
        while (_argv [offset]) {
            assert ((unsigned long) offset < options.size());
            char *error = nullptr;
            int opt = fb_interpret_recurse (parser, _argv + offset, _argn + offset, &error);
            if (opt < 0) {
                if (conn) {
                    conn->commandError (opt, error);
                }
                return opt;
            }
            int status = invokeHandleOption (opt, result);
            if (status != FB_PARSE_SUCCESS) {
                if (conn) {
                    conn->commandError (status, _argv [offset]);
                }
                return status;
            }
            // Offset until we find the terminator ("error") term.
            assert (_argv [offset] != error);
            while (_argv [offset] != error) {
                assert (_argv [offset]);
                offset++;
            }
        }
        return FB_PARSE_SUCCESS;
    }

    /** Retrieve the value of a named element.
     @param name The element name.
     @return The named argument's value, or NULL if it is not present. */
    const char *GenericOptionParser::argv (const char *name) {
        int pos = argnPosition (name);
        if (pos < 0) return nullptr;
        return _argv [pos];
    }
    /** Retrieve an element by index.
     @param index The element position.
     @return The argument's value. */
    const char *GenericOptionParser::argv (int index) {
#ifndef NDEBUG
        for (int i = 0; i < index; i++) {
            assert (_argv [offset + i]);
        }
#endif
        return _argv [offset + index];
    }
    /** Indicate if a named element has a value.
     @param name The element name.
     @param value The expected value.  May be NULL to expect value is not present.
     @return true if values match, false otherwise. */
    bool GenericOptionParser::argvEquals (const char *name, const char *value) {
        const char *actual = argv (name);
        if (value == nullptr && actual == nullptr) return true;
        if (value == nullptr || actual == nullptr) return false;
        return (strcasecmp (value, actual) == 0);
    }


}
