///
/// Pianod specializations of Football connections & services.
///	@file		connection.h - pianod
///	@author		Perette Barella
///	@date		2014-11-28
///	@copyright	Copyright (c) 2014-2017 Devious Fish. All rights reserved.
///

#ifndef __pianod__connection__
#define __pianod__connection__

#include <config.h>

#include <football.h>

#include "fundamentals.h"
#include "user.h"

class PianodService;
namespace Media {
    class Source;
}
class ThingieList;

/// Details of an event being awaited.
class WaitEvent {
public:
    /// Kinds of events that can be waited for.
    enum class Type {
        None,
        SourceReady,
        TrackEnded,
        TrackStarted
    };
    Type event = Type::None; ///< The kind of event being awaited.
    const void *parameter; ///< Identifies a specific event instance.
    time_t timeout = 0; ///< Time at which to stop waiting.
    bool close_after_event = false; ///< True if connection is 'AS USER' and should closed after event.

    ///< Next timeout time, here to avoid unnecessary recalculation.
    static time_t nextTimeout;
};


/** Connection to a pianod client, along with context and state
 of that connection. */
class PianodConnection : public Football::Connection {
    // Override original event handlers 
    virtual void newConnection () override;
    virtual void connectionClose () override;

    // Connection state
    Media::Source *_source = nullptr; ///< The current source.
    WaitEvent pending; ///< An event this connection is waited for.

    // Service-wide parameters
    static bool broadcast_user_actions;

public:
    virtual ~PianodConnection () override;

    // Now add pianod junk
    // Visibility of this item should be adjusted later??? ZZZZZ 
    User *user = nullptr;

    virtual void commandError(ssize_t reason, const char *where) override;
    virtual void commandException (std::exception_ptr except) override;
    virtual void permissionDenied() override;

    void close_after_events ();
    /** Determine if the connection is authenticated, i.e., has a user.
        @return true if the connection is authenticated. */
    inline bool authenticated (void) { return user != nullptr; };
    void updateConnection ();
    inline std::string username (void) { return authenticated() ? user->username() : "A visitor"; };
    /** Get the connection's selected source.
        @return The selected source. */
    inline Media::Source *source() const { return _source; };
    /** Set the selected source.
        @param source The new source.
        @param announce True (or omitted) to announce the new source in protocol. */
    inline void source (Media::Source *source, bool announce = true) {
        _source = source;
        if (announce) sendSelectedSource();
    };
    bool haveRank (Rank rank);
    Rank effectiveRank (void);
    bool havePrivilege (Privilege priv);

    bool waitForEventWithOptions (WaitEvent::Type type, const void *detail);
    void waitForEvent (WaitEvent::Type type, const void *detail);
    void event (WaitEvent::Type type, const void *detail, RESPONSE_CODE reply);
    void checkTimeouts ();

    void sendSelectedSource ();
    void sendEffectivePrivileges ();
    void announce (RESPONSE_CODE code, const char *parameter = nullptr);
    inline void announce (RESPONSE_CODE code, const std::string &param) {
        announce (code, param.c_str());
    }
    /** Enable or disable announcing user actions to all connections.
        @param v True to enable, false to disable. */
    inline void broadcastUserActions (bool v) {
        broadcast_user_actions = v;
    }
    /** Check whether user actions are announced to all connections.
        @return True if enabled, false if not. */
    inline bool broadcastingActions (void) {
        return broadcast_user_actions;
    }
    inline PianodService &service (void) {
        return *((PianodService *) Football::Connection::service());
    }
};


class AudioEngine;
class ServiceManager;

/// Pianod service, a customized FootballService for Pianod connections.
class PianodService : public Football::Service<PianodConnection> {
    friend class ServiceManager;
private:
    AudioEngine *engine = nullptr;
    const std::string room_name;
public:
    ~PianodService ();
    PianodService (const FB_SERVICE_OPTIONS &options,
                   const std::string &room,
                   PianodService *parent) : Service (options, parent), room_name (room) { };
    inline AudioEngine *audioEngine() { return engine; };
    void serviceShutdown (void);
    inline const std::string &roomName (void) { return room_name; };
    void usersChangedNotification (void);
};

#endif // defined(__pianod__connection__) 
