/*
 *  fb_service.h
 *  Player
 *
 *  Created by Perette Barella on 2012-03-03.
 *  Copyright 2012-2014 Devious Fish. All rights reserved.
 *
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdbool.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "fb_public.h"

#ifndef __FB_SERVICE__
#define __FB_SERVICE__


typedef enum fb_socket_type_t {
	FB_SOCKTYPE_SERVICE = 0x3692, /* Random numbers */
	FB_SOCKTYPE_CONNECTION = 0x5285,
	FB_SOCKTYPE_USER = 0xa9f7,
	FB_SOCKTYPE_EVENT = 0xbd53
} FB_SOCKETTYPE;

typedef enum fb_socket_state_t {
    FB_SOCKET_STATE_GREETING = 0,
    FB_SOCKET_STATE_GATHERING_HEADER,
	FB_SOCKET_STATE_OPEN,
	FB_SOCKET_STATE_FLUSHING,
	FB_SOCKET_STATE_CLOSING
} FB_SOCKETSTATE;

typedef enum fb_socketid_t {
    FB_SOCKET_LINE_IP4,
    FB_SOCKET_LINE_IP6,
    FB_SOCKET_HTTP_IP4,
    FB_SOCKET_HTTP_IP6,
    FB_SOCKET_COUNT
} FB_SOCKETID;

typedef union fb_socketaddr_t {
    struct sockaddr_in ip4;
#ifdef HAVE_IPV6
	struct sockaddr_in6 ip6;
#endif
} FB_SOCKETADDR;

#define fb_http_socket(id) ((id) == FB_SOCKET_HTTP_IP4 || (id) == FB_SOCKET_HTTP_IP6)
#define fb_ip6_socket(id) ((id) == FB_SOCKET_LINE_IP6 || (id) == FB_SOCKET_HTTP_IP6)

struct fb_service_t {
	FB_SOCKETTYPE type;
	FB_SOCKETSTATE state;	/* Is the service open or closing? */
    FB_SERVICE_OPTIONS options;
	int socket [FB_SOCKET_COUNT]; /* Stuff about the sockets */
	FB_SOCKETADDR address [FB_SOCKET_COUNT];
	size_t connection_count;	/* Number of active connections in the collection */
	size_t connections_size;	/* Total number of slots in the collection */
	struct fb_connection_t **connections;	/* Connections to services */
	struct fb_service_t *next; /* For reap queue */
};

/* Nomenclature:
   Message - Contains the message data and a use count, allowing
             use in several Qs for efficiency.
   Q a/k/a MessageList - Per-connection linked list of Messages.
   Queue - header data structure for Q MessageList.
 */
/* Message structure.  May appear in multiple connections' queues. */
typedef struct fb_message_t {
	int usecount;
	ssize_t length;
	char *message;
} FB_MESSAGE;

/* Q list structure.  Per-connection list of its messages. */
typedef struct fb_messagelist_t {
    struct fb_messagelist_t *next;
	FB_MESSAGE *message;
} FB_MESSAGELIST;

/* Queue structure */
typedef struct fb_ioqueue_t {
    FB_MESSAGELIST *first;
    FB_MESSAGELIST *last;
    ssize_t consumed; /* */
} FB_IOQUEUE;

typedef struct fb_inputbuffer_t {
    size_t size;
    size_t capacity;
    char *message;
} FB_INPUTBUFFER;

typedef struct http_request_t {
    bool unsupported;
    bool headonly;
	char *http;
    char *host;
	char *service_name;
    char *filename;
	char *upgrade_type;
	char *websocket_key;
	char *websocket_protocol;
	char *websocket_version;
    bool invalid;
} FB_HTTPREQUEST;

struct fb_connection_t {
	FB_SOCKETTYPE type;
	FB_SERVICE *service;	/* Service to which connection is made */
	int socket;	
	FB_SOCKETSTATE state;	/* Are we open, flushing, or closing? */
    bool greeted; /* Did we already get a greeting in ALLOW mode? */
    bool http;
    FB_HTTPREQUEST request;
    FB_IOQUEUE assembly;
    FB_IOQUEUE out;
    FB_INPUTBUFFER in;
	int domain;
	union {
		struct sockaddr_in ip4addr;
#ifdef HAVE_IPV6
		struct sockaddr_in6 ip6addr;
#endif
	} origin;
	char *filename;		/* For file connections */
	FILE *file;			/* read() on file doesn't handle canonical stuff */
	void *context;		/* For user use, if desired/needed */
};

/* User iterators */
struct fb_iterator_t {
	FB_SERVICE *service;
	ssize_t iteration;
};


/* Message data management */
extern void fb_free_freelists ();
extern FB_MESSAGE *fb_messagealloc();
extern void fb_messagefree(FB_MESSAGE *freethis);

extern FB_MESSAGELIST *fb_qalloc();
extern void fb_qfree (FB_MESSAGELIST *freethis);

extern bool fb_queue_add (FB_IOQUEUE *queue, FB_MESSAGE *message);
extern bool fb_queue_empty (FB_IOQUEUE *q);
extern void fb_queue_consume (FB_IOQUEUE *q, size_t consume);
extern void fb_queue_destroy (FB_IOQUEUE *q);

/* Destroy the service when all connections are closed. */
extern void fb_destroy_service (struct fb_service_t *service);
extern void fb_schedule_reap (FB_SERVICE *service);

/* Register/Unregister a connection with the socket manager */
extern void fb_queue_event (FB_EVENT *event);
extern void fb_dispose_event (FB_EVENT *event);
extern bool fb_register (int socket, FB_SOCKETTYPE type, void *thing);
extern void fb_unregister (int socket);

/* Accept/Close & release a socket and its resources */
extern FB_CONNECTION *fb_accept_connection (FB_SERVICE *service, FB_SOCKETID id);
extern void fb_destroy_connection (FB_CONNECTION *connection);

/* Event handling functions */
extern FB_EVENT *fb_read_input (FB_EVENT *event, FB_CONNECTION *connection);
extern FB_EVENT *fb_new_connect (FB_EVENT *event, FB_SERVICE *service);
extern FB_EVENT *fb_send_output (FB_EVENT *event, FB_CONNECTION *connection);

/* HTTP & Websocket support */
extern FB_EVENT *fb_read_http_input (FB_EVENT *event, FB_CONNECTION *connection);
extern bool fb_http_encode (FB_CONNECTION *connection);
extern void fb_destroy_httprequest (FB_HTTPREQUEST *request);
extern void fb_collect_http_request (FB_EVENT *event, FB_HTTPREQUEST *request);
extern void fb_collect_http_parameter (char *line, FB_HTTPREQUEST *request);
extern bool fb_http_command (const char *command);
extern FB_EVENT *fb_execute_http_request (FB_EVENT *event, FB_CONNECTION *connection);


/* Enabling/disabling events for the socket manager */
extern void fb_set_readable (int socket, bool enable);
extern void fb_set_writable (int socket, bool enable);

/* Command line parsing */
int fb_create_argv (const char *commandline, char ***result);
void fb_destroy_argv (char **argv);

#endif
