/*
 *  fb_handle_event.c
 *  Player
 *
 *  Created by Perette Barella on 2012-03-03.
 *  Copyright 2012 Devious Fish. All rights reserved.
 *
 */

#include <config.h>

#ifndef __FreeBSD__
#define _BSD_SOURCE /* snprintf() */
#endif
#if !defined(__FreeBSD__) && !defined(__APPLE__)
#define _POSIX_C_SOURCE 1 /* required by getaddrinfo() */
#endif



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <stdbool.h>

#include <assert.h>

#ifndef HAVE_FGETLN
#include "fgetln.h"
#endif

#include "fb_service.h"


/* Create and populate a new message block. */
static FB_MESSAGE *fb_create_message (const char *format, va_list parameters) {
	FB_MESSAGE *message;
	if ((message = fb_messagealloc ())) {
		message->length = vasprintf(&message->message, format, parameters);
		if (message->length >= 0) {
			return message;
		}
		fb_messagefree(message);
	}
	return NULL;
}


/* Put a message into a connection's queue.
   Return length on success, -1 on failure. */
static ssize_t fb_queue_single (FB_CONNECTION *connection, FB_MESSAGE *message) {
	ssize_t length = message->length;
	if (length == 0 || connection->state == FB_SOCKET_STATE_CLOSING ||
        connection->state < FB_SOCKET_STATE_OPEN || connection->filename) {
		/* Don't queue empty messages or queue stuff for an input file, or if the connection
           isn't fully open or is closing, but feign success in all these cases. */
		fb_messagefree (message);
		return length;
	}
    if (connection->http) {
        if (fb_queue_add (&connection->assembly, message)) {
            fb_http_encode (connection);
        } else {
            length = -1;
        }
    } else {
        if (!fb_queue_add (&connection->out, message)) {
            length = -1;
        }
    }
    /* This connection has nothing queued.  Try to send now, bypassing queueing. */
    fb_send_output (NULL, connection);
	return length;
}

/* Put a message into all a service's queues.
 Return length on success (queued to all connections successfully).
 On failure or partial failure, return -1. */
static ssize_t fb_queue_broadcast (FB_SERVICE *service, FB_MESSAGE *message) {
	size_t length = message->length;
	if (message->length == 0) {
		/* Don't queue empty messages, but feign success. */
		fb_messagefree (message);
		return 0;
	}
	unsigned int i;
	for (i = 0; i < service->connection_count; i++) {
		if (service->connections[i]->state != FB_SOCKET_STATE_CLOSING && !service->connections[i]->filename) {
			message->usecount++;
			if (fb_queue_single (service->connections[i], message) == -1) {
				length = -1;
			}
		}
	}
	/* When allocated, use count is already 1.  Correct for this now. */
	fb_messagefree (message);
	return length;
}

/* Add a message message to a service, connection, or event,
   because polymorphism is friendly.  Perhaps I should be using an object language?
   Looks at the type magic number inside the thing to determine which kind it is.
   Return the length of the message, or -1 on error, as per stdio.
   In case of only partial success... Success. */
ssize_t fb_queue_message (void *thing, FB_MESSAGE *message, bool broadcast) {
	assert (thing);
	assert (message);
	FB_SOCKETTYPE type = *(FB_SOCKETTYPE *)thing;
	if (message) {
		switch (type) {
			case FB_SOCKTYPE_SERVICE:
				return fb_queue_broadcast (((FB_SERVICE *)thing), message);
			case FB_SOCKTYPE_CONNECTION:
				if (broadcast) {
					return fb_queue_broadcast (((FB_CONNECTION *)thing)->service, message);
				}
				return fb_queue_single ((FB_CONNECTION *)thing, message);
			case FB_SOCKTYPE_EVENT:
				if (broadcast) {
					return fb_queue_broadcast (((FB_EVENT *)thing)->connection->service, message);
				}
				return fb_queue_single (((FB_EVENT *)thing)->connection, message);
			default:
				/* Nothing */
				break;
		}
		assert (0);
		fprintf (stderr, "fb_queue_message: thing of unknown type %d passed\n", (int) type);
		fb_messagefree (message);
	}
	return -1;
}
	
/* Add messages to output queues in various forms and styles.
   All forms take an opaque type and determine if it's a connection
   or service, so you can write code once and deal with both
   single and broadcast messages.
   'b' variants broadcast, given either a connection or service.
	   Services always broadcast, regardless of service.
   'v' accepts a pointer to the format arguments. */
ssize_t fb_fprintf (void *thing, const char *format, ...) {
	va_list parameters;
	va_start(parameters, format);
	FB_MESSAGE *message = fb_create_message (format, parameters);
	va_end (parameters);
	return fb_queue_message (thing, message, false);
}

ssize_t fb_vfprintf (void *thing, const char *format, va_list parameters) {
	FB_MESSAGE *message = fb_create_message (format, parameters);
	return fb_queue_message (thing, message, false);
}
	
ssize_t fb_bfprintf (void *thing, const char *format, ...) {
	va_list parameters;
	va_start(parameters, format);
	FB_MESSAGE *message = fb_create_message (format, parameters);
	va_end (parameters);
	return fb_queue_message (thing, message, true);
}

ssize_t fb_bvfprintf (void *thing, const char *format, va_list parameters) {
	FB_MESSAGE *message = fb_create_message (format, parameters);
	return fb_queue_message (thing, message, true);
}



/* Write output to a connection from the queue.  Events are generated for connection
   closure only; there are none for writing.  If a NULL event is passed then these
   are delayed (along with moving state toward closure) until a later time, when
   an event can be returned for those state changes.
   If the socket is in a closing state, when the buffer is empty, close the connection. */
FB_EVENT *fb_send_output (FB_EVENT *event, FB_CONNECTION *connection) {
	assert (connection);
	if (fb_queue_empty (&connection->out)) {
        if (!event) return NULL;
		/* Output queue is empty. */
		if (connection->state == FB_SOCKET_STATE_FLUSHING) {
			/* The output buffer is empty, send a close event before finalizing */
			connection->state = FB_SOCKET_STATE_CLOSING;
			event->type = FB_EVENT_CLOSE;
			return event;
		} else if (connection->state == FB_SOCKET_STATE_CLOSING) {
			/* If the service is closing and this is the last connection, schedule reap */
			if (connection->service->connection_count == 1 &&
				connection->service->state == FB_SOCKET_STATE_CLOSING) {
				fb_schedule_reap (connection->service);
			}
			/* Disassemble the connection */
			fb_unregister (connection->socket);
			fb_destroy_connection (connection);
		} else {
			/* The queue is empty, turn off write monitoring */
			fb_set_writable (connection->socket, false);
		}
	} else {
        ssize_t written;
		do {
            FB_MESSAGE *message = connection->out.first->message;
            size_t outlen = message->length - connection->out.consumed;
            written = write (connection->socket, message->message+connection->out.consumed,outlen);
            if (written >= 0) {
                fb_queue_consume (&connection->out, written);
            } else {
                /* Socket has closed/connection is lost. */
                fb_queue_destroy (&connection->out);
                fb_close_connection (connection);
                break;
            }
        } while (written > 0 && !fb_queue_empty (&connection->out));
	}
	return NULL;
}

/* Read input from a connection, return it as an event.
   The event has the raw command in 'command'.
   The event has a broken up command in 'argv'.  On failure, 'argv' will be null.
   If the event is EOF, first time change the socket state and return an event.
   If it's another EOF, then close and destroy the connection.
 */
FB_EVENT *fb_read_line_input (FB_EVENT *event, FB_CONNECTION *connection) {
	assert (connection);
	assert (event);
	char *buffer = NULL;
	size_t buffilled = 0;
	
	/* If we got here, socket_data will be for a connection */
	event->type = FB_EVENT_INPUT;
	char *line = fgetln (connection->file, &buffilled);
	if (!line || buffilled == 0) {
		/* EOF/Connection closed from other end. Initiate closure. */
        fb_close_connection (connection);  /* Not _now variant! */
		return NULL;
	}

	/* Make a copy of the line */
	buffer = malloc (buffilled + 1);
	if (!buffer) {
		perror ("fb_read_input:malloc");
		return NULL;
	}
	memcpy (buffer, line, buffilled);
	/* strip returns off the end of the unparsed line */
	while (buffilled > 0 && (buffer [buffilled - 1] == '\r' || buffer [buffilled - 1] == '\n')) {
		buffilled--;
	}
	buffer [buffilled] = '\0';
	
	/* Store the command and parse it into an argv array */
	event->command = buffer;
	event->argc = fb_create_argv(event->command, &event->argv);
	if (event->argc < 0) {
		event->argc = 0;
	}
	
	return (event);
}

/* Read input based on protocol and state. */
FB_EVENT *fb_read_input (FB_EVENT *ev, FB_CONNECTION *connection) {
    FB_SERVICE *service = connection->service;
    FB_EVENT *event;
    switch (connection->state) {
        case FB_SOCKET_STATE_GREETING:
            event = fb_read_line_input (ev, connection);
            if (event && event->argc) {
                if (service->options.greeting_mode != FB_GREETING_OFF &&
                    strcasecmp (event->argv [0], service->options.greeting) == 0) {
                    connection->state = FB_SOCKET_STATE_OPEN;
                    connection->http = false;
                    event->type = FB_EVENT_CONNECT;
                    return (event);
                }
                if (fb_http_command (event->argv [0])) {
                    fb_collect_http_request (event, &connection->request);
                    connection->state = FB_SOCKET_STATE_GATHERING_HEADER;
                    connection->http = true;
                    return NULL;
                }
                if (service->options.greeting_mode == FB_GREETING_FALLBACK) {
                    /* Make a copy of the event and queue it for delivery later.a */
                    static FB_EVENT message;
                    fb_dispose_event(&message);
                    message = *event;
                    fb_queue_event (&message);
                    /* Return a connection event now. */
                    event->argv = NULL;
                    event->argc = 0;
                    event->command = NULL;
                    connection->state = FB_SOCKET_STATE_OPEN;
                    event->type = FB_EVENT_CONNECT;
                    connection->http = false;
                    return (event);
                }
                /* Protocol error.  Report error and drop connection. */
                fb_unregister (connection->socket);
                fb_destroy_connection(event->connection);
            }
            return NULL;
        case FB_SOCKET_STATE_GATHERING_HEADER:
            event = fb_read_line_input (ev, connection);
            if (event && event->argc) {
                /* Stash line in header collection */
                fb_collect_http_parameter (event->command, &connection->request);
                return NULL;
            }
            /* On end of header, do something. */
            return event ? fb_execute_http_request (event, connection) : NULL;
        case FB_SOCKET_STATE_OPEN:
            if (!connection->http && service->options.greeting_mode == FB_GREETING_ALLOW && !connection->greeted) {
                event = fb_read_line_input (ev, connection);
                connection->greeted = (event != NULL);
                return event && event->argc && strcasecmp (event->argv [0], service->options.greeting) == 0 ?
                        NULL : event;
            }
            return connection->http ? fb_read_http_input (ev, connection) :
                                      fb_read_line_input (ev, connection);
        case FB_SOCKET_STATE_FLUSHING:
            assert (0); /* Could happen in rare case user code re-enables socket */
			fb_set_readable (connection->socket, false);
            return NULL;
        default:
            assert (0);
            return NULL;
    }
    assert (0);
    return NULL;
}

/* Accept a new connection and return an event announcing it. */
FB_EVENT *fb_new_connect (FB_EVENT *event, FB_SERVICE *service) {
	assert (event);
	assert (service);
    FB_SOCKETID id;
    for (id = 0; id < FB_SOCKET_COUNT; id++) {
        if (service->socket [id] == event->socket) break;
    }
    assert (id != FB_SOCKET_COUNT);
	event->connection = fb_accept_connection (service, id);
	if (event->connection) {
		if (fb_register(event->connection->socket, FB_SOCKTYPE_CONNECTION, event->connection)) {
			event->socket = event->connection->socket;
			event->context = event->connection->context;
			event->service = service;
			event->type = FB_EVENT_CONNECT;
            if (fb_http_socket (id) || (service->options.greeting_mode != FB_GREETING_OFF && service->options.greeting_mode != FB_GREETING_ALLOW)) {
                return NULL; /* Don't connect until greeting or handshake complete. */
            }
            event->connection->state = FB_SOCKET_STATE_OPEN;
			return (event);
		}
		fb_destroy_connection(event->connection);
	}
	return NULL;
}



