/*
 *  fb_socketmgr.c
 *  Player
 *
 *  Created by Perette Barella on 2012-03-03.
 *  Copyright 2012 Devious Fish. All rights reserved.
 *
 */

#include <config.h>

#ifndef __FreeBSD__
#define _POSIX_C_SOURCE 1 /* fileno() */
#define _BSD_SOURCE /* strdup() */
#define _DARWIN_C_SOURCE /* strdup() on OS X */
#endif
#ifdef __linux__
#include <linux/posix_types.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/select.h>

#include <assert.h>

#include "fb_service.h"

/* Some Linuxes seem to be missing these */
#ifndef FD_COPY
#define FD_COPY(from,to) memmove(to, from, sizeof(*(from)))
#endif

/* The strategy here is to just register sockets in an array by their socket number.
   When socket N needs attention, it's easy and fast to find.  We also store the
   socket number in with that data, so can refer back. */
typedef struct socket_data_t {
	int socket;
	FB_SOCKETTYPE type;
	union {
		FB_CONNECTION *connection;
		FB_SERVICE *service;
		void *user;
	} thingie; /* Holds one of those thingies, y'know? */
} FB_SOCKET_DATA;

/* We need 3 bit arrays for the select() system call, one for each possible
   I/O state for each socket. */
typedef enum select_action_t {
	ACTION_WRITING,
	ACTION_READING,
	ACTION_FAULTING,
	ACTION_COUNT
} ACTION;


static FB_SERVICE *reapq = NULL;
static FB_SOCKET_DATA **sockets;
static int maxsockets = 0;
static fd_set select_state [3];	/* What we'll use on the next select() */
static fd_set last_state [3];	/* What came back on the last select() */


void fb_schedule_reap (FB_SERVICE *service) {
	/* It's called reapq, but they're stacked. Order doesn't matter. */
	service->next = reapq;
	reapq = service;
}


/* Add a socket to the registry.
   Return 0 on failure, 1 on success. */
bool fb_register (int socket_fd, FB_SOCKETTYPE type, void *thing) {
	/* Initialize the select state if this is the first registration */
	if (maxsockets == 0) {
		memset (&select_state, 0, sizeof (select_state));
	}
	
	/* We're limited by select(2), unless we want to do unsupported things */
	if (socket_fd >= FD_SETSIZE) {
		return false;
	}
	
	/* Make sure we've got enough space in the socket registry */
	if (maxsockets <= socket_fd) {
		/* maxsockets affects the select(), so used a balanced
		 approach to growing the collection */
		int newsize = maxsockets + (maxsockets / 4) + 10;
		if (newsize > FD_SETSIZE) {
			newsize = FD_SETSIZE;
		}
		FB_SOCKET_DATA **newsockets = realloc (sockets, newsize * sizeof (FB_SOCKET_DATA *));
		if (newsockets == NULL) {
			perror ("fb_register: realloc");
			return false;
		}
		/* Zero out the new portion of the collection */
		memset (newsockets + maxsockets, 0, (newsize - maxsockets) * sizeof (FB_SOCKET_DATA *));
		sockets = newsockets;
		maxsockets = newsize;
	}
	
	/* Insert the socket into the registry. */
	assert (sockets [socket_fd] == NULL);		/* There shouldn't be anything there yet. */
	if ((sockets [socket_fd] = malloc (sizeof (FB_SOCKET_DATA)))) {
		sockets [socket_fd]->thingie.user = thing;
		sockets [socket_fd]->socket = socket_fd;
		sockets [socket_fd]->type = type;
		FD_SET (socket_fd, &select_state[ACTION_READING]); /* Enable input */
		return true;
	}
	perror ("fb_register:malloc");
	return false;
}

/* Remove a socket from the registry.
 Always succeeds, unless there's a bug. */
void fb_unregister (int socket_fd) {
	assert (socket_fd > 0 && socket_fd < maxsockets);	/* We shouldn't be freeing sockets that don't fit in the registry. */
	assert (sockets [socket_fd]);	/* The socket to be freed should exist in the registry. */
	
	/* Turn off listening to it and cancel any pending processing */
	int i;
	for (i = 0; i < ACTION_COUNT; i++) {
		FD_CLR(socket_fd, &select_state[i]);
		FD_CLR(socket_fd, &last_state[i]);
	}
	
	free (sockets [socket_fd]);
	sockets [socket_fd] = NULL;
}


/* Enable/disable writing on a socket.  This is used within Football, typically
   when fresh data is queued for a socket. */
void fb_set_writable (int socket_fd, bool enable) {
	assert (socket_fd > 0 && socket_fd < maxsockets);
	assert (sockets [socket_fd] != NULL);
	
	if (sockets [socket_fd]) {
		if (enable) {
			FD_SET(socket_fd, &select_state[ACTION_WRITING]);
		} else {
			FD_CLR(socket_fd, &select_state[ACTION_WRITING]);
		}
	}
}

/* Enable/disable reading on a socket.  This is used within Football, typically
 when fresh data is queued for a socket. */
void fb_set_readable (int socket_fd, bool enable) {
	assert (socket_fd > 0 && socket_fd < maxsockets);
	assert (sockets [socket_fd] != NULL);

	if (sockets [socket_fd]) {
		if (enable) {
			FD_SET(socket_fd, &select_state[ACTION_READING]);
		} else {
			FD_CLR(socket_fd, &select_state[ACTION_READING]);
		}
	}
}

/* Public interface to turn reads on/off on a connection */
void fb_accept_input (FB_CONNECTION *connection, bool input) {
	assert (connection);
	assert (connection->state == FB_SOCKET_STATE_OPEN);
	
	fb_set_readable (connection->socket, input && connection->state == FB_SOCKET_STATE_OPEN);
}

/* Called by the poll routine, this function takes a socket and an action
   and processes it, either inline or by calling another function to do
   the work.  If appropriate, it returns an event; if things are handled
   completely internally, NULL is returned. */
FB_EVENT *fb_process_event (int socket_fd, ACTION action) {
	static FB_EVENT event;
	assert (socket_fd >= 0 && socket_fd < maxsockets);
	assert (action >= 0 && action < ACTION_COUNT);
	assert (sockets [socket_fd]);

	/* Look up the type of the event originator */
	FB_SOCKET_DATA *socket_data = sockets [socket_fd];
	if (socket_data == NULL) {
		return NULL;
	}

	/* Release any even resources, then clear it out */
	free (event.command);
	fb_destroy_argv (event.argv);
	memset (&event, 0, sizeof (event));

	/* Initialize the event in case we need it */
	event.magic = FB_SOCKTYPE_EVENT;
	event.socket = socket_fd;

	switch (socket_data->type) {
		case FB_SOCKTYPE_SERVICE:
			switch (action) {
				case ACTION_READING:
					return fb_new_connect (&event, socket_data->thingie.service);
				default:
					assert (0);
					break;
			}
			break;
		case FB_SOCKTYPE_CONNECTION:
			event.context = socket_data->thingie.connection->context;
			event.connection = socket_data->thingie.connection;
			event.service = event.connection->service;
			switch (action) {
				case ACTION_READING:
					return fb_read_input (&event, socket_data->thingie.connection);
				case ACTION_WRITING:
					return fb_send_output (&event, socket_data->thingie.connection);
				default:
					assert (0);
					break;
			}
			break;
		case FB_SOCKTYPE_USER:
			event.context = socket_data->thingie.user;
			switch (action) {
				case ACTION_READING:
					event.type = FB_EVENT_READABLE;
					return &event;
				case ACTION_WRITING:
					event.type = FB_EVENT_WRITABLE;
					return &event;
				case ACTION_FAULTING:
					event.type = FB_EVENT_FAULTING;
					return &event;
				default:
					assert (0);
					break;
			}
			break;
		default:
			fprintf (stderr, "fb_dispatch_event: Invalid socket type %d in switch\n", socket_data->type);
			assert (0);
			return NULL;
	}
	fprintf (stderr, "fb_dispatch_event: Event type not handled: type=%d, action=%d", socket_data->type, action);
	return NULL;
}



/* Poll the stuff in the registry.
 This is the worker function; the four exposed functions with various timeout
 mechanisms follow.  Returns an event structure, or NULL on failure. */
static FB_EVENT *fb_poll_for (struct timeval *timeout) {
pollagain:;
	static FB_EVENT event; /* Reusable timeout event */
	static int events_remaining = 0;
	static ACTION process_action = 0;
	static int process_fd = 0;
	
	/* Something should have been registered by now. */
	assert (maxsockets > 0);
	if (maxsockets == 0) {
		fprintf (stderr, "fb_poll_for: Called before registering anything.\n");
		return NULL;
	}
	/* See if there's any reaping queued; if so, do it. */
	if (reapq) {
		memset (&event, 0, sizeof (event));
		event.type = FB_EVENT_STOPPED;
		event.service = reapq;
		reapq = event.service->next;
		fb_destroy_service (event.service);
		fb_free_freelists ();
		return &event;
	}
	
	
	/* See if it's time to refill the coffers, so-to-speak */
	if (events_remaining == 0) {
		/* Reset the event searching loops */
		process_action = 0;
		process_fd = 0;
		
		/* Create a fresh copy of the selector masks */
		int i;
		for (i = 0; i < ACTION_COUNT; i++) {
			FD_COPY(&select_state[i], &last_state [i]);
		}
		/* Select, repeating if we get an interrupted system call */
		do {
			events_remaining = select (maxsockets, &last_state [ACTION_READING],
									   &last_state [ACTION_WRITING], &last_state [ACTION_FAULTING], timeout);
		} while (events_remaining < 0 && errno == EINTR);
		/* Check for/handle (don't handle) errors */
		if (events_remaining < 0) {
			events_remaining = 0;
			perror ("fb_poll_for:select");
			return NULL;
		}
		/* Check for/handle timeout */
		if (events_remaining == 0) {
			/* We'll just keep reusing this, just clearing it out each time. */
			memset (&event, 0, sizeof (event));
			event.magic = FB_SOCKTYPE_EVENT;
			event.type = FB_EVENT_TIMEOUT;
			return (&event);
		}
	}
	/* Search the mask returned by select to find what needs attention */
	while (process_action < ACTION_COUNT) {
		while (process_fd < maxsockets) {
			if (FD_ISSET (process_fd, &last_state [process_action])) {
				/* This needs attention.  Count and process it. */
				events_remaining--;
				FB_EVENT *fd_event = fb_process_event (process_fd, process_action);
				if (fd_event) {
					/* An event resulted from the processing. Bump to the */
					/* next socket so we don't repeat, then return the event. */
					if (++process_fd >= maxsockets) {
						process_fd = 0;
						process_action++;
					}
					return (fd_event);
				}
#ifndef NDEBUG
				/* If there's no more events, stop looking.  But in test mode,
				 continue to make sure we counts confirm good behavior. */
				if (events_remaining <= 0) goto pollagain;
#endif
			}
			process_fd++;
		}
		process_fd = 0;
		process_action++;
	}
	if (events_remaining != 0) {
		fprintf (stderr, "fb_poll_for: %d event not found\n", events_remaining);
	}
	events_remaining = 0;
	goto pollagain; /* Tail recursion would probably collapse and work too */
}

/* Quick poll with no timeout */
FB_EVENT *fb_poll () {
	struct timeval zero;
	zero.tv_sec = 0;
	zero.tv_usec = 0;
	return fb_poll_for (&zero);
}
/* Poll with a duration */
FB_EVENT *fb_poll_with_timeout (double timeout) {
	struct timeval _timeout;
	_timeout.tv_sec = timeout;
	_timeout.tv_usec = (unsigned int) ((long) (timeout * 1000000) % 1000000);
	return fb_poll_for (&_timeout);
}
/* Poll indefinitely */
FB_EVENT *fb_wait () {
	return fb_poll_for (NULL);
}
/* Poll with a timeout at a specific time */
FB_EVENT *fb_poll_until (time_t untilwhen) {
	struct timeval timeout;
	time_t now = time(NULL);
	if (now == -1) { /* According to the man page time() can actually fail */
		perror ("time");
		timeout.tv_sec = 1; /* *shrug*, there's really bigger problems if you don't know what time it is */
	} else {
		long timeLeft = untilwhen - now;
		timeout.tv_sec = (timeLeft < 0) ? 0 : timeLeft;
	}
	timeout.tv_usec = 0;
	return fb_poll_for (&timeout);
}


