/*
 *  fb_service.c
 *  Player
 *
 *  Created by Perette Barella on 2012-03-03.
 *  Copyright 2012 Devious Fish. All rights reserved.
 *
 */

#include <config.h>

#if !defined(__FreeBSD__) && !defined(__APPLE__)
#define _POSIX_C_SOURCE 1 /* fileno,fdopen() */
#endif
#ifndef __FreeBSD__
#define _BSD_SOURCE /* strdup */
#endif

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <assert.h>

#include "fb_service.h"








/* Initialize and set up the IP V4 socket.
 Return 0 on failure, 1 on success. */
static int fb_setup_ip4_socket (FB_SERVICE *service) {
	if ((service->ip4socket = socket (PF_INET, SOCK_STREAM, 0)) >= 0) {
		int on = 1;
		/* We got a socket, bind and listen on it. */
		service->ip4addr.sin_family = PF_INET;
		service->ip4addr.sin_addr.s_addr = INADDR_ANY;
		service->ip4addr.sin_port = htons (service->port_number);
		if (setsockopt(service->ip4socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0)
		{
			perror("fb_setup_ip6_socket: setsockopt(SO_REUSEADDR)");
			/* That's annoying but not critical, so keep going. */
		}
		if (bind (service->ip4socket,
				  (struct sockaddr*) &(service->ip4addr), sizeof (service->ip4addr)) >= 0) {
			if (listen (service->ip4socket, service->queue_size) >= 0) {
				if (fb_register (service->ip4socket, FB_SOCKTYPE_SERVICE, service)) {
					return 1;
				}
				/* Errors announced by fb_register */
			}
		} else {
			perror ("fb_setup_ip4_socket: bind");
		}
		close (service->ip4socket);
		service->ip4socket = 0;
	} else {
		perror ("fb_setup_ip4_socket: socket");
	};
	return 0;
}

/* Initialize and set up the IP V6 socket.
 Return 0 on failure, 1 on success.
 Define HAVE_IPV6 to enable IP6 support. */
static int fb_setup_ip6_socket (FB_SERVICE *service) {
#ifdef HAVE_IPV6
	if ((service->ip6socket = socket (PF_INET6, SOCK_STREAM, 0)) >= 0) {
		int on = 1;
		/* We got a socket, bind and listen on it. */
#ifdef SIN6_LEN
		service->ip6addr.sin6_len = sizeof (service->ip6addr);
#endif
		service->ip6addr.sin6_family = PF_INET6;
		service->ip6addr.sin6_addr = in6addr_any;
		service->ip6addr.sin6_port = htons (service->port_number);
		if (setsockopt(service->ip6socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0)
		{
			perror("fb_setup_ip6_socket: setsockopt(SO_REUSEADDR)");
			/* That's annoying but not critical, so keep going. */
		}
		if (bind (service->ip6socket,
				  (struct sockaddr *) &(service->ip6addr), sizeof (service->ip6addr)) >= 0) {
			if (listen (service->ip6socket, service->queue_size) >= 0) {
				if (fb_register (service->ip6socket, FB_SOCKTYPE_SERVICE, service)) {
					return 1;
				}
				/* Errors announced by fb_register */
			}
		} else {
			perror ("fb_setup_ip6_socket: bind");
		}
		close (service->ip6socket);
		service->ip6socket = 0;
	} else {
		perror ("fb_setup_ip6_socket: socket");
	};
#endif
	return 0;
}

/* Create a new service and initialize its listeners.
 Return a pointer to the service, or NULL on failure. */
FB_SERVICE *fb_create_service (in_port_t port, int nqueue, size_t contextsize) {
	/* Allocate and initialize memory for the service */
	FB_SERVICE *service;
	if (!(service = malloc (sizeof (*service)))) {
		perror ("fb_create_service: malloc");
		return NULL;
	}
	memset (service, 0, sizeof (*service));
	service->type = FB_SOCKTYPE_SERVICE;
	service->port_number = port;
	service->queue_size = nqueue;
	service->contextsize = contextsize;
	
	/* Initialize and set up the sockets */
	int successes = fb_setup_ip4_socket (service) +
	fb_setup_ip6_socket (service);
	
	if (successes == 0) {
		free (service);
		service = NULL;
	}
	return service;
}


/* Destroy a service's resources */
void fb_destroy_service (FB_SERVICE *service) {
	/* Close all the connections first */
	assert (service);
	assert (service->connection_count == 0);
	while (service->connection_count > 0) {
		fb_destroy_connection (service->connections [0]);
	}
	/* Close our listeners */
	if (service->ip4socket) {
		fb_unregister (service->ip4socket);
		close (service->ip4socket);
	}
	if (service->ip6socket) {
		fb_unregister (service->ip6socket);
		close (service->ip6socket);
	}
	free (service->connections); /* Pre Man page, freeing null is okay */
	free (service);
}


/* Initiate closure of a service.  Returns nothing. */
void fb_close_service (FB_SERVICE *service) {
	unsigned int i;
	assert (service);
	assert (service->state == FB_SOCKET_STATE_OPEN);
	service->state = FB_SOCKET_STATE_CLOSING;
	for (i = 0; i < service->connection_count; i++) {
		if (service->connections[i]->state == FB_SOCKET_STATE_OPEN) {
			fb_close_connection (service->connections[i]);
		}
	}
	/* Stop listening so we don't accept new connections */	
	if (service->ip4socket) {
		fb_set_readable (service->ip4socket, false);
	}
	if (service->ip6socket) {
		fb_set_readable (service->ip4socket, false);
	}
	if (service->connection_count == 0) {
		/* If there's no connections left, schedule for reaping. */
		fb_schedule_reap (service);
	}
}



/* Make space in the service's connection array,
   allocate a connection and (if requested) a user context. */
static FB_CONNECTION *fb_new_connection (FB_SERVICE *service) {
	/* Expand the connection list for this service if needed. */
	assert (service);
	assert (service->connection_count <= service->connections_size); /* If this is off, something already went wrong */
	if (service->connection_count >= service->connections_size) {
		size_t newsize = service->connections_size * 3 + 15;
		FB_CONNECTION **newconns = realloc (service->connections, newsize * sizeof (FB_CONNECTION *));
		if (newconns == NULL) {
			perror ("fb_new_connection: realloc");
			return NULL;
		}
		service->connections_size = newsize;
		service->connections = newconns;
	}
	/* Allocate the connection, initialize it, and add a context if necessary */
	FB_CONNECTION *connection = malloc (sizeof (*connection));
	if (connection) {
		memset (connection, 0, sizeof (*connection));
		connection->type = FB_SOCKTYPE_CONNECTION;
		connection->service = service;
		/* If a context isn't needed, we're good */
		if (service->contextsize == 0) {
			return (connection);
		}
		connection->context = malloc (service->contextsize);
		if (connection->context) {
			memset (connection->context, 0, service->contextsize);
			return (connection);
		}
		free (connection);
	}
	perror ("fb_new_connection: malloc");
	return NULL;
}	

/* Accept a connection.
   Return a pointer to the connection, or NULL on failure. */
FB_CONNECTION *fb_accept_connection (FB_SERVICE *service, int domain) {
	assert (service);
	FB_CONNECTION *connection = fb_new_connection (service);
	if (!connection) {
		/* If there's not enough memory to store the connection, just reject it */
		/* TODO: REJECT CONNECTION */
		return NULL;
	}
	/* Allocate and initialize memory for the service */
	connection->domain = domain;
	socklen_t addr_size = (socklen_t) sizeof (connection->origin);
	
	/* Accept the connection */
	if ((connection->socket = accept (domain == PF_INET ? service->ip4socket : service->ip6socket,
									  (struct sockaddr *) &(connection->origin), &addr_size)) >= 0) {
		if ((connection->file = fdopen (connection->socket, "r"))) {
			/* Add to the connection list */
			service->connections [service->connection_count++] = connection;
			/* put the socket in non-blocking mode */
			if (fcntl (connection->socket, F_SETFL, fcntl (connection->socket, F_GETFL) | O_NONBLOCK) == -1) {
				perror ("fb_accept_connection:fcntl");
			}
			/* select() and stdio buffering don't play nice... shut off buffering */
			setbuf (connection->file, NULL);
			/* Disable the dreaded killer sigpipe */
#ifdef HAVE_SO_NOSIGPIPE
			int option_value = 1; /* We're setting NOSIGPIPE to ON */
			if (setsockopt (connection->socket, SOL_SOCKET, SO_NOSIGPIPE, &option_value, sizeof (option_value)) < 0) {
				perror ("fb_accept_connection:setsockopt");
			}
#elif defined(HAVE_MSG_NOSIGNAL)
			if (send(connection->socket, NULL, 0, MSG_NOSIGNAL) < 0) {
				perror ("fb_accept_connection:send(,,,MSG_NOSIGNAL)");
			}
#endif
			return (connection);
		} else {
			perror ("fb_accept_connection:fdopen");
		}
		close (connection->socket);
	} else {
		perror ("fb_accept_connection: accept");
	}
	free (connection);
	return NULL;
}



/* Close and destroy a connection.  Returns nothing. */
void fb_destroy_connection (FB_CONNECTION *connection) {
	assert (connection);
	
	FB_SERVICE *service = connection->service;
	/* Take the connection out of the service's list */
	unsigned int i;
	for (i = 0; i < service->connection_count && service->connections [i] != connection; i++)
		/* nothing */;
	assert (i < service->connection_count); /* If we didn't find it, there's a bug */
	service->connection_count--;
	unsigned int j;
	for (j = i; j < service->connection_count; j++) {
		service->connections [j] = service->connections [j+1];
	}
	
	/* Close the socket and free resources */
	fclose (connection->file);
	free (connection->filename);
	free (connection->context);
	free (connection);
}



/* Initiate connection closure.  If the socket is in an open state, change
 its state and enable writes.  The next write attempt will generate
 a close event.
 If the connection is not in an open state, it is already on its way out;
 so leave it alone */
void fb_close_connection (FB_CONNECTION *connection) {
	assert (connection);
	assert (connection->state == FB_SOCKET_STATE_OPEN);
	
	if (connection->state == FB_SOCKET_STATE_OPEN) {
		connection->state = FB_SOCKET_STATE_FLUSHING;
		fb_set_writable (connection->socket, true);
		/* Ignore further input: if service is closing, this avoids a connection
		   requesting individual closure and causing a duplicate close. */
		fb_set_readable (connection->socket, false);
	}
}



/* Accept a connection.
 Return a pointer to the connection, or NULL on failure. */
FB_EVENT *fb_accept_file (FB_SERVICE *service, char *filename) {
	assert (service);
	assert (filename && *filename);
	static FB_EVENT event;
	/* Allocate and initialize memory for the service */
	memset (&event, 0, sizeof (event));
	event.magic = FB_SOCKTYPE_EVENT;
	event.type = FB_EVENT_CONNECT;
	event.service = service;
	/* Allocate a new connection and fill it in; create an event for the new connection. */
	event.connection = fb_new_connection (service);
	if (event.connection) {
		if ((event.connection->filename = strdup (filename))) {
			/* All the allocations were successful, so open the file. */
			if ((event.connection->file = fopen (filename, "r"))) {
				/* select() and stdio buffering don't play nice... shut off buffering */
				setbuf (event.connection->file, NULL);
				event.context = event.connection->context;
				event.connection->socket = fileno (event.connection->file);
				event.socket = event.connection->socket;
				if (fb_register (event.connection->socket, FB_SOCKTYPE_CONNECTION, event.connection)) {
					/* Add to the connection list */
					service->connections [service->connection_count++] = event.connection;
					return &event;
				}
				fclose (event.connection->file);
			} else {
				perror (filename);
			}
			free (event.connection->filename);
		} else {
			perror ("fb_accept_file:strdup");
		}
		free (event.connection);
	}
	return NULL;
}


/* Create and initialize a new iterator for a service.
   Returns an iterator or NULL on error. */
FB_ITERATOR *fb_new_iterator (FB_SERVICE *service) {
	assert (service);
	FB_ITERATOR *it = calloc (sizeof (FB_ITERATOR), 1);
	if (it) {
		it->service = service;
		it->iteration = service->connection_count;
	} else {
		perror ("calloc");
	}
	return (it);
}


/* Get the next iteration.  Returns an event, or NULL if there
   are no more connections. */
FB_EVENT *fb_iterate_next (FB_ITERATOR *it) {
	static FB_EVENT event;
	assert (it);
	assert (it->service);
	assert (it->iteration >= 0);
	if (it->iteration > 0) {
		it->iteration--;
		
		memset (&event, 0, sizeof (event));
		event.magic = FB_SOCKTYPE_EVENT;
		event.service = it->service;
		event.connection = it->service->connections [it->iteration];
		event.type = event.connection->state == FB_SOCKET_STATE_OPEN ? FB_EVENT_ITERATOR : FB_EVENT_ITERATOR_CLOSE;
		event.socket = event.connection->socket;
		event.context = event.connection->context;		
		return (&event);
	}
	return NULL;
}


void fb_destroy_iterator (FB_ITERATOR *it) {
	assert (it);
	free (it);
}
