/*
 *  fb_handle_event.c
 *  Player
 *
 *  Created by Perette Barella on 2012-03-03.
 *  Copyright 2012 Devious Fish. All rights reserved.
 *
 */

#include <config.h>

#ifndef __FreeBSD__
#define _BSD_SOURCE /* snprintf() */
#endif
#if !defined(__FreeBSD__) && !defined(__APPLE__)
#define _POSIX_C_SOURCE 1 /* required by getaddrinfo() */
#endif



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <stdbool.h>

#include <assert.h>

#ifndef HAVE_FGETLN
#include "fgetln.h"
#endif

#include "fb_service.h"

typedef struct fb_message_t {
	int usecount;
	ssize_t length;
	char *message;
} FB_MESSAGE;

typedef struct fb_outputq_t {
	struct fb_outputq_t *next;
	FB_MESSAGE *message;
} FB_OUTPUTQ;

static FB_MESSAGE *freemessages = NULL;
static FB_OUTPUTQ *freeq = NULL;


/* Free the free message lists */
void fb_free_freelists () {
	FB_OUTPUTQ *free_q;
	FB_MESSAGE *free_messages;
	while ((free_q = freeq)) {
		freeq = freeq->next;
		free (free_q);
	}
	while ((free_messages = freemessages)) {
		freemessages = (FB_MESSAGE *) freemessages->message;
		free (free_messages);
	}
}

/* Allocate a message block structure.
   Get it off the free list if its available, otherwise malloc a new one.
   Returns the message block, or NULL on failure. */
static FB_MESSAGE *fb_messagealloc() {
	FB_MESSAGE *result = freemessages;
	if (result) {
		freemessages = (FB_MESSAGE *) (result->message);
	} else {
		result = malloc (sizeof (*result));
	}
	if (result) {
		memset (result, 0, sizeof (*result));
		result->usecount = 1;
	} else {
		perror ("fb_messagealloc:malloc");
	}
	return result;
}

/* "Free" a message block by putting it on the free list. */
static void fb_messagefree(FB_MESSAGE *freethis) {
	assert (freethis->usecount > 0);
	if (--freethis->usecount <= 0) {
		if (freethis->message) {
			free (freethis->message);
		}
		freethis->message = (char *) freemessages;
		freemessages = freethis;
	}
}

/* Allocate a message queue structure.
   Allocate it off the free list if available, otherwise malloc a new one.
   Returns the message block, or NULL on failure.
*/
static FB_OUTPUTQ *fb_qalloc() {
	FB_OUTPUTQ *result = freeq;
	if (result) {
		freeq = result->next;
	} else {
		result = malloc (sizeof (*result));
	}
	if (result) {
		memset (result, 0, sizeof (*result));
	} else {
		perror ("fb_qalloc:malloc");
	}
	return result;
}

/* "Free" a message q by putting it on the free list. */
static void fb_qfree(FB_OUTPUTQ *freethis) {
	if (freethis->message) {
		fb_messagefree (freethis->message);
	}
	freethis->next = freeq;
	freeq = freethis;
}


/* Create and populate a new message block. */
static FB_MESSAGE *fb_create_message (const char *format, va_list parameters) {
	FB_MESSAGE *message;
	if ((message = fb_messagealloc ())) {
		message->length = vasprintf(&message->message, format, parameters);
		if (message->length >= 0) {
			return message;
		}
		fb_messagefree(message);
	}
	return NULL;
}


/* Put a message into a connection's queue.
   Return length on success, -1 on failure. */
static ssize_t fb_queue_single (FB_CONNECTION *connection, FB_MESSAGE *message) {
	ssize_t length = message->length;
	if (length == 0 || connection->state == FB_SOCKET_STATE_CLOSING || connection->filename) {
		/* Don't queue empty messages or queue stuff for an input file, but feign success. */
		fb_messagefree (message);
		return length;
	}
	ssize_t sent = 0;
	if (connection->outfirst == NULL) {
		/* This connection has nothing queued.  Try to send now, bypassing queueing. */
		assert (!connection->outlast);
		sent = write (connection->socket, message->message, length);
		if (sent >= message->length) {
			/* The entire buffer was sent.  Free it and return success. */
			fb_messagefree (message);
			return length;
		}
	}
	FB_OUTPUTQ *q = fb_qalloc();
	if (q) {
		q->message = message;
		/* Insert the message at the end of the queue */
		if (connection->outfirst) {
			assert (connection->outlast);
			connection->outlast->next = q;
			connection->outlast = q;
		} else {
			/* Create the queue and enable writes */
			assert (!connection->outlast);
			connection->outfirst = q;
			connection->outlast = q;
			connection->outpoint = (sent > 0) ? sent : 0;
			fb_set_writable (connection->socket, true);
		}
		return length;
	}
	fb_messagefree (message);
	return -1;
}

/* Put a message into all a service's queues.
 Return length on success (queued to all connections successfully).
 On failure or partial failure, return -1. */
static ssize_t fb_queue_broadcast (FB_SERVICE *service, FB_MESSAGE *message) {
	size_t length = message->length;
	if (message->length == 0) {
		/* Don't queue empty messages, but feign success. */
		fb_messagefree (message);
		return 0;
	}
	unsigned int i;
	for (i = 0; i < service->connection_count; i++) {
		if (service->connections[i]->state != FB_SOCKET_STATE_CLOSING && !service->connections[i]->filename) {
			message->usecount++;
			if (fb_queue_single (service->connections[i], message) == -1) {
				length = -1;
			}
		}
	}
	/* When allocated, use count is already 1.  Correct for this now. */
	fb_messagefree (message);
	return length;
}

/* Add a message message to a service, connection, or event,
   because polymorphism is friendly.  Perhaps I should be using an object language?
   Looks at the type magic number inside the thing to determine which kind it is.
   Return the length of the message, or -1 on error, as per stdio.
   In case of only partial success... Success. */
ssize_t fb_queue_message (void *thing, FB_MESSAGE *message, bool broadcast) {
	assert (thing);
	assert (message);
	FB_SOCKETTYPE type = *(FB_SOCKETTYPE *)thing;
	if (message) {
		switch (type) {
			case FB_SOCKTYPE_SERVICE:
				return fb_queue_broadcast (((FB_SERVICE *)thing), message);
			case FB_SOCKTYPE_CONNECTION:
				if (broadcast) {
					return fb_queue_broadcast (((FB_CONNECTION *)thing)->service, message);
				}
				return fb_queue_single ((FB_CONNECTION *)thing, message);
			case FB_SOCKTYPE_EVENT:
				if (broadcast) {
					return fb_queue_broadcast (((FB_EVENT *)thing)->connection->service, message);
				}
				return fb_queue_single (((FB_EVENT *)thing)->connection, message);
			default:
				/* Nothing */
				break;
		}
		assert (0);
		fprintf (stderr, "fb_queue_message: thing of unknown type %d passed\n", (int) type);
		fb_messagefree (message);
	}
	return -1;
}
	
/* Add messages to output queues in various forms and styles.
   All forms take an opaque type and determine if it's a connection
   or service, so you can write code once and deal with both
   single and broadcast messages.
   'b' variants broadcast, given either a connection or service.
	   Services always broadcast, regardless of service.
   'v' accepts a pointer to the format arguments. */
ssize_t fb_fprintf (void *thing, const char *format, ...) {
	va_list parameters;
	va_start(parameters, format);
	FB_MESSAGE *message = fb_create_message (format, parameters);
	va_end (parameters);
	return fb_queue_message (thing, message, false);
}

ssize_t fb_vfprintf (void *thing, const char *format, va_list parameters) {
	FB_MESSAGE *message = fb_create_message (format, parameters);
	return fb_queue_message (thing, message, false);
}
	
ssize_t fb_bfprintf (void *thing, const char *format, ...) {
	va_list parameters;
	va_start(parameters, format);
	FB_MESSAGE *message = fb_create_message (format, parameters);
	va_end (parameters);
	return fb_queue_message (thing, message, true);
}

ssize_t fb_bvfprintf (void *thing, const char *format, va_list parameters) {
	FB_MESSAGE *message = fb_create_message (format, parameters);
	return fb_queue_message (thing, message, true);
}



/* Write output to a connection.  No events generated.
   If the socket is in a closing state, when the buffer is empty, close the connection. */
FB_EVENT *fb_send_output (FB_EVENT *event, FB_CONNECTION *connection) {
	assert (event);
	assert (connection);
	if (connection->outfirst == NULL) {
		/* Output queue is empty. */
		if (connection->state == FB_SOCKET_STATE_FLUSHING) {
			/* The output buffer is empty, send a close event before finalizing */
			connection->state = FB_SOCKET_STATE_CLOSING;
			event->type = FB_EVENT_CLOSE;
			return event;
		} else if (connection->state == FB_SOCKET_STATE_CLOSING) {
			/* If the service is closing and this is the last connection, schedule reap */
			if (connection->service->connection_count == 1 &&
				connection->service->state == FB_SOCKET_STATE_CLOSING) {
				fb_schedule_reap (connection->service);
			}
			/* Disassemble the connection */
			fb_unregister (connection->socket);
			fb_destroy_connection (connection);
		} else {
			/* The queue is empty, turn off write monitoring */
			fb_set_writable (connection->socket, false);
		}
	} else {
		FB_MESSAGE *message = connection->outfirst->message;
		size_t outlen = message->length - connection->outpoint;
		ssize_t written = write (connection->socket, message->message+connection->outpoint,outlen);
		if (written >= 0) {
			/* Skip past the portion transmitted */
			connection->outpoint += written;
			assert (connection->outpoint <= message->length);
			if (connection->outpoint >= message->length) {
				/* We finished with this message. Free it and move to the next. */
				connection->outpoint = 0;
				FB_OUTPUTQ *freethis = connection->outfirst;
				connection->outfirst = connection->outfirst->next;
				if (connection->outfirst == NULL) {
					connection->outlast = NULL;
				}
				fb_qfree (freethis);
			}
		} else {
			/* Socket has closed/connection is lost. */
			if (connection->state == FB_SOCKET_STATE_OPEN) {
				fb_close_connection (connection);
			}
			/* Free all message blocks attached to the output q. */
			FB_OUTPUTQ *q;
			for (q = connection->outfirst; q != NULL; q = q->next) {
				fb_messagefree (q->message);
			}
			/* Insert the whole list on the front of the free list */
			if (connection->outfirst) {
				connection->outlast->next = freeq;
				freeq = connection->outfirst;
			}
			connection->outfirst = NULL;
			connection->outlast = NULL;
		}
	}
	return NULL;
}

/* Read input from a connection, return it as an event.
   The event has the raw command in 'command'.
   The event has a broken up command in 'argv'.  On failure, 'argv' will be null.
   If the event is EOF, first time change the socket state and return an event.
   If it's another EOF, then close and destroy the connection.
 */
FB_EVENT *fb_read_input (FB_EVENT *event, FB_CONNECTION *connection) {
	assert (connection);
	assert (event);
	char *buffer = NULL;
	size_t buffilled = 0;
	
	/* If we got here, socket_data will be for a connection */
	event->type = FB_EVENT_INPUT;
	char *line = fgetln (connection->file, &buffilled);
	if (!line || buffilled == 0) {
		/* EOF/Connection closed from other end. Initiate closure. */
		if (connection->state == FB_SOCKET_STATE_OPEN) {
			fb_close_connection (connection);  /* Not _now variant! */
		}
		return NULL;
	}

	/* Make a copy of the line */
	buffer = malloc (buffilled + 1);
	if (!buffer) {
		perror ("fb_read_input:malloc");
		return NULL;
	}
	memcpy (buffer, line, buffilled);
	/* strip returns off the end of the unparsed line */
	while (buffilled > 0 && (buffer [buffilled - 1] == '\r' || buffer [buffilled - 1] == '\n')) {
		buffilled--;
	}
	buffer [buffilled] = '\0';
	
	/* Store the command and parse it into an argv array */
	event->command = buffer;
	event->argc = fb_create_argv(event->command, &event->argv);
	if (event->argc < 0) {
		event->argc = 0;
	}
	
	return (event);
}


/* Accept a new connection and return an event announcing it. */
FB_EVENT *fb_new_connect (FB_EVENT *event, FB_SERVICE *service) {
	assert (event);
	assert (service);
	
	event->connection = fb_accept_connection (service, event->socket == service->ip4socket ? PF_INET : PF_INET6);
	if (event->connection) {
		if (fb_register(event->connection->socket, FB_SOCKTYPE_CONNECTION, event->connection)) {
			event->socket = event->connection->socket;
			event->context = event->connection->context;
			event->service = service;
			event->type = FB_EVENT_CONNECT;
			return (event);
		}
		fb_destroy_connection(event->connection);
	}
	return NULL;
}



