/*
  +----------------------------------------------------------------------+
  | PHP Version 5                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Author: Paul Chandler <pestilence669@php.net>                        |
  +----------------------------------------------------------------------+
*/

/* $Id: xrange.c 326859 2012-07-28 23:20:09Z felipe $ */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "php.h"
#include "php_ini.h"
#include "ext/standard/info.h"
#include "zend_exceptions.h"
#include "zend_interfaces.h"

#include <math.h>
#include "ext/spl/spl_exceptions.h"
#include "ext/spl/spl_iterators.h"
#include "php_xrange.h"

/* {{{ xrange_functions[]
 */
zend_function_entry xrange_functions[] = {
	PHP_FE(xrange, NULL)
	{NULL, NULL, NULL}	/* Must be last */
};
/* }}} */

/* {{{ xrange_module_entry
 */
zend_module_entry xrange_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	PHP_XRANGE_EXTNAME,
	xrange_functions,
	PHP_MINIT(xrange),
	PHP_MSHUTDOWN(xrange),
	NULL,
	NULL,
	PHP_MINFO(xrange),
#if ZEND_MODULE_API_NO >= 20010901
	PHP_XRANGE_VERSION,
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

/* {{{ XRangeIterator seek method argument info */
ZEND_BEGIN_ARG_INFO(arginfo_xrange_xri_seek, 0) 
	ZEND_ARG_INFO(0, position)
ZEND_END_ARG_INFO();
/* }}} */

/* {{{ XRangeIterator methods */
zend_class_entry *php_xrange_xri_entry;
static zend_function_entry php_xrange_xri_functions[] = {
	PHP_ME(
		PHP_XRANGE_XRI_NAME, __construct,
		NULL, ZEND_ACC_PUBLIC | ZEND_ACC_CTOR
	)
	/* Iterator interface methods */
	PHP_ME(PHP_XRANGE_XRI_NAME, current,    NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, key,        NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, next,       NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, rewind,     NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, valid,      NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, seek,       arginfo_xrange_xri_seek, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, count,      NULL, ZEND_ACC_PUBLIC)
	PHP_ME(PHP_XRANGE_XRI_NAME, __toString, NULL, ZEND_ACC_PUBLIC)
	{NULL, NULL, NULL} /* Must be last */
};
/* }}} */

/* {{{ OddFilterIterator methods */
zend_class_entry *php_xrange_OddFilterIterator_entry;
static zend_function_entry php_xrange_OddFilterIterator_functions[] = {
	PHP_ME(PHP_XRANGE_ODDFILTERITERATOR_NAME, accept, NULL, ZEND_ACC_PUBLIC)
	{NULL, NULL, NULL} /* Must be last */
};
/* }}} */

/* {{{ EvenFilterIterator methods */
zend_class_entry *php_xrange_EvenFilterIterator_entry;
static zend_function_entry php_xrange_EvenFilterIterator_functions[] = {
	PHP_ME(PHP_XRANGE_EVENFILTERITERATOR_NAME, accept, NULL, ZEND_ACC_PUBLIC)
	{NULL, NULL, NULL} /* Must be last */
};
/* }}} */

/* {{{ NumericFilterIterator methods */
zend_class_entry *php_xrange_NumericFilterIterator_entry;
static zend_function_entry php_xrange_NumericFilterIterator_functions[] = {
	PHP_ME(PHP_XRANGE_NUMERICFILTERITERATOR_NAME, accept, NULL, ZEND_ACC_PUBLIC)
	{NULL, NULL, NULL} /* Must be last */
};
/* }}} */


#ifdef COMPILE_DL_XRANGE
ZEND_GET_MODULE(xrange)
#endif

/* {{{ xrange_xri_object_free_storage */
static void xrange_xri_object_free_storage(void *object TSRMLS_DC)
{
	xrange_module_storage *internalStorage = (xrange_module_storage *) object;
	zend_object_std_dtor(&internalStorage->obj TSRMLS_CC);
	efree(object);
}
/* }}} */

/* {{{ xrange_xri_object_new */
zend_object_value xrange_xri_object_new(zend_class_entry *class_type TSRMLS_DC)
{
	zend_object_value retval;
#if PHP_VERSION_ID < 50399
	zval *tmp;
#endif

	xrange_module_storage *internalStorage =
		ecalloc(1, sizeof(xrange_module_storage));
	zend_object_std_init(&internalStorage->obj, class_type TSRMLS_CC);
#if PHP_VERSION_ID < 50399
	zend_hash_copy(
		internalStorage->obj.properties,
		&php_xrange_xri_entry->default_properties,
		(copy_ctor_func_t) zval_add_ref,
		(void *) &tmp, sizeof(zval *)
	);
#else
	object_properties_init(&internalStorage->obj, php_xrange_xri_entry);
#endif

	retval.handle = zend_objects_store_put(
		internalStorage,
		(zend_objects_store_dtor_t) zend_objects_destroy_object,
		(zend_objects_free_object_storage_t) xrange_xri_object_free_storage,
		NULL TSRMLS_CC
	);
	retval.handlers = zend_get_std_object_handlers();

	return retval;
}
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(xrange)
{
	zend_class_entry ce;

	/* Register Class: XRangeIterator */
	INIT_CLASS_ENTRY(ce, PHP_XRANGE_XRI_NAME, php_xrange_xri_functions);
	ce.create_object = xrange_xri_object_new;
    ce.name_length = strlen(PHP_XRANGE_XRI_NAME); 
	php_xrange_xri_entry = zend_register_internal_class(&ce TSRMLS_CC);

	zend_class_implements(php_xrange_xri_entry TSRMLS_CC, 1, spl_ce_SeekableIterator);
	zend_class_implements(php_xrange_xri_entry TSRMLS_CC, 1, spl_ce_Countable);

	/* Register Class: OddFilterIterator */
	memset(&ce, sizeof(ce), '\0');
	INIT_CLASS_ENTRY(ce, PHP_XRANGE_ODDFILTERITERATOR_NAME, php_xrange_OddFilterIterator_functions);
    ce.name_length = strlen(PHP_XRANGE_ODDFILTERITERATOR_NAME); 
	php_xrange_OddFilterIterator_entry =
		zend_register_internal_class_ex(&ce, spl_ce_FilterIterator, NULL TSRMLS_CC);

	/* Register Class: EvenFilterIterator */
	memset(&ce, sizeof(ce), '\0');
	INIT_CLASS_ENTRY(ce, PHP_XRANGE_EVENFILTERITERATOR_NAME, php_xrange_EvenFilterIterator_functions);
    ce.name_length = strlen(PHP_XRANGE_EVENFILTERITERATOR_NAME); 
	php_xrange_EvenFilterIterator_entry =
		zend_register_internal_class_ex(&ce, spl_ce_FilterIterator, NULL TSRMLS_CC);

	/* Register Class: NumericFilterIterator */
	memset(&ce, sizeof(ce), '\0');
	INIT_CLASS_ENTRY(ce, PHP_XRANGE_NUMERICFILTERITERATOR_NAME, php_xrange_NumericFilterIterator_functions);
    ce.name_length = strlen(PHP_XRANGE_NUMERICFILTERITERATOR_NAME); 
	php_xrange_NumericFilterIterator_entry =
		zend_register_internal_class_ex(&ce, spl_ce_FilterIterator, NULL TSRMLS_CC);

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
PHP_MSHUTDOWN_FUNCTION(xrange)
{
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(xrange)
{
	php_info_print_table_start();
	php_info_print_table_header(2, "xrange support", "enabled");
	php_info_print_table_row(2, "version", PHP_XRANGE_VERSION);
	php_info_print_table_row(2, "Classes", "EvenFilterIterator, NumericFilterIterator, OddFilterIterator, XRangeIterator");
	php_info_print_table_end();
}
/* }}} */

/* {{{ proto XRangeIterator xrange(mixed $low, mixed $high [, number $step])
   Return a configured range iterator / generator */
PHP_FUNCTION(xrange)
{
	if (return_value_used) {
		int param_count = ZEND_NUM_ARGS();

		/* because I'm passing the arguments as an array, I'll need to manually
		   check arg length. */
		if (param_count != 2 && param_count != 3) WRONG_PARAM_COUNT;

		/* retrieve the function's argument list */
		zval ***params = (zval ***) safe_emalloc(param_count, sizeof(zval*), 0);
		if (zend_get_parameters_array_ex(param_count, params) == FAILURE) {
			efree(params);
			WRONG_PARAM_COUNT;
		}

		/* shove new XRangeIterator into return_value */
		Z_TYPE_P(return_value) = IS_OBJECT;
		return_value->value.obj = xrange_xri_object_new(
			php_xrange_xri_entry TSRMLS_CC
		);

		/* setup call to XRangeIterator's constructor (must do manually) */
		zval *retval = NULL;
		zval methodName;
		ZVAL_STRING(&methodName, "__construct", 0);

		/* pass all arguments through to the XRangeIterator constructor */
		if (call_user_function_ex(
			NULL, &return_value, &methodName, &retval, param_count, params, 0,
			NULL TSRMLS_CC
		) == FAILURE) {
			/* if this fails, something catastrophic has occurred. will
			   return an unconstructed object. */
			efree(params);
			php_error_docref(
				NULL TSRMLS_CC, E_ERROR,
				"Unable to call constructor for XRangeIterator"
			);
			RETURN_FALSE;
		}

		/* clean-up */
		efree(params);
		if (retval) {
			if (Z_TYPE_P(retval) && !Z_BVAL_P(retval)) {
				//zend_objects_store_del_ref(return_value TSRMLS_CC);
				// TODO: free unused & orphaned instance				
				zval_ptr_dtor(&retval);
				RETURN_FALSE;
			}
			zval_ptr_dtor(&retval);
		}
	} else {
		/* in case we're being called uselessly, save the processing time.
		   this should be profoundly difficult to unit test. */
		return;
	}
}
/* }}} */

/* {{{ proto XRangeIterator __construct(mixed $low, mixed $high [, mixed $step])
   Return a configured range iterator / generator */
PHP_METHOD(PHP_XRANGE_XRI_NAME, __construct)
{
	if (!getThis()) {
		php_error_docref(
			NULL TSRMLS_CC, E_WARNING, "Don't call the constructor statically"
		);
		RETURN_FALSE;
	}

	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;

	/* parse argument list */
	internalStorage->step = 1.0; /* default */

	if (zend_parse_parameters(
		ZEND_NUM_ARGS() TSRMLS_CC, "dd|d",
		&internalStorage->low, &internalStorage->high, &internalStorage->step
	) == FAILURE) RETURN_FALSE;

	/* if every value can be truncated, then always return an int */
	internalStorage->type =
		floor(internalStorage->low) != internalStorage->low
		|| floor(internalStorage->high) != internalStorage->high
		|| floor(internalStorage->step) != internalStorage->step
		? IS_DOUBLE : IS_LONG;

	/* automatically adjust the sign of the "step" to match direction */
	if (
		(
			internalStorage->high < internalStorage->low
			&& internalStorage->step > 0
		) || (
			internalStorage->high > internalStorage->low
			&& internalStorage->step < 0
		)
	) internalStorage->step *= -1;

	/* calculate the total number of iterations before completion */
	double iterations = fabs(
		(internalStorage->high - internalStorage->low) / internalStorage->step
	);

	/* take advantage of the truncation during casting. don't overflow. */
#if defined(PHP_WIN32) && _MSC_VER < 1300
	internalStorage->iterations = (__int64) iterations;
#else
	internalStorage->iterations = (long long) iterations;
#endif
	internalStorage->iteration = 0;
}

/* {{{ proto mixed current()
   Return the current value (generated at call time) */
PHP_METHOD(PHP_XRANGE_XRI_NAME, current)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;

	if (internalStorage->type == IS_LONG) {
		long currentTmpValue =
			(long) internalStorage->low +
			(long) internalStorage->step *
			(long) internalStorage->iteration;
	
		RETURN_LONG(currentTmpValue);
	} else {
		double currentTmpValue =
			internalStorage->low +
			internalStorage->step *
			(double) internalStorage->iteration;
	
		RETURN_DOUBLE(currentTmpValue);
	}
}
/* }}} */

/* {{{ proto int key()
   Return the current key. 0 .. end iteration */
PHP_METHOD(PHP_XRANGE_XRI_NAME, key)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	RETURN_LONG(internalStorage->iteration);
}
/* }}} */

/* {{{ proto void next()
   Increments the iterator */
PHP_METHOD(PHP_XRANGE_XRI_NAME, next)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	++internalStorage->iteration;
}
/* }}} */

/* {{{ proto void rewind()
   Rewinds the iterator back to the beginning. */
PHP_METHOD(PHP_XRANGE_XRI_NAME, rewind)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	internalStorage->iteration = 0;
}
/* }}} */

/* {{{ proto bool valid()
   Current entry is valid? */
PHP_METHOD(PHP_XRANGE_XRI_NAME, valid)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	RETURN_BOOL(internalStorage->iteration <= internalStorage->iterations);
}
/* }}} */

/* {{{ proto void seek(int $index)
   Current entry is valid? */
PHP_METHOD(PHP_XRANGE_XRI_NAME, seek)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	long seekIndex = 0;

	if (zend_parse_parameters(
		ZEND_NUM_ARGS() TSRMLS_CC, "l", &seekIndex
	) == FAILURE) RETURN_FALSE;

	if (seekIndex < 0 || seekIndex > internalStorage->iterations) {
		/* are we out of bounds? throw */
		zend_throw_exception(
			spl_ce_OutOfBoundsException, "Invalid seek position",
			0 TSRMLS_CC
		);
	} else {
		/* in-bounds, update our position */
		internalStorage->iteration = seekIndex;
	}
}
/* }}} */

/* {{{ proto int count()
   Count of range */
PHP_METHOD(PHP_XRANGE_XRI_NAME, count)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;
	RETURN_LONG(internalStorage->iterations + 1);
}
/* }}} */

/* {{{ proto string __toString()
   String representation of object */
PHP_METHOD(PHP_XRANGE_XRI_NAME, __toString)
{
	xrange_module_storage *internalStorage = PHP_XRANGE_ZOS_GET;

	char *str;
	spprintf(
		&str, 0, "XRangeIterator(%g, %g, %g)",
		internalStorage->low,
		internalStorage->high,
		internalStorage->step
	);

	ZVAL_STRING(return_value, str, 0);
}
/* }}} */

/******************************************************************************/

/* {{{ proto bool OddFilterIterator::accept(void)
   Current entry is odd? */
PHP_METHOD(PHP_XRANGE_ODDFILTERITERATOR_NAME, accept)
{
	zval *currentValue;

	// method A: bypass getInnerIterator() call
	spl_dual_it_object *intern =
		(spl_dual_it_object*) zend_object_store_get_object(getThis() TSRMLS_CC);
	zend_call_method_with_0_params(
		&intern->inner.zobject, intern->inner.ce,
		NULL, "current", &currentValue
	);
	// TODO: method B - use getInnerIterator() w/ compilation option

	if (Z_TYPE_P(currentValue) != IS_LONG) convert_to_long(currentValue);
	int isOdd = Z_LVAL_P(currentValue) & 1;

	zval_ptr_dtor(&currentValue); /* clean-up */
	RETURN_BOOL(isOdd);
}
/* }}} */

/* {{{ proto bool EvenFilterIterator::accept(void)
   Current entry is even? */
PHP_METHOD(PHP_XRANGE_EVENFILTERITERATOR_NAME, accept)
{
	zval *currentValue;

	/* method A: bypass getInnerIterator() call */
	spl_dual_it_object *intern =
		(spl_dual_it_object*) zend_object_store_get_object(getThis() TSRMLS_CC);
	zend_call_method_with_0_params(
		&intern->inner.zobject, intern->inner.ce,
		NULL, "current", &currentValue
	);
	/* TODO: method B - use getInnerIterator() w/ compilation option */

	if (Z_TYPE_P(currentValue) != IS_LONG) convert_to_long(currentValue);
	int isEven = !(Z_LVAL_P(currentValue) & 1);

	zval_ptr_dtor(&currentValue); /* clean-up */
	RETURN_BOOL(isEven);
}
/* }}} */

/* {{{ proto bool NumericFilterIterator::accept(void)
   Current entry is numeric? */
PHP_METHOD(PHP_XRANGE_NUMERICFILTERITERATOR_NAME, accept)
{
	zval *currentValue;

	/* method A: bypass getInnerIterator() call */
	spl_dual_it_object *intern =
		(spl_dual_it_object*) zend_object_store_get_object(getThis() TSRMLS_CC);
	zend_call_method_with_0_params(
		&intern->inner.zobject, intern->inner.ce,
		NULL, "current", &currentValue
	);
	/* TODO: method B - use getInnerIterator() w/ compilation option */

	int isNumeric;

	/* this code comes from is_numeric() the implementation. it's here to
	 * to eliminate the overhead of a PHP function call. */
	switch (Z_TYPE_P(currentValue)) {
	case IS_LONG:
	case IS_DOUBLE:
		isNumeric = 1;
		break;
	case IS_STRING:
		isNumeric =
			is_numeric_string(
				Z_STRVAL_P(currentValue), Z_STRLEN_P(currentValue),
				NULL, NULL, 0
			) ? 1 : 0;
		break;
	default:
		isNumeric = 0;
		break;
	}

	zval_ptr_dtor(&currentValue); /* clean-up */
	RETURN_BOOL(isNumeric);
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
