/*
   +----------------------------------------------------------------------+
   | This source file is subject to version 3.0 of the PHP license,       |
   | that is bundled with this package in the file LICENSE, and is        |
   | available through the world-wide-web at the following url:           |
   | http://www.php.net/license/3_0.txt.                                  |
   | If you did not receive a copy of the PHP license and are unable to   |
   | obtain it through the world-wide-web, please send a note to          |
   | license@php.net so we can mail you a copy immediately.               |
   +----------------------------------------------------------------------+
   | Authors: Hartmut Holzgraefe <hartmut@php.net>                        |
   +----------------------------------------------------------------------+
*/

/* $ Id: $ */ 

#include "php_html_parse.h"

#if HAVE_HTML_PARSE

static void _start_cb(void *data, ekhtml_string_t *tag, ekhtml_attr_t *attrs) {
  php_html_parser *parser = (php_html_parser *)data;
	zval *cb_retval, *cb_tag, *cb_array;
	zval **cb_args[2];
	TSRMLS_FETCH();

  if (parser->startElementHandler) {
	/* create zval string parameter for tag name */
	MAKE_STD_ZVAL(cb_tag);
	ZVAL_STRINGL(cb_tag, (char *)tag->str, tag->len, 1); 
	cb_args[0] = &cb_tag;
	
	/* create zval array parameter for tag attributes */
	MAKE_STD_ZVAL(cb_array);
	array_init(cb_array);
	while(attrs) {
	  /* hash key argument needs to be zero-terminated, 
		 so we have to create a copy */
	  char *name = emalloc(attrs->name.len +1);
	  strlcpy(name, attrs->name.str, attrs->name.len +1);

	  /* in HTML attributes may be just set without a value */
	  if(attrs->isBoolean) {
		add_assoc_bool(cb_array, name, 1);
	  } else {
		add_assoc_stringl(cb_array, name, (char *)attrs->val.str, attrs->val.len, 1);
	  }

	  /* free hash key name copy */
	  efree(name);

	  /* get next attribute */
	  attrs = attrs->next;
	}
	cb_args[1] = &cb_array;

	/* now call the handler function with the prepared parameters */
	  if (call_user_function_ex(EG(function_table), NULL, parser->startElementHandler, &cb_retval, 2, cb_args, 0, NULL TSRMLS_CC) == SUCCESS && cb_retval) {
		  zval_ptr_dtor(&cb_retval);
	}

	/* cleanup */
	  zval_ptr_dtor(&cb_tag);
	  zval_ptr_dtor(&cb_array);
  }
} 

static void _end_cb(void *data, ekhtml_string_t *tag) {
  php_html_parser *parser = (php_html_parser *)data;
	zval *cb_retval, *cb_tag;
	zval **cb_args[1];
	TSRMLS_FETCH();

  if (parser->endElementHandler) {
	MAKE_STD_ZVAL(cb_tag);
	ZVAL_STRINGL(cb_tag, (char *)tag->str, tag->len, 1); 
	cb_args[0] = &cb_tag;
	  if (call_user_function_ex(EG(function_table), NULL, parser->endElementHandler, &cb_retval, 1, cb_args, 0, NULL TSRMLS_CC) == SUCCESS && cb_retval) {
		  zval_ptr_dtor(&cb_retval);
	}
	  zval_ptr_dtor(&cb_tag);
  }
} 

static void _comment_cb(void *data,  ekhtml_string_t *comment) {
  php_html_parser *parser = (php_html_parser *)data;
	zval *cb_retval, *cb_comment;
	zval **cb_args[1];
	TSRMLS_FETCH();

  if (parser->commentHandler) {
	MAKE_STD_ZVAL(cb_comment);
	ZVAL_STRINGL(cb_comment, (char *)comment->str, comment->len, 1); 
	cb_args[0] = &cb_comment;
	  if (call_user_function_ex(EG(function_table), NULL, parser->commentHandler, &cb_retval, 1, cb_args, 0, NULL TSRMLS_CC) == SUCCESS && cb_retval) {
		  zval_ptr_dtor(&cb_retval);
	}
	  zval_ptr_dtor(&cb_comment);
  }
} 

static void _data_cb(void *data,  ekhtml_string_t *characterData) {
  php_html_parser *parser = (php_html_parser *)data;
	zval *cb_retval, *cb_data;
	zval **cb_args[1];
	TSRMLS_FETCH();

  if (parser->characterDataHandler) {
	MAKE_STD_ZVAL(cb_data);
	ZVAL_STRINGL(cb_data, (char *)characterData->str, characterData->len, 1); 
	cb_args[0] = &cb_data;
	  if (call_user_function_ex(EG(function_table), NULL, parser->characterDataHandler, &cb_retval, 1, cb_args, 0, NULL TSRMLS_CC) == SUCCESS && cb_retval) {
		  zval_ptr_dtor(&cb_retval);
	}
	  zval_ptr_dtor(&cb_data);
  }
} 
/* {{{ Resource destructors */
int le_html_parser;
void html_parser_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
	php_html_parser * resource = (php_html_parser *)(rsrc->ptr);

	do {
		/* destroy HTML parser */
		if(resource->parser) {
		  ekhtml_parser_destroy(resource->parser);
		}

		/* destruct any set callback handlers */
		  if (resource->startElementHandler) {
			  zval_ptr_dtor(&resource->startElementHandler);
		  }
		  if (resource->endElementHandler) {
			  zval_ptr_dtor(&resource->endElementHandler);
		  }
		  if (resource->characterDataHandler) {
			  zval_ptr_dtor(&resource->characterDataHandler);
		  }
		  if (resource->commentHandler) {
			zval_ptr_dtor(&resource->commentHandler);
		  }
	} while (0);

	efree(resource);
}

/* }}} */

/* {{{ html_parse_functions[] */
function_entry html_parse_functions[] = {
	PHP_FE(html_parser_create  , html_parser_create_arg_info)
	PHP_FE(html_parser_free    , html_parser_free_arg_info)
	PHP_FE(html_parser_starttag_handler, html_parser_starttag_handler_arg_info)
	PHP_FE(html_parser_endtag_handler, html_parser_endtag_handler_arg_info)
	PHP_FE(html_parser_data_handler, html_parser_data_handler_arg_info)
	PHP_FE(html_parser_comment_handler, html_parser_comment_handler_arg_info)
	PHP_FE(html_parser_parse   , html_parser_parse_arg_info)
	{ NULL, NULL, NULL }
};
/* }}} */


/* {{{ html_parse_module_entry
 */
zend_module_entry html_parse_module_entry = {
	STANDARD_MODULE_HEADER,
	"html_parse",
	html_parse_functions,
	PHP_MINIT(html_parse),     /* Replace with NULL if there is nothing to do at php startup   */ 
	PHP_MSHUTDOWN(html_parse), /* Replace with NULL if there is nothing to do at php shutdown  */
	PHP_RINIT(html_parse),     /* Replace with NULL if there is nothing to do at request start */
	PHP_RSHUTDOWN(html_parse), /* Replace with NULL if there is nothing to do at request end   */
	PHP_MINFO(html_parse),
	"1.0.0", 
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_HTML_PARSE
ZEND_GET_MODULE(html_parse)
#endif


/* {{{ PHP_MINIT_FUNCTION */
PHP_MINIT_FUNCTION(html_parse)
{
	le_html_parser = zend_register_list_destructors_ex(html_parser_dtor, 
						   NULL, "html_parser", module_number);

	/* add your stuff here */

	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MSHUTDOWN_FUNCTION */
PHP_MSHUTDOWN_FUNCTION(html_parse)
{

	/* add your stuff here */

	return SUCCESS;
}
/* }}} */


/* {{{ PHP_RINIT_FUNCTION */
PHP_RINIT_FUNCTION(html_parse)
{
	/* add your stuff here */

	return SUCCESS;
}
/* }}} */


/* {{{ PHP_RSHUTDOWN_FUNCTION */
PHP_RSHUTDOWN_FUNCTION(html_parse)
{
	/* add your stuff here */

	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MINFO_FUNCTION */
PHP_MINFO_FUNCTION(html_parse)
{
	php_info_print_box_start(0);
	php_printf("<p>HTML parser extenion</p>\n");
	php_printf("<p>Version 1.0.0stable (2007-11-28)</p>\n");
	php_printf("<p><b>Authors:</b></p>\n");
	php_printf("<p>Hartmut Holzgraefe &lt;hartmut@php.net&gt; (lead)</p>\n");
	php_info_print_box_end();
	/* add your stuff here */

}
/* }}} */


/* {{{ proto resource html_parser html_parser_create(void)
  create an HTML parser */
PHP_FUNCTION(html_parser_create)
{
	php_html_parser * return_res = (php_html_parser *)ecalloc(1, sizeof(php_html_parser));
	long return_res_id = -1;



	if (ZEND_NUM_ARGS()>0)  {
		WRONG_PARAM_COUNT;
	}


	do {
		return_res->parser = ekhtml_parser_new(return_res);
		if(! return_res->parser) RETURN_FALSE;

		/* initially no PHP callbacks are set */
		return_res->startElementHandler  = NULL;
		return_res->endElementHandler    = NULL;
		return_res->characterDataHandler = NULL;
		return_res->commentHandler       = NULL;

		/* all C callbacks are set right away, whether a PHP callback
		   is set is tested within the C callback */
		ekhtml_parser_startcb_add(return_res->parser, NULL, _start_cb);
		ekhtml_parser_endcb_add(return_res->parser, NULL, _end_cb);
		ekhtml_parser_commentcb_set(return_res->parser, _comment_cb);
		ekhtml_parser_datacb_set(return_res->parser, _data_cb);
	} while (0);

	return_res_id = ZEND_REGISTER_RESOURCE(return_value, return_res, le_html_parser);
}
/* }}} html_parser_create */


/* {{{ proto bool html_parser_free(resource html_parser parser)
  free an HTML parser */
PHP_FUNCTION(html_parser_free)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;




	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "r", &parser_res) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);



	do {
		FREE_RESOURCE(parser_res);
	} while (0);
}
/* }}} html_parser_free */


/* {{{ proto bool html_parser_starttag_handler(resource html_parser parser, callback handler)
  set callback function for start tags */
PHP_FUNCTION(html_parser_starttag_handler)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;

	zval * handler = NULL;



	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rz", &parser_res, &handler) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);
	if (!zend_is_callable(handler, 0, NULL)) {
	  php_error(E_WARNING, "Invalid comparison function.");
	  return;    }



	do {
		/* parameter check */
		if (!zend_is_callable(handler, 0, NULL)) {
		  php_error_docref(NULL TSRMLS_CC, E_WARNING, "2nd argument is not a valid callback");
		  RETURN_FALSE;
		}

		/* have previous? -> dispose */
		if(parser->startElementHandler) {
		  zval_ptr_dtor(&parser->startElementHandler);
		} 

		/* assign new callback handler */
		ALLOC_ZVAL(parser->startElementHandler);
		*parser->startElementHandler = *handler;
		zval_copy_ctor(parser->startElementHandler);

		RETURN_TRUE;
	} while (0);
}
/* }}} html_parser_starttag_handler */


/* {{{ proto bool html_parser_endtag_handler(resource html_parser parser, callback handler)
  set callback function for end tags */
PHP_FUNCTION(html_parser_endtag_handler)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;

	zval * handler = NULL;



	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rz", &parser_res, &handler) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);
	if (!zend_is_callable(handler, 0, NULL)) {
	  php_error(E_WARNING, "Invalid comparison function.");
	  return;    }



	do {
		if (!zend_is_callable(handler, 0, NULL)) {
		  php_error_docref(NULL TSRMLS_CC, E_WARNING, "2nd argument is not a valid callback");
		  RETURN_FALSE;
		}
		if(parser->endElementHandler) {
		  zval_ptr_dtor(&parser->endElementHandler);
		} 
		ALLOC_ZVAL(parser->endElementHandler);
		*parser->endElementHandler = *handler;
		zval_copy_ctor(parser->endElementHandler);
		RETURN_TRUE;
	} while (0);
}
/* }}} html_parser_endtag_handler */


/* {{{ proto bool html_parser_data_handler(resource html_parser parser, callback handler)
  set callback function for character data */
PHP_FUNCTION(html_parser_data_handler)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;

	zval * handler = NULL;



	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rz", &parser_res, &handler) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);
	if (!zend_is_callable(handler, 0, NULL)) {
	  php_error(E_WARNING, "Invalid comparison function.");
	  return;    }



	do {
		if (!zend_is_callable(handler, 0, NULL)) {
		  php_error_docref(NULL TSRMLS_CC, E_WARNING, "2nd argument is not a valid callback");
		  RETURN_FALSE;
		}
		if(parser->characterDataHandler) {
		  zval_ptr_dtor(&parser->characterDataHandler);
		} 
		ALLOC_ZVAL(parser->characterDataHandler);
		*parser->characterDataHandler = *handler;
		zval_copy_ctor(parser->characterDataHandler);
		RETURN_TRUE;
	} while (0);
}
/* }}} html_parser_data_handler */


/* {{{ proto bool html_parser_comment_handler(resource html_parser parser, callback handler)
  set callback function for HTML comments */
PHP_FUNCTION(html_parser_comment_handler)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;

	zval * handler = NULL;



	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rz", &parser_res, &handler) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);
	if (!zend_is_callable(handler, 0, NULL)) {
	  php_error(E_WARNING, "Invalid comparison function.");
	  return;    }



	do {
		if (!zend_is_callable(handler, 0, NULL)) {    php_error_docref(NULL TSRMLS_CC, E_WARNING, "2nd argument is not a valid callback");
		  RETURN_FALSE;
		}
		if(parser->commentHandler) {
		  zval_ptr_dtor(&parser->commentHandler);
		} 
		ALLOC_ZVAL(parser->commentHandler);
		*parser->commentHandler = *handler;
		zval_copy_ctor(parser->commentHandler);
		RETURN_TRUE;
	} while (0);
}
/* }}} html_parser_comment_handler */


/* {{{ proto bool html_parser_parse(resource html_parser parser, string data [, bool finish])
  feed input into parser */
PHP_FUNCTION(html_parser_parse)
{
	zval * parser_res = NULL;
	int parser_resid = -1;
	php_html_parser * parser;

	const char * data = NULL;
	int data_len = 0;
	zend_bool finish = 0;



	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "rs|b", &parser_res, &data, &data_len, &finish) == FAILURE) {
		return;
	}
	ZEND_FETCH_RESOURCE(parser, php_html_parser *, &parser_res, parser_resid, "html_parser", le_html_parser);



	do {
		{
		  /* prepare input data as needed by ekhtml_parser_feed */
		  ekhtml_string_t str;
		  str.str = data;
		  str.len = data_len;
		
		  /* feed data */
		  ekhtml_parser_feed(parser->parser, &str);
		
		  /* request parsing of feeded data */
		  ekhtml_parser_flush(parser->parser, finish);
		}
	} while (0);
}
/* }}} html_parser_parse */

#endif /* HAVE_HTML_PARSE */


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
