/* vim: set expandtab tabstop=4 shiftwidth=4: */
/*
  +----------------------------------------------------------------------+
  | PHP Version 5                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2004 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.0 of the PHP license,       |
  | that is bundled with this package in the file LICENSE, and is        |
  | available at through the world-wide-web at                           |
  | http://www.php.net/license/3_0.txt                                   |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Author: Piotr Klaban <makler@php.net>                                |
  +----------------------------------------------------------------------+

  $Id: esmtp.c,v 1.1 2004/09/02 07:34:35 makler Exp $ 
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#if HAVE_LIBESMTP

#include "php.h"
#include "php_globals.h"
#include "ext/standard/info.h"
#include "php_streams.h"
#include "php_esmtp.h"
#include "ext/standard/php_var.h"
#include "ext/standard/php_smart_str.h"
#include <auth-client.h>
#include <libesmtp.h>

static const int do_serialize = 1;

static int le_esmtp_session;
static int le_esmtp_message;
static int le_esmtp_recipient;
static int le_esmtp_etrnnode;
static int le_esmtp_auth;

static zend_class_entry *esmtp_session_class_entry;
static zend_class_entry *esmtp_message_class_entry;
static zend_class_entry *esmtp_recipient_class_entry;
static zend_class_entry *esmtp_etrnnode_class_entry;
static zend_class_entry *esmtp_auth_class_entry;

static smtp_session_t _php_get_esmtp_session(zval *id TSRMLS_DC);
static smtp_message_t _php_get_esmtp_message(zval *id TSRMLS_DC);
static smtp_recipient_t _php_get_esmtp_recipient(zval *id TSRMLS_DC);
static smtp_etrn_node_t _php_get_esmtp_etrnnode(zval *id TSRMLS_DC);
static auth_context_t _php_get_esmtp_auth(zval *id TSRMLS_DC);

/* {{{ internal function OBJgetProperty
 */
static void *OBJgetProperty(zval *id, char *name, int namelen, int proptype TSRMLS_DC)
{
	zval **tmp;
	int id_to_find;
	void *property;
	int type;
  
	if (id) {
		if (zend_hash_find(Z_OBJPROP_P(id), name, namelen+1, (void **)&tmp) == FAILURE) {
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find property %s", name);
			return NULL;
		}
		id_to_find = Z_LVAL_PP(tmp);
	} else {
		return NULL;
	}

	property = zend_list_find(id_to_find, &type);

	if (!property || type != proptype) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find identifier (%d)", id_to_find);
		return NULL;
	}

	return property;
}
/* }}} */


/* {{{ internal function _php_get_esmtp_session */
static smtp_session_t _php_get_esmtp_session(zval *id TSRMLS_DC)
{
    void *obj = OBJgetProperty(id, "Esmtp_Session", sizeof("Esmtp_Session")-1, le_esmtp_session TSRMLS_CC);
    if (!obj) {
        php_error_docref(NULL TSRMLS_CC, E_ERROR, "Called object is not an esmtp session");
    }
    return (smtp_session_t)obj;
}
/* }}} */

/* {{{ internal function _php_get_esmtp_message */
static smtp_message_t _php_get_esmtp_message(zval *id TSRMLS_DC)
{
    void *obj = OBJgetProperty(id, "Esmtp_Message", sizeof("Esmtp_Message")-1, le_esmtp_message TSRMLS_CC);
    if (!obj) {
        php_error_docref(NULL TSRMLS_CC, E_ERROR, "Called object is not an esmtp message");
    }
    return (smtp_message_t)obj;
}
/* }}} */

/* {{{ internal function _php_get_esmtp_recipient */
static smtp_recipient_t _php_get_esmtp_recipient(zval *id TSRMLS_DC)
{
    void *obj = OBJgetProperty(id, "Esmtp_Recipient", sizeof("Esmtp_Recipient")-1, le_esmtp_recipient TSRMLS_CC);
    if (!obj) {
        php_error_docref(NULL TSRMLS_CC, E_ERROR, "Called object is not an esmtp recipient");
    }
    return (smtp_recipient_t)obj;
}
/* }}} */

/* {{{ internal function _php_get_esmtp_etrnnode */
static smtp_etrn_node_t _php_get_esmtp_etrnnode(zval *id TSRMLS_DC)
{
    void *obj = OBJgetProperty(id, "Esmtp_Etrnnode", sizeof("Esmtp_Etrnnode")-1, le_esmtp_etrnnode TSRMLS_CC);
    if (!obj) {
        php_error_docref(NULL TSRMLS_CC, E_ERROR, "Called object is not an esmtp etrnnode");
    }
    return (smtp_etrn_node_t)obj;
}
/* }}} */

/* {{{ internal function _php_get_esmtp_auth */
static auth_context_t _php_get_esmtp_auth(zval *id TSRMLS_DC)
{
    void *obj = OBJgetProperty(id, "Esmtp_Auth", sizeof("Esmtp_Auth")-1, le_esmtp_auth TSRMLS_CC);
    if (!obj) {
        php_error_docref(NULL TSRMLS_CC, E_ERROR, "Called object is not an esmtp authcontext");
    }
    return (auth_context_t)obj;
}
/* }}} */

function_entry esmtp_functions[] = {
    ZEND_FALIAS(esmtp_version,                  smtp_version,             NULL)
    ZEND_FALIAS(esmtp_errno,                    smtp_errno,               NULL)
    ZEND_FALIAS(esmtp_strerror,                 smtp_strerror,            NULL)
    ZEND_FALIAS(esmtp_starttls_set_password_cb, smtp_starttls_set_password_cb,            NULL)
    {NULL, NULL, NULL, 0}
};

/**
 * Universal php destructor for all the esmtp objects
 **/
static void _php_free_esmtp_object(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
    char *p = NULL;

    if (rsrc->type == le_esmtp_session) {
#ifdef _auth_client_h
        smtp_auth_set_context(rsrc->ptr, NULL);
#endif
        p = (char *)smtp_set_application_data(rsrc->ptr, NULL);
        if (p)
            efree(p);
        smtp_destroy_session(rsrc->ptr);
    } else if (rsrc->type == le_esmtp_message) {
        p = (char *)smtp_message_set_application_data(rsrc->ptr, NULL);
        if (p)
            efree(p);
    } else if (rsrc->type == le_esmtp_recipient) {
        p = (char *)smtp_recipient_set_application_data(rsrc->ptr, NULL);
        if (p)
            efree(p);
    } else if (rsrc->type == le_esmtp_etrnnode) {
        p = (char *)smtp_etrn_set_application_data(rsrc->ptr, NULL);
        if (p)
            efree(p);
    } else if (rsrc->type == le_esmtp_auth) {
    }
}
    
zend_module_entry esmtp_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
    STANDARD_MODULE_HEADER,
#endif
    "esmtp",
    esmtp_functions,
    PHP_MINIT(esmtp),
    PHP_MSHUTDOWN(esmtp),
    PHP_RINIT(esmtp),    
    PHP_RSHUTDOWN(esmtp),
    PHP_MINFO(esmtp),
#if ZEND_MODULE_API_NO >= 20010901
    "0.2",
#endif
    STANDARD_MODULE_PROPERTIES
};


#ifdef COMPILE_DL_ESMTP
ZEND_GET_MODULE(esmtp)
#endif

/* {{{ php_esmtp_init_globals
 */
static void php_esmtp_init_globals(zend_esmtp_globals *esmtp_globals)
{
            esmtp_globals->starttls_data = NULL;
}


/* {{{ proto string smtp_version()
   Retrieve version information for the libESMTP in use. */
PHP_FUNCTION(smtp_version)
{
    char buf[256];
    
    if (ZEND_NUM_ARGS() != 0) {
        WRONG_PARAM_COUNT;
    }

    if (!smtp_version (buf, 255, 0)) {
        RETURN_FALSE;
    } else {
        RETURN_STRING(buf, 1);
    }
}
/* }}} */

/* {{{ proto int smtp_errno()
   Retrieve the error code for the most recently failed API in the calling thread. */
PHP_FUNCTION(smtp_errno)
{
    if (ZEND_NUM_ARGS() != 0) {
        WRONG_PARAM_COUNT;
    }
    
    RETURN_LONG(smtp_errno());
}
/* }}} */

/* {{{ proto string smtp_strerror(int error)
   Translate a libESMTP error number to a string suitable for use in an application error message. */
PHP_FUNCTION(smtp_strerror)
{
    int argc = ZEND_NUM_ARGS();
    long error;
    char buf[1024];
    char *ret = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &error) == FAILURE) 
        return;

    if ((ret = smtp_strerror (error, buf, 1022)) == NULL) {
        RETURN_NULL();
    } else {
        RETURN_STRING(ret, 1);
    }

}
/* }}} */

/* {{{ smtp_starttls_passwordcb_t */
static int
callback_esmtp_starttls_password(char *buf, int buflen, int rwflag, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[2];
    int param_len = 1, ret = 0;
    TSRMLS_FETCH();

    if (!arg || !buf || buflen == 0)
        return 0;

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (starttls_password)");
        return 0;
    }

    MAKE_STD_ZVAL(params[0]);
    convert_to_long(params[0]);
    Z_LVAL_P(params[0]) = rwflag;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[1] = *z_userdata;
    }
    MAKE_STD_ZVAL(retval);
    /* callback prototype: function */
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (starttls_password)");
    } else {
        convert_to_string(retval);
        if (!retval || Z_TYPE_P(retval) != IS_STRING ||
            Z_STRLEN_P(retval) == 0 || Z_STRLEN_P(retval) > buflen) {
            ret = 0;
        } else {
            strcpy(buf, Z_STRVAL_P(retval));
            ret = Z_STRLEN_P(retval);
        }
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);

    return ret;
}
/* }}} */

/* {{{ proto int smtp_starttls_set_password_cb(mixed cb [, mixed arg])
   Call the callback function when OpenSSL requires a password. */
PHP_FUNCTION(smtp_starttls_set_password_cb)
{
    int argc = ZEND_NUM_ARGS();
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    ZVAL_ADDREF(cb);
    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        ZVAL_ADDREF(arg);
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }

    ret = smtp_starttls_set_password_cb(callback_esmtp_starttls_password, z_data);
    
    /* save z_data in global variable to free at exit */
    if (ESMTP_G(starttls_data)) {
        zval_ptr_dtor(&ESMTP_G(starttls_data));
        ESMTP_G(starttls_data) = NULL;
    }
    ESMTP_G(starttls_data) = z_data;

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto object esmtp_session::__construct()
   Create session object. */
PHP_METHOD(esmtp_session, __construct)
{
    smtp_session_t session;
    
    if (ZEND_NUM_ARGS() != 0) {
        WRONG_PARAM_COUNT;
    }

    session = smtp_create_session();
    if (session) {
        int ret = zend_list_insert(session, le_esmtp_session);
        object_init_ex(getThis(), esmtp_session_class_entry);
        add_property_resource(getThis(), "Esmtp_Session", ret);
        zend_list_addref(ret);
    } else {
        RETURN_NULL();
    }
}
/* }}} */

/* {{{ proto object esmtp_sessin::add_message()
   Add a message to the list of messages to be transferred. */
PHP_METHOD(esmtp_session, add_message)
{
    smtp_session_t session = NULL;
    smtp_message_t message = NULL;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    message = smtp_add_message(session);

    if (message) {
        int ret = zend_list_insert(message, le_esmtp_message);
        object_init_ex(return_value, esmtp_message_class_entry);
        add_property_resource(return_value, "Esmtp_Message", ret);
        zend_list_addref(ret);
    } else {
        RETURN_NULL();
    }
}
/* }}} */

/* {{{ smtp_enumerate_messagecb_t */
static void
callback_esmtp_enumerate_messages(smtp_message_t message, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[2];
    int param_len = 1, rnum;
    TSRMLS_FETCH();

    if (!arg)
        return;

    MAKE_STD_ZVAL(params[0]);

    rnum = zend_list_insert(message, le_esmtp_message);
    object_init_ex(params[0], esmtp_message_class_entry);
    add_property_resource(params[0], "Esmtp_Message", rnum);
    // ref or not ref? zend_list_addref(rnum);

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (enumerate_messages)");
        zval_ptr_dtor(&params[0]);
        return;
    }

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[1] = *z_userdata;
    }
    /* would not check for return value */
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (enumerate_messages)");
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);
}
/* }}} */

/* {{{ proto int esmtp_session::enumerate_messages(mixed cb [, mixed arg])
   Call the callback function once for each message in an smtp session. */
PHP_METHOD(esmtp_session, enumerate_messages)
{
    int argc = ZEND_NUM_ARGS();
    smtp_session_t session = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }
    
    ret = smtp_enumerate_messages (session, callback_esmtp_enumerate_messages, z_data);

    zval_ptr_dtor(&z_data);

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_session::set_hostname(string hostname)
   Set the name of the localhost. */
PHP_METHOD(esmtp_session, set_hostname)
{
    char *hostname = NULL;
    int argc = ZEND_NUM_ARGS();
    int hostname_len;
    smtp_session_t session = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &hostname, &hostname_len) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_hostname(session, hostname));
}
/* }}} */

/* {{{ proto int esmtp_session::set_server(string hostport)
   Set the host name and service for the client connection. */
PHP_METHOD(esmtp_session, set_server)
{
    char *hostport = NULL;
    int argc = ZEND_NUM_ARGS();
    int hostport_len;
    smtp_session_t session = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &hostport, &hostport_len) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_server(session, hostport));
}
/* }}} */

/* {{{ proto long esmtp_session::set_timeout(int which, long value)
   Set the timeouts. */
PHP_METHOD(esmtp_session, set_timeout)
{
    int argc = ZEND_NUM_ARGS();
    long which;
    long value;
    smtp_session_t session = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "ll", &which, &value) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_timeout(session, which, value));
}
/* }}} */

/* {{{ proto int esmtp_session::option_require_all_recipients(int state)
   Some applications can't handle one recipient from many failing particularly well. */
PHP_METHOD(esmtp_session, option_require_all_recipients)
{
    int argc = ZEND_NUM_ARGS();
    zend_bool state;
    smtp_session_t session = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "b", &state) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_option_require_all_recipients(session, state));
}
/* }}} */

/* {{{ smtp_eventcb_t */
static void
callback_esmtp_session_event(smtp_session_t session, int event_no, void *arg, ...)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[10];
    int param_len = 1, rnum;
    va_list ap;
    smtp_message_t message;
    smtp_recipient_t recipient;
    char *text;
    int intarg, *intptr = NULL;
    long longarg;
    TSRMLS_FETCH();

    if (!arg)
        return;

    MAKE_STD_ZVAL(params[0]);

    rnum = zend_list_insert(session, le_esmtp_session);
    object_init_ex(params[0], esmtp_session_class_entry);
    add_property_resource(params[0], "Esmtp_Session", rnum);
    // ref or not ref? zend_list_addref(rnum);

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (session_event)");
        zval_ptr_dtor(&params[0]);
        return;
    }

    /* send event_no identifier */
    MAKE_STD_ZVAL(params[1]);
    convert_to_long(params[1]);
    Z_LVAL_P(params[1]) = event_no;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[2] = *z_userdata;
    }

    /* rest of the parameters */
    va_start (ap, arg);
    /* Protocol progress */
    switch (event_no) {
    case SMTP_EV_CONNECT:        /* <empty arg list> */
        break;
    case SMTP_EV_MAILSTATUS:    /* char* message->reverse_path_mailbox, message */
        text = va_arg (ap, char *);
        message = va_arg (ap, smtp_message_t);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_string(params[param_len]);
        ZVAL_STRING(params[param_len], text, 1);
        ++param_len;
        
        MAKE_STD_ZVAL(params[param_len]);
        rnum = zend_list_insert(message, le_esmtp_message);
        object_init_ex(params[param_len], esmtp_message_class_entry);
        add_property_resource(params[param_len], "Esmtp_Message", rnum);
        ++param_len;
        break;
    case SMTP_EV_RCPTSTATUS: /* char* session->rsp_recipient->mailbox, session->rsp_recipient */
        text = va_arg (ap, char *);
        recipient = va_arg (ap, smtp_recipient_t);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_string(params[param_len]);
        ZVAL_STRING(params[param_len], text, 1);
        ++param_len;
        
        MAKE_STD_ZVAL(params[param_len]);
        rnum = zend_list_insert(recipient, le_esmtp_recipient);
        object_init_ex(params[param_len], esmtp_recipient_class_entry);
        add_property_resource(params[param_len], "Esmtp_Recipient", rnum);
        ++param_len;
        break;
    case SMTP_EV_MESSAGEDATA:    /* session->current_message, int len */
        message = va_arg (ap, smtp_message_t);
        intarg = va_arg (ap, int);
        
        MAKE_STD_ZVAL(params[param_len]);
        rnum = zend_list_insert(message, le_esmtp_message);
        object_init_ex(params[param_len], esmtp_message_class_entry);
        add_property_resource(params[param_len], "Esmtp_Message", rnum);
        ++param_len;
        
        MAKE_STD_ZVAL(params[param_len]);
        convert_to_long(params[param_len]);
        Z_LVAL_P(params[param_len]) = intarg;
        ++param_len;
        break;
    case SMTP_EV_MESSAGESENT:    /* session->current_message */
        message = va_arg (ap, smtp_message_t);

        MAKE_STD_ZVAL(params[param_len]);
        rnum = zend_list_insert(message, le_esmtp_message);
        object_init_ex(params[param_len], esmtp_message_class_entry);
        add_property_resource(params[param_len], "Esmtp_Message", rnum);
        ++param_len;
        break;
    case SMTP_EV_DISCONNECT:    /* <empty arg list> */
        break;

    /* Protocol extension progress */
    case SMTP_EV_ETRNSTATUS:    /* int node->option, char* node->domain */
        intarg = va_arg (ap, int);
        text = va_arg (ap, char *);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_long(params[param_len]);
        Z_LVAL_P(params[param_len]) = intarg;
        ++param_len;

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_string(params[param_len]);
        ZVAL_STRING(params[param_len], text, 1);
        ++param_len;
        break;

    /* Required extensions */
    case SMTP_EV_EXTNA_DSN:        /* int &quit_now */
        intptr = va_arg (ap, int *);
    case SMTP_EV_EXTNA_8BITMIME:    /* <empty arg list> */
        break;
    case SMTP_EV_EXTNA_STARTTLS:    /* NULL (protocol.c) */
        break;
    case SMTP_EV_EXTNA_ETRN:    /* int &quit_now */
    case SMTP_EV_EXTNA_CHUNKING:    /* int &quit_now */
        intptr = va_arg (ap, int *);
        break;
    case SMTP_EV_EXTNA_BINARYMIME:    /* <empty arg list> */
        break;

    /* Extensions specific events */
    case SMTP_EV_DELIVERBY_EXPIRED:    /* long session->min_by_time - by_time, int &adjust */
        longarg = va_arg (ap, long);
        intptr = va_arg (ap, int *);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_long(params[param_len]);
        Z_LVAL_P(params[param_len]) = longarg;
        ++param_len;
        break;

    /* STARTTLS */
    case SMTP_EV_WEAK_CIPHER:        /* int bits, int &ok */
        intarg = va_arg (ap, int);
        intptr = va_arg (ap, int *);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_long(params[param_len]);
        Z_LVAL_P(params[param_len]) = intarg;
        ++param_len;
        break;
    case SMTP_EV_STARTTLS_OK: /* SSL* ssl, SSL_get_cipher (ssl), SSL_get_cipher_bits (ssl, NULL) */
        /* UNSUPPORTED */
        break;
    case SMTP_EV_INVALID_PEER_CERTIFICATE:    /* long vfy_result, int &ok */
        longarg = va_arg (ap, long);
        intptr = va_arg (ap, int *);

        MAKE_STD_ZVAL(params[param_len]);
        convert_to_long(params[param_len]);
        Z_LVAL_P(params[param_len]) = longarg;
        ++param_len;
        break;
    case SMTP_EV_NO_PEER_CERTIFICATE:    /* int &ok */
    case SMTP_EV_WRONG_PEER_CERTIFICATE:    /* int &ok */
        intptr = va_arg (ap, int *);
        break;
    case SMTP_EV_NO_CLIENT_CERTIFICATE:    /* int &ok */
        break;
    case SMTP_EV_UNUSABLE_CLIENT_CERTIFICATE:    /* <empty arg list> */
        break;
    case SMTP_EV_UNUSABLE_CA_LIST:    /* <empty arg list> */
        break;
    }
    va_end (ap);
    
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (session_event)");
    } else {
        if (!ZVAL_IS_NULL(retval)) {
            convert_to_long(retval);
            *intptr = Z_LVAL_P(retval);
        }
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);
    zval_ptr_dtor(&params[1]);
    if (param_len > 3) {
        int i;
        for (i=3; i<param_len; ++i) {
            zval_ptr_dtor(&params[i]);
        }
    }
}
/* }}} */

/* {{{ proto int esmtp_session::set_eventcb(mixed cb [, mixed arg])
   Set a callback function to process protocol events during the SMTP session. */
PHP_METHOD(esmtp_session, set_eventcb)
{
    int argc = ZEND_NUM_ARGS();
    smtp_session_t session = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);
    
    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    ZVAL_ADDREF(cb);
    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        ZVAL_ADDREF(arg);
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }

    ret = smtp_set_eventcb(session, callback_esmtp_session_event, z_data);
 
    /* save z_data in global variable to free at exit */
    if (ESMTP_G(eventcb_data)) {
        zval_ptr_dtor(&ESMTP_G(eventcb_data));
        ESMTP_G(eventcb_data) = NULL;
    }
    ESMTP_G(eventcb_data) = z_data;

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ smtp_monitorcb_t */
static void
callback_esmtp_session_monitor(const char *buf, int buflen, int writing, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[3];
    int param_len = 1;
    TSRMLS_FETCH();

    if (!arg || !buf || buflen == 0)
        return;

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (session_monitor)");
        return;
    }

    MAKE_STD_ZVAL(params[0]);
    convert_to_string(params[0]);
    ZVAL_STRINGL(params[0], (char *)buf, buflen, 1);
    
    MAKE_STD_ZVAL(params[1]);
    convert_to_long(params[1]);
    Z_LVAL_P(params[1]) = writing;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[2] = *z_userdata;
    }
    /* would not check for return value */
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (session_monitor)");
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);
    zval_ptr_dtor(&params[1]);
}
/* }}} */

/* {{{ proto int esmtp_session::set_monitorcb(mixed cb [, mixed arg [, int headers]])
   Set a callback function to monitor the SMTP session with the server. */
PHP_METHOD(esmtp_session, set_monitorcb)
{
    int argc = ZEND_NUM_ARGS();
    long headers = 0;
    smtp_session_t session = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|zl", &cb, &arg, &headers) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    ZVAL_ADDREF(cb);
    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    add_property_zval(getThis(), "header_regex_funcname", cb);
    if (arg) {
        ZVAL_ADDREF(arg);
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
        add_property_zval(getThis(), "header_regex_userdata", arg);
    }

    ret = smtp_set_monitorcb(session, callback_esmtp_session_monitor, z_data, headers);

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_session::start_session(void)
   Initiate a mail submission session with an SMTP server. */
PHP_METHOD(esmtp_session, start_session)
{
    smtp_session_t session = NULL;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_start_session(session));
}
/* }}} */

/* {{{ proto object esmtp_session::etrn_add_node(int option, string node)
   Add an ETRN node to the SMTP session. */
PHP_METHOD(esmtp_session, etrn_add_node)
{
    char *node = NULL;
    int argc = ZEND_NUM_ARGS();
    int node_len;
    long option;
    smtp_session_t session = NULL;
    smtp_etrn_node_t etrn_node;

    if (zend_parse_parameters(argc TSRMLS_CC, "ls", &option, &node, &node_len) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    etrn_node = smtp_etrn_add_node(session, option, node);
    if (etrn_node) {
        int ret = zend_list_insert(etrn_node, le_esmtp_etrnnode);
        object_init_ex(return_value, esmtp_etrnnode_class_entry);
        add_property_resource(return_value, "Esmtp_Etrnnode", ret);
        zend_list_addref(ret);
    } else {
        RETURN_NULL();
    }
}
/* }}} */

/* {{{ proto int esmtp_session::auth_set_context(void)
   Enable or disable the SMTP AUTH verb. */
PHP_METHOD(esmtp_session, auth_set_context)
{
    int argc = ZEND_NUM_ARGS();
    zval *z_context = NULL;
    smtp_session_t session = NULL;
    auth_context_t context = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "o", &z_context) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);
    context = _php_get_esmtp_auth(z_context TSRMLS_CC);

    RETURN_LONG(smtp_auth_set_context(session, context));
}
/* }}} */

/* {{{ proto int esmtp_session::starttls_enable(int how)
   Enable or disable the SMTP STARTTLS verb. */
PHP_METHOD(esmtp_session, starttls_enable)
{
    int argc = ZEND_NUM_ARGS();
    long how;
    smtp_session_t session = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &how) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_starttls_enable(session, how));
}
/* }}} */

/* {{{ proto int esmtp_message::set_reverse_path(string mailbox)
   Set the reverse path (envelope sender) mailbox address. */
PHP_METHOD(esmtp_message, set_reverse_path)
{
    char *mailbox = NULL;
    int argc = ZEND_NUM_ARGS();
    int mailbox_len;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &mailbox, &mailbox_len) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_reverse_path(message, mailbox));
}
/* }}} */

/* {{{ proto object esmtp_message::add_recipient(string mailbox)
   Add a recipient to the message. */
PHP_METHOD(esmtp_message, add_recipient)
{
    char *mailbox = NULL;
    int argc = ZEND_NUM_ARGS();
    int mailbox_len;
    smtp_message_t message = NULL;
    smtp_recipient_t recipient = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &mailbox, &mailbox_len) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    recipient = smtp_add_recipient(message, mailbox);
    if (recipient) {
        int ret = zend_list_insert(recipient, le_esmtp_recipient);
        object_init_ex(return_value, esmtp_recipient_class_entry);
        add_property_resource(return_value, "Esmtp_Recipient", ret);
        zend_list_addref(ret);
    } else {
        RETURN_NULL();
    }
}
/* }}} */

/* {{{ smtp_enumerate_recipientcb_t */
static void
callback_esmtp_enumerate_recipient(smtp_recipient_t recipient, const char *mailbox, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[3];
    int param_len = 1, rnum;
    TSRMLS_FETCH();

    if (!arg)
        return;

    MAKE_STD_ZVAL(params[0]);

    rnum = zend_list_insert(recipient, le_esmtp_recipient);
    object_init_ex(params[0], esmtp_recipient_class_entry);
    add_property_resource(params[0], "Esmtp_Recipient", rnum);

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (enumerate_recipient)");
        zval_ptr_dtor(&params[0]);
        return;
    }

    MAKE_STD_ZVAL(params[1]);
    ZVAL_STRING(params[1], (char *)mailbox, 1);
    ++param_len;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[2] = *z_userdata;
    }
    /* would not check for return value */
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (enumerate_recipient)");
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);
    zval_ptr_dtor(&params[1]);
}
/* }}} */

/* {{{ proto int esmtp_message::enumerate_recipients(mixed cb [, mixed arg])
   Call the callback function once for each recipient in the SMTP message. */
PHP_METHOD(esmtp_message, enumerate_recipients)
{
    int argc = ZEND_NUM_ARGS();
    smtp_message_t message = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }
 
    ret = smtp_enumerate_recipients(message, callback_esmtp_enumerate_recipient, z_data);

    zval_ptr_dtor(&z_data);

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_message::set_header(string header)
   ...) Set an RFC 2822 message header. */
PHP_METHOD(esmtp_message, set_header)
{
    char *header = NULL;
    int argc = ZEND_NUM_ARGS();
    int header_len;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &header, &header_len) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_header(message, header));
}
/* }}} */

/* {{{ proto int esmtp_message::set_header_option(string header, int option)
   ...) Set an RFC 2822 message header option for header. */
PHP_METHOD(esmtp_message, set_header_option)
{
    char *header = NULL;
    int argc = ZEND_NUM_ARGS();
    int header_len;
    long option;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "sl", &header, &header_len, &option) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_header_option(message, header, option));
}
/* }}} */

/* {{{ proto int esmtp_message::set_resent_headers(bool onoff)
   Request special processing of headers which have a Resent-variation. */
PHP_METHOD(esmtp_message, set_resent_headers)
{
    int argc = ZEND_NUM_ARGS();
    zend_bool onoff;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "b", &onoff) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_resent_headers(message, onoff));
}
/* }}} */

/* {{{ smtp_messagecb_t */
/**
 *
 * if ($len == null) [begining] -> rewind file etc., return is discarded
 * else -> return as much from the file
 */
static const char *
callback_esmtp_message(void **buf, int *len, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[2];
    int param_len = 1;
    TSRMLS_FETCH();

    if (!arg)
        return 0;

    MAKE_STD_ZVAL(params[0]);
    if (len == NULL) {
        ZVAL_NULL(params[0]);
    } else {
        ZVAL_LONG(params[0], *len);
    }

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (read_message)");
        zval_ptr_dtor(&params[0]);
        return 0;
    }

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[1] = *z_userdata;
    }
    /* would not check for return value */
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (read_message)");
    } else if (len != NULL) {
        convert_to_string(retval);
        if (Z_STRLEN_P(retval) <= 0) {
            *len = 0;
        } else {
            *buf = realloc (*buf, Z_STRLEN_P(retval));
            strncpy (*buf, Z_STRVAL_P(retval), Z_STRLEN_P(retval));
            *len = Z_STRLEN_P(retval);
        }
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);

    if (len == NULL)
        return NULL;
    else
        return *buf;
}
/* }}} */

/* {{{ proto int esmtp_message::set_messagecb(mixed cb [, mixed arg])
   Set a callback function to read an RFC 2822 formatted message from the application. */
PHP_METHOD(esmtp_message, set_messagecb)
{
    int argc = ZEND_NUM_ARGS();
    smtp_message_t message = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    ZVAL_ADDREF(cb);
    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        ZVAL_ADDREF(arg);
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }

    ret = smtp_set_messagecb(message, callback_esmtp_message, z_data);
 
    /* save z_data in global variable to free at exit */
    if (ESMTP_G(messagecb_data)) {
        zval_ptr_dtor(&ESMTP_G(messagecb_data));
        ESMTP_G(messagecb_data) = NULL;
    }
    ESMTP_G(messagecb_data) = z_data;

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_message::set_message_fp(resource fp)
   Callback function to read the message from a file. */
PHP_METHOD(esmtp_message, set_message_fp)
{
    int argc = ZEND_NUM_ARGS();
    int ret = 0;
    smtp_message_t message = NULL;
    zval *fp = NULL;
    php_stream *stream;

    if (zend_parse_parameters(argc TSRMLS_CC, "r", &fp) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    php_stream_from_zval(stream, &fp);
    if (php_stream_is(stream, PHP_STREAM_IS_STDIO)) {
        FILE *file;
        if (SUCCESS == php_stream_cast(stream, PHP_STREAM_AS_STDIO, (void**)&file, REPORT_ERRORS)) {
            ret = smtp_set_message_fp(message, file);
        }
    }
    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_message::set_message_str(string str)
   Callback function to read the message from a string. */
PHP_METHOD(esmtp_message, set_message_str)
{
    char *str = NULL;
    int argc = ZEND_NUM_ARGS();
    int str_len;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &str, &str_len) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_set_message_str(message, str));
}
/* }}} */

/* {{{ proto array esmtp_message::transfer_status(void)
   Retrieve the message transfer success/failure status. */
PHP_METHOD(esmtp_message, transfer_status)
{
    smtp_message_t message = NULL;
    const smtp_status_t *status;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    status = smtp_message_transfer_status(message);

    if (array_init(return_value) == FAILURE) {
        RETURN_FALSE;
    }

    add_assoc_long(return_value, "code", status->code);
    add_assoc_string(return_value, "text", status->text, 1);
    add_assoc_long(return_value, "enh_class", status->enh_class);
    add_assoc_long(return_value, "enh_subject", status->enh_subject);
    add_assoc_long(return_value, "enh_detail", status->enh_detail);
}
/* }}} */

/* {{{ proto array esmtp_message::reverse_path_status(void)
   Retrieve the reverse path status from a previous SMTP session. */
PHP_METHOD(esmtp_message, reverse_path_status)
{
    smtp_message_t message = NULL;
    const smtp_status_t *status;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    status = smtp_reverse_path_status(message);

    if (array_init(return_value) == FAILURE) {
        RETURN_FALSE;
    }

    add_assoc_long(return_value, "code", status->code);
    add_assoc_string(return_value, "text", status->text, 1);
    add_assoc_long(return_value, "enh_class", status->enh_class);
    add_assoc_long(return_value, "enh_subject", status->enh_subject);
    add_assoc_long(return_value, "enh_detail", status->enh_detail);
}
/* }}} */

/* {{{ proto int esmtp_message::message_reset_status(void)
   Reset the message status to the state it would have before start_session() */
PHP_METHOD(esmtp_message, reset_status)
{
    smtp_message_t message = NULL;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_message_reset_status(message));
}
/* }}} */

/* {{{ proto int esmtp_message::deliverby_set_mode(long time, int mode, int trace)
   Set the DELIVERBY parameters for the message. */
PHP_METHOD(esmtp_message, deliverby_set_mode)
{
    int argc = ZEND_NUM_ARGS();
    long time;
    long mode;
    long trace;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "lll", &time, &mode, &trace) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_deliverby_set_mode(message, time, mode, trace));
}
/* }}} */

/* {{{ proto int esmtp_message::dsn_set_ret(int flags)
   Instruct the reporting MTA whether to include the full content of the original message in the Delivery Status Notification, or just the headers. */
PHP_METHOD(esmtp_message, dsn_set_ret)
{
    int argc = ZEND_NUM_ARGS();
    long flags;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &flags) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_dsn_set_ret(message, flags));
}
/* }}} */

/* {{{ proto int esmtp_message::dsn_set_envid(string envid)
   Set the envelope identifier. */
PHP_METHOD(esmtp_message, dsn_set_envid)
{
    char *envid = NULL;
    int argc = ZEND_NUM_ARGS();
    int envid_len;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &envid, &envid_len) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_dsn_set_envid(message, envid));
}
/* }}} */

/* {{{ proto int esmtp_message::size_set_estimate(long size)
   Used by the application to supply an estimate of the size of the message to be transferred. */
PHP_METHOD(esmtp_message, size_set_estimate)
{
    int argc = ZEND_NUM_ARGS();
    long size;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &size) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_size_set_estimate(message, size));
}
/* }}} */

/* {{{ proto int esmtp_message::8bitmime_set_body(int body)
   Sets the 8-bit MIME extension mode. */
PHP_METHOD(esmtp_message, 8bitmime_set_body)
{
    int argc = ZEND_NUM_ARGS();
    long body;
    smtp_message_t message = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &body) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_8bitmime_set_body(message, body));
}
/* }}} */

/* {{{ proto array esmtp_recipient::status(void)
   Retrieve the recipient success/failure status from a previous SMTP session. */
PHP_METHOD(esmtp_recipient, status)
{
    smtp_recipient_t recipient = NULL;
    const smtp_status_t *status;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    status = smtp_recipient_status(recipient);

    if (array_init(return_value) == FAILURE) {
        RETURN_FALSE;
    }

    add_assoc_long(return_value, "code", status->code);
    add_assoc_string(return_value, "text", status->text, 1);
    add_assoc_long(return_value, "enh_class", status->enh_class);
    add_assoc_long(return_value, "enh_subject", status->enh_subject);
    add_assoc_long(return_value, "enh_detail", status->enh_detail);
}
/* }}} */

/* {{{ proto int esmtp_recipient::check_complete(void)
   Check whether processing is complete for the specified recipient of the message. */
PHP_METHOD(esmtp_recipient, check_complete)
{
    smtp_recipient_t recipient = NULL;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_recipient_check_complete(recipient));
}
/* }}} */

/* {{{ proto int esmtp_recipient::reset_status(void)
   Reset the recipient status to the state it would have before start_session(). */
PHP_METHOD(esmtp_recipient, reset_status)
{
    smtp_recipient_t recipient = NULL;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_recipient_reset_status(recipient));
}
/* }}} */

/* {{{ proto int esmtp_recipient::dsn_set_notify(int flags)
   Set the DSN notify options. */
PHP_METHOD(esmtp_recipient, dsn_set_notify)
{
    int argc = ZEND_NUM_ARGS();
    long flags;
    smtp_recipient_t recipient = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &flags) == FAILURE) 
        return;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_dsn_set_notify(recipient, flags));
}
/* }}} */

/* {{{ proto int esmtp_recipient::dsn_set_orcpt(string address_type, string address)
   Set the DSN ORCPT option. */
PHP_METHOD(esmtp_recipient, dsn_set_orcpt)
{
    char *address_type = NULL;
    char *address = NULL;
    int argc = ZEND_NUM_ARGS();
    int address_type_len;
    int address_len;
    smtp_recipient_t recipient = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "ss", &address_type, &address_type_len, &address, &address_len) == FAILURE) 
        return;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    RETURN_LONG(smtp_dsn_set_orcpt(recipient, address_type, address));
}
/* }}} */

/* {{{ smtp_etrn_enumerate_nodecb_t */
static void
callback_esmtp_enumerate_nodes(smtp_etrn_node_t node, int option, const char *domain, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[4];
    int param_len = 1, rnum;
    TSRMLS_FETCH();

    if (!arg)
        return;

    MAKE_STD_ZVAL(params[0]);

    rnum = zend_list_insert(node, le_esmtp_etrnnode);
    object_init_ex(params[0], esmtp_etrnnode_class_entry);
    add_property_resource(params[0], "Esmtp_Etrnnode", rnum);

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (enumerate_etrnnode)");
        zval_ptr_dtor(&params[0]);
        return;
    }

    MAKE_STD_ZVAL(params[1]);
    ZVAL_LONG(params[1], option);
    ++param_len;

    MAKE_STD_ZVAL(params[2]);
    ZVAL_STRING(params[2], (char *)domain, 1);
    ++param_len;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[3] = *z_userdata;
    }
    /* would not check for return value */
    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (enumerate_etrnnode)");
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);
    zval_ptr_dtor(&params[1]);
    zval_ptr_dtor(&params[2]);
}
/* }}} */

/* {{{ proto int esmtp_session::etrn_enumerate_nodes(mixed cb [, mixed arg])
   Call the callback function once for each etrn node in the smtp session. */
PHP_METHOD(esmtp_session, etrn_enumerate_nodes)
{
    int argc = ZEND_NUM_ARGS();
    smtp_session_t session = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }
    
    ret = smtp_etrn_enumerate_nodes(session, callback_esmtp_enumerate_nodes, z_data);

    zval_ptr_dtor(&z_data);

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto array esmtp_etrnnode::status(void)
   Retrieve the ETRN node success/failure status from a previous SMTP session. */
PHP_METHOD(esmtp_etrnnode, status)
{
    smtp_etrn_node_t etrn_node = NULL;
    const smtp_status_t *status;

    etrn_node = _php_get_esmtp_etrnnode(getThis() TSRMLS_CC);

    status = smtp_etrn_node_status(etrn_node);

    if (array_init(return_value) == FAILURE) {
        RETURN_FALSE;
    }

    add_assoc_long(return_value, "code", status->code);
    add_assoc_string(return_value, "text", status->text, 1);
    add_assoc_long(return_value, "enh_class", status->enh_class);
    add_assoc_long(return_value, "enh_subject", status->enh_subject);
    add_assoc_long(return_value, "enh_detail", status->enh_detail);
}
/* }}} */

/* {{{ proto mixed esmtp_session::set_application_data(mixed data)
   Sets application data. */
PHP_METHOD(esmtp_session, set_application_data)
{
    int argc = ZEND_NUM_ARGS();
    smtp_session_t session = NULL;
    char *p, *cdata = NULL;
    zval *data;

    if (zend_parse_parameters(argc TSRMLS_CC, "z", &data) == FAILURE) 
        return;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

	if (do_serialize) {
		smart_str app_var = {0};
		php_serialize_data_t var_hash;

		PHP_VAR_SERIALIZE_INIT(var_hash);
		php_var_serialize(&app_var, &data, &var_hash TSRMLS_CC);
		PHP_VAR_SERIALIZE_DESTROY(var_hash);
		
        cdata = (char *)estrndup(app_var.c, app_var.len + 1);
		smart_str_free(&app_var);
	} else {
		convert_to_string_ex(&data);
        cdata = (char *)estrndup(Z_STRVAL_P(data), Z_STRLEN_P(data) + 1);
	}

    p = (char *)smtp_set_application_data(session, cdata);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_session::get_application_data(void)
   Gets application data. */
PHP_METHOD(esmtp_session, get_application_data)
{
    smtp_session_t session = NULL;
    char *p;

    session = _php_get_esmtp_session(getThis() TSRMLS_CC);

    p = (char *)smtp_get_application_data(session);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_message::set_application_data(mixed data)
   Sets application data. */
PHP_METHOD(esmtp_message, set_application_data)
{
    int argc = ZEND_NUM_ARGS();
    smtp_message_t message = NULL;
    char *p, *cdata = NULL;
    zval *data = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "z", &data) == FAILURE) 
        return;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);
    
	if (do_serialize) {
		smart_str app_var = {0};
		php_serialize_data_t var_hash;

		PHP_VAR_SERIALIZE_INIT(var_hash);
		php_var_serialize(&app_var, &data, &var_hash TSRMLS_CC);
		PHP_VAR_SERIALIZE_DESTROY(var_hash);
		
        cdata = (char *)estrndup(app_var.c, app_var.len + 1);
		smart_str_free(&app_var);
	} else {
		convert_to_string_ex(&data);
        cdata = (char *)estrndup(Z_STRVAL_P(data), Z_STRLEN_P(data) + 1);
	}

    p = (char *)smtp_message_set_application_data(message, cdata);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_message::get_application_data(void)
   Gets application data. */
PHP_METHOD(esmtp_message, get_application_data)
{
    smtp_message_t message = NULL;
    char *p;

    message = _php_get_esmtp_message(getThis() TSRMLS_CC);

    p = (char *)smtp_message_get_application_data(message);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_recipient::set_application_data(mixed data)
   Sets application data. */
PHP_METHOD(esmtp_recipient, set_application_data)
{
    int argc = ZEND_NUM_ARGS();
    smtp_recipient_t recipient = NULL;
    char *p, *cdata = NULL;
    zval *data = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "z", &data) == FAILURE) 
        return;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

	if (do_serialize) {
		smart_str app_var = {0};
		php_serialize_data_t var_hash;

		PHP_VAR_SERIALIZE_INIT(var_hash);
		php_var_serialize(&app_var, &data, &var_hash TSRMLS_CC);
		PHP_VAR_SERIALIZE_DESTROY(var_hash);
		
        cdata = (char *)estrndup(app_var.c, app_var.len + 1);
		smart_str_free(&app_var);
	} else {
		convert_to_string_ex(&data);
        cdata = (char *)estrndup(Z_STRVAL_P(data), Z_STRLEN_P(data) + 1);
	}

    p = (char *)smtp_recipient_set_application_data(recipient, cdata);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_recipient::get_application_data(void)
   Gets application data. */
PHP_METHOD(esmtp_recipient, get_application_data)
{
    smtp_recipient_t recipient = NULL;
    char *p;

    recipient = _php_get_esmtp_recipient(getThis() TSRMLS_CC);

    p = (char *)smtp_recipient_get_application_data(recipient);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_etrnnode::set_application_data(mixed data)
   Sets application data. */
PHP_METHOD(esmtp_etrnnode, set_application_data)
{
    int argc = ZEND_NUM_ARGS();
    zval *data = NULL;
    char *p, *cdata = NULL;
    smtp_etrn_node_t etrn_node = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "z", &data) == FAILURE) 
        return;

    etrn_node = _php_get_esmtp_etrnnode(getThis() TSRMLS_CC);

	if (do_serialize) {
		smart_str app_var = {0};
		php_serialize_data_t var_hash;

		PHP_VAR_SERIALIZE_INIT(var_hash);
		php_var_serialize(&app_var, &data, &var_hash TSRMLS_CC);
		PHP_VAR_SERIALIZE_DESTROY(var_hash);
		
        cdata = (char *)estrndup(app_var.c, app_var.len + 1);
		smart_str_free(&app_var);
	} else {
		convert_to_string_ex(&data);
        cdata = (char *)estrndup(Z_STRVAL_P(data), Z_STRLEN_P(data) + 1);
	}

    p = (char *)smtp_etrn_set_application_data(etrn_node, cdata);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
        efree(p);
    } else {
        ZVAL_STRING(return_value, p, 1);
        efree(p);
    }
}
/* }}} */

/* {{{ proto mixed esmtp_etrnnode::get_application_data(void)
   Gets application data. */
PHP_METHOD(esmtp_etrnnode, get_application_data)
{
    smtp_etrn_node_t etrn_node = NULL;
    char *p;

    etrn_node = _php_get_esmtp_etrnnode(getThis() TSRMLS_CC);

    p = (char *)smtp_etrn_get_application_data(etrn_node);

    if (!p) {
        RETURN_FALSE;
    }

    if (do_serialize) {
        php_unserialize_data_t var_hash;
        zval *tmp = NULL;
        const char *q = p;

        MAKE_STD_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_INIT(var_hash);
        if (!php_var_unserialize(&tmp, &q, p + strlen(p), &var_hash TSRMLS_CC)) {
            php_error_docref(NULL TSRMLS_CC, E_WARNING, "application data corrupted");
            RETVAL_FALSE;
        }
        REPLACE_ZVAL_VALUE(&return_value, tmp, 0);
        FREE_ZVAL(tmp);
        PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
    } else {
        ZVAL_STRING(return_value, p, 1);
    }
}
/* }}} */

/* {{{ proto object esmtp_auth::__construct()
   Creates new Auth object. */
PHP_METHOD(esmtp_auth, __construct)
{
    auth_context_t context = NULL;
    
    if (ZEND_NUM_ARGS() != 0) {
        WRONG_PARAM_COUNT;
    }

    context = auth_create_context();

    if (context) {
        int ret = zend_list_insert(context, le_esmtp_auth);
        object_init_ex(getThis(), esmtp_auth_class_entry);
        add_property_resource(getThis(), "Esmtp_Auth", ret);
        zend_list_addref(ret);
    } else {
        RETURN_NULL();
    }
}
/* }}} */

/* {{{ proto int esmtp_auth::set_mechanism_flags(int set, int clear)
   Sets a plugin authentication flags. */
PHP_METHOD(esmtp_auth, set_mechanism_flags)
{
    int argc = ZEND_NUM_ARGS();
    long set;
    long clear;
    auth_context_t context = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "ll", &set, &clear) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_set_mechanism_flags(context, set, clear));
}
/* }}} */

/* {{{ proto int esmtp_auth::set_mechanism_ssf(int min_ssf)
   Sets a minimal security strength. */
PHP_METHOD(esmtp_auth, set_mechanism_ssf)
{
    int argc = ZEND_NUM_ARGS();
    long min_ssf;
    auth_context_t context = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "l", &min_ssf) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_set_mechanism_ssf(context, min_ssf));
}
/* }}} */

/* {{{ proto int esmtp_auth::get_ssf(void)
   Gets a current security strength for the client connection. */
PHP_METHOD(esmtp_auth, get_ssf)
{
    auth_context_t context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_get_ssf(context));
}
/* }}} */

/* {{{ auth_interact_t */
static int
callback_esmtp_auth_interact(auth_client_request_t request, char **result, int fields, void *arg)
{
    zval *z_data, **z_funcname, **z_userdata = NULL;
    zval *retval, *params[2];
    int param_len = 1, ret = 0, i;
    TSRMLS_FETCH();

    if (!arg)
        return 0;

    z_data = (zval *) arg;

    if (zend_hash_find(Z_ARRVAL_P(z_data), "funcname", sizeof("funcname"), (void **)&z_funcname) == FAILURE) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to find callback function name (enumerate_recipient)");
        return 0;
    }

    if (zend_hash_find(Z_ARRVAL_P(z_data), "userdata", sizeof("userdata"), (void **)&z_userdata) == SUCCESS && z_userdata) {
        ++param_len;
        params[1] = *z_userdata;
    }

    /* first argument is an array of requests */
    MAKE_STD_ZVAL(params[0]);
    array_init(params[0]);

    for (i=0; i<fields; ++i) {
        zval *z_request;

        MAKE_STD_ZVAL(z_request);
        array_init(z_request);

        add_assoc_string(z_request, "name", (char *)request[i].name, 1);
        add_assoc_long(z_request, "flags", request[i].flags);
        add_assoc_string(z_request, "prompt", (char *)request[i].prompt, 1);
        add_assoc_long(z_request, "size", request[i].size);
        add_next_index_zval(params[0], z_request);
    }

    MAKE_STD_ZVAL(retval);
    if (call_user_function(EG(function_table), NULL, *z_funcname, retval, param_len, params TSRMLS_CC) != SUCCESS) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Unable to call callback function (enumerate_recipient)");
    } else {
        if (Z_TYPE_P(retval)==IS_ARRAY) {
            if (zend_hash_num_elements(Z_ARRVAL_P(retval)) == fields) {
                zend_hash_internal_pointer_reset(Z_ARRVAL_P(retval));
                for (i=0; i<fields; ++i) {
                    zval *z_res;

                    zend_hash_get_current_data(Z_ARRVAL_P(retval), (void *) &z_res);
                    /* FIXME: possible memory leak ... but have no other method to do it */
                    result[i] = (char *)estrndup(Z_STRVAL_P(z_res), Z_STRLEN_P(z_res));
                    zend_hash_move_forward(Z_ARRVAL_P(retval));
                }
            }
        } else if (fields == 1) {
        }
    }
    
    zval_ptr_dtor(&retval);
    zval_ptr_dtor(&params[0]);

    return ret;
}
/* }}} */

/* {{{ proto int esmtp_auth::set_interact_cb(mixed interact [, mixed arg])
   Call the callback function for collecting responses to plugin questions. */
PHP_METHOD(esmtp_auth, set_interact_cb)
{
    int argc = ZEND_NUM_ARGS();
    auth_context_t context = NULL;
    zval *cb = NULL;
    zval *arg = NULL;
    zval *z_data = NULL;
    int ret = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "z|z", &cb, &arg) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    MAKE_STD_ZVAL(z_data);
    array_init(z_data);

    add_assoc_zval_ex(z_data, "funcname", sizeof("funcname"), cb);
    if (arg) {
        add_assoc_zval_ex(z_data, "userdata", sizeof("userdata"), arg);
    }
    
    ret = auth_set_interact_cb(context, callback_esmtp_auth_interact, z_data);

    if (ESMTP_G(authinteractcb_data)) {
        zval_ptr_dtor(&ESMTP_G(authinteractcb_data));
        ESMTP_G(authinteractcb_data) = NULL;
    }
    ESMTP_G(authinteractcb_data) = z_data;

    RETURN_LONG(ret);
}
/* }}} */

/* {{{ proto int esmtp_auth::client_enabled(void)
   Perform various checks to see if SASL is usable. */
PHP_METHOD(esmtp_auth, client_enabled)
{
    auth_context_t context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_client_enabled(context));
}
/* }}} */

/* {{{ proto int esmtp_auth::set_mechanism(string name)
   Load a plugin with a mechanism keyword specified in name. */
PHP_METHOD(esmtp_auth, set_mechanism)
{
    char *name = NULL;
    int argc = ZEND_NUM_ARGS();
    int name_len;
    auth_context_t context = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &name, &name_len) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_set_mechanism(context, name));
}
/* }}} */

/* {{{ proto string auth_mechanism_name(void)
   Gets a mechanism keyword from the SASL plugin information. */
PHP_METHOD(esmtp_auth, mechanism_name)
{
    auth_context_t context = _php_get_esmtp_auth(getThis() TSRMLS_CC);
    const char *text = NULL;

    text = auth_mechanism_name(context);

    if (text != NULL) {
        RETURN_STRING((char *)text, 1);
    } else {
        RETURN_FALSE;
    }
}
/* }}} */

/* {{{ proto string auth_response(string challenge)
   Gets a reponse string for the current state and plugin. */
PHP_METHOD(esmtp_auth, response)
{
    char *challenge = NULL;
    int argc = ZEND_NUM_ARGS();
    int challenge_len;
    auth_context_t context = NULL;
    const char *text = NULL;
    int text_len = 0;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &challenge, &challenge_len) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    text = auth_response(context, challenge, &text_len);

    if (text != NULL) {
        RETURN_STRINGL((char *)text, text_len, 1);
    } else {
        RETURN_FALSE;
    }
}
/* }}} */

/* {{{ proto int esmtp_auth::set_external_id(string identity)
   Sets SASL external id as defined in SASL EXTERNAL mechanism (RFC 2222). */
PHP_METHOD(esmtp_auth, set_external_id)
{
    char *identity = NULL;
    int argc = ZEND_NUM_ARGS();
    int identity_len;
    auth_context_t context = NULL;

    if (zend_parse_parameters(argc TSRMLS_CC, "s", &identity, &identity_len) == FAILURE) 
        return;

    context = _php_get_esmtp_auth(getThis() TSRMLS_CC);

    RETURN_LONG(auth_set_external_id(context, identity));
}
/* }}} */

/* {{{ esmtp_session_functions
   */
static zend_function_entry esmtp_session_functions[] = {
    PHP_ME(esmtp_session, __construct,                NULL, 0)
    PHP_ME(esmtp_session, add_message,                NULL, 0)
    PHP_ME(esmtp_session, enumerate_messages,         NULL, 0)
    PHP_ME(esmtp_session, set_hostname,               NULL, 0)
    PHP_ME(esmtp_session, set_server,                 NULL, 0)
    PHP_ME(esmtp_session, set_eventcb,                NULL, 0)
    PHP_ME(esmtp_session, set_monitorcb,              NULL, 0)
    PHP_ME(esmtp_session, start_session,              NULL, 0)
    PHP_ME(esmtp_session, set_application_data,       NULL, 0)
    PHP_ME(esmtp_session, get_application_data,       NULL, 0)
    PHP_ME(esmtp_session, option_require_all_recipients,  NULL, 0)
    PHP_ME(esmtp_session, auth_set_context,           NULL, 0)
    PHP_ME(esmtp_session, set_timeout,                NULL, 0)
    PHP_ME(esmtp_session, starttls_enable,            NULL, 0)
    PHP_ME(esmtp_session, etrn_add_node,              NULL, 0)
    PHP_ME(esmtp_session, etrn_enumerate_nodes,       NULL, 0)
    {NULL, NULL, NULL, 0}
};
/* }}} */

/* {{{ esmtp_message_functions
   */
static zend_function_entry esmtp_message_functions[] = {
    PHP_ME(esmtp_message, set_reverse_path,           NULL, 0)
    PHP_ME(esmtp_message, add_recipient,              NULL, 0)
    PHP_ME(esmtp_message, enumerate_recipients,       NULL, 0)
    PHP_ME(esmtp_message, set_header,                 NULL, 0)
    PHP_ME(esmtp_message, set_header_option,          NULL, 0)
    PHP_ME(esmtp_message, set_resent_headers,         NULL, 0)
    PHP_ME(esmtp_message, set_messagecb,              NULL, 0)
    PHP_ME(esmtp_message, set_message_fp,             NULL, 0)
    PHP_ME(esmtp_message, set_message_str,            NULL, 0)
    PHP_ME(esmtp_message, transfer_status,            NULL, 0)
    PHP_ME(esmtp_message, reverse_path_status,        NULL, 0)
    PHP_ME(esmtp_message, reset_status,               NULL, 0)
    PHP_ME(esmtp_message, set_application_data,       NULL, 0)
    PHP_ME(esmtp_message, get_application_data,       NULL, 0)
    PHP_ME(esmtp_message, dsn_set_ret,                NULL, 0)
    PHP_ME(esmtp_message, dsn_set_envid,              NULL, 0)
    PHP_ME(esmtp_message, size_set_estimate,          NULL, 0)
    PHP_ME(esmtp_message, 8bitmime_set_body,          NULL, 0)
    PHP_ME(esmtp_message, deliverby_set_mode,         NULL, 0)
    {NULL, NULL, NULL, 0}
};
/* }}} */

/* {{{ esmtp_recipient_functions
   */
static zend_function_entry esmtp_recipient_functions[] = {
    PHP_ME(esmtp_recipient, status,                NULL, 0)
    PHP_ME(esmtp_recipient, check_complete,        NULL, 0)
    PHP_ME(esmtp_recipient, reset_status,          NULL, 0)
    PHP_ME(esmtp_recipient, set_application_data,  NULL, 0)
    PHP_ME(esmtp_recipient, get_application_data,  NULL, 0)
    PHP_ME(esmtp_recipient, dsn_set_notify,        NULL, 0)
    PHP_ME(esmtp_recipient, dsn_set_orcpt,         NULL, 0)
    {NULL, NULL, NULL, 0}
};
/* }}} */

/* {{{ esmtp_etrnnode_functions
   */
static zend_function_entry esmtp_etrnnode_functions[] = {
    PHP_ME(esmtp_etrnnode, status,                NULL, 0)
    PHP_ME(esmtp_etrnnode, set_application_data,  NULL, 0)
    PHP_ME(esmtp_etrnnode, get_application_data,  NULL, 0)
    {NULL, NULL, NULL, 0}
};
/* }}} */

/* {{{ esmtp_auth_functions
   */
static zend_function_entry esmtp_auth_functions[] = {
    PHP_ME(esmtp_auth, __construct,             NULL, 0)
    PHP_ME(esmtp_auth, set_mechanism_flags,     NULL, 0)
    PHP_ME(esmtp_auth, set_mechanism_ssf,       NULL, 0)
    PHP_ME(esmtp_auth, get_ssf,                 NULL, 0)
    PHP_ME(esmtp_auth, set_interact_cb,         NULL, 0)
    PHP_ME(esmtp_auth, client_enabled,          NULL, 0)
    PHP_ME(esmtp_auth, set_mechanism,           NULL, 0)
    PHP_ME(esmtp_auth, mechanism_name,          NULL, 0)
    PHP_ME(esmtp_auth, response,                NULL, 0)
    PHP_ME(esmtp_auth, set_external_id,         NULL, 0)
    {NULL, NULL, NULL, 0}
};
/* }}} */


PHP_MINIT_FUNCTION(esmtp)
{
    zend_class_entry ce;

#ifdef _auth_client_h
    auth_client_init();
#endif

    le_esmtp_session   = zend_register_list_destructors_ex(_php_free_esmtp_object, NULL, "Esmtp_Session", module_number);
    le_esmtp_message   = zend_register_list_destructors_ex(_php_free_esmtp_object, NULL, "Esmtp_Message", module_number);
    le_esmtp_recipient = zend_register_list_destructors_ex(_php_free_esmtp_object, NULL, "Esmtp_Recipient", module_number);
    le_esmtp_etrnnode  = zend_register_list_destructors_ex(_php_free_esmtp_object, NULL, "Esmtp_Etrnnode", module_number);
    le_esmtp_auth      = zend_register_list_destructors_ex(_php_free_esmtp_object, NULL, "Esmtp_Auth", module_number);

    INIT_OVERLOADED_CLASS_ENTRY(ce, "Esmtp_Session",      esmtp_session_functions, NULL, NULL, NULL);
    esmtp_session_class_entry = zend_register_internal_class_ex(&ce, NULL, NULL TSRMLS_CC);

    INIT_OVERLOADED_CLASS_ENTRY(ce, "Esmtp_Message",      esmtp_message_functions, NULL, NULL, NULL);
    esmtp_message_class_entry = zend_register_internal_class_ex(&ce, NULL, NULL TSRMLS_CC);

    INIT_OVERLOADED_CLASS_ENTRY(ce, "Esmtp_Recipient",    esmtp_recipient_functions, NULL, NULL, NULL);
    esmtp_recipient_class_entry = zend_register_internal_class_ex(&ce, NULL, NULL TSRMLS_CC);

    INIT_OVERLOADED_CLASS_ENTRY(ce, "Esmtp_Etrnnode",     esmtp_etrnnode_functions, NULL, NULL, NULL);
    esmtp_etrnnode_class_entry = zend_register_internal_class_ex(&ce, NULL, NULL TSRMLS_CC);

    INIT_OVERLOADED_CLASS_ENTRY(ce, "Esmtp_Auth",         esmtp_auth_functions, NULL, NULL, NULL);
    esmtp_auth_class_entry = zend_register_internal_class_ex(&ce, NULL, NULL TSRMLS_CC);


    ZEND_INIT_MODULE_GLOBALS(esmtp, php_esmtp_init_globals, NULL);

#ifdef _auth_client_h
    REGISTER_LONG_CONSTANT("AUTH_USER", AUTH_USER, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("AUTH_REALM", AUTH_REALM, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("AUTH_PASS", AUTH_PASS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("AUTH_CLEARTEXT", AUTH_CLEARTEXT, CONST_CS | CONST_PERSISTENT);
  
    /* For $auth->set_mechanism_flags() */
    REGISTER_LONG_CONSTANT("AUTH_PLUGIN_ANONYMOUS", AUTH_PLUGIN_ANONYMOUS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("AUTH_PLUGIN_PLAIN", AUTH_PLUGIN_PLAIN, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("AUTH_PLUGIN_EXTERNAL", AUTH_PLUGIN_EXTERNAL, CONST_CS | CONST_PERSISTENT);
#endif

    /* For the $mode in $message->deliverby_set_mode(): */
    REGISTER_LONG_CONSTANT("BY_NOTSET", By_NOTSET, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("BY_NOTIFY", By_NOTIFY, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("BY_RETURN", By_RETURN, CONST_CS | CONST_PERSISTENT);

    /* For the $body in $message->8bitmime_set_body(): */
    REGISTER_LONG_CONSTANT("E8BITMIME_NOTSET", E8bitmime_NOTSET, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("E8BITMIME_7BIT", E8bitmime_7BIT, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("E8BITMIME_8BITMIME", E8bitmime_8BITMIME, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("E8BITMIME_BINARYMIME", E8bitmime_BINARYMIME, CONST_CS | CONST_PERSISTENT);

    /* For the $header_option in $session->set_header_option(): */
    REGISTER_LONG_CONSTANT("HDR_OVERRIDE", Hdr_OVERRIDE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("HDR_PROHIBIT", Hdr_PROHIBIT, CONST_CS | CONST_PERSISTENT);

    /* For the $flags in $recipient->dsn_set_notify(): */
    REGISTER_LONG_CONSTANT("NOTIFY_NOTSET", Notify_NOTSET, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("NOTIFY_NEVER", Notify_NEVER, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("NOTIFY_SUCCESS", Notify_SUCCESS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("NOTIFY_FAILURE", Notify_FAILURE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("NOTIFY_DELAY", Notify_DELAY, CONST_CS | CONST_PERSISTENT);

    /* For the $flags in $message->dsn_set_ret() */

    REGISTER_LONG_CONSTANT("RET_NOTSET", Ret_NOTSET, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("RET_FULL", Ret_FULL, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("RET_HDRS", Ret_HDRS, CONST_CS | CONST_PERSISTENT);

    /* For the $event_no in the callback in $session->set_eventcb(): */
    REGISTER_LONG_CONSTANT("SMTP_EV_CONNECT", SMTP_EV_CONNECT, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_MAILSTATUS", SMTP_EV_MAILSTATUS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_RCPTSTATUS", SMTP_EV_RCPTSTATUS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_MESSAGEDATA", SMTP_EV_MESSAGEDATA, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_MESSAGESENT", SMTP_EV_MESSAGESENT, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_DISCONNECT", SMTP_EV_DISCONNECT, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_ETRNSTATUS", SMTP_EV_ETRNSTATUS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_DSN", SMTP_EV_EXTNA_DSN, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_8BITMIME", SMTP_EV_EXTNA_8BITMIME, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_STARTTLS", SMTP_EV_EXTNA_STARTTLS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_ETRN", SMTP_EV_EXTNA_ETRN, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_CHUNKING", SMTP_EV_EXTNA_CHUNKING, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_EXTNA_BINARYMIME", SMTP_EV_EXTNA_BINARYMIME, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_DELIVERBY_EXPIRED", SMTP_EV_DELIVERBY_EXPIRED, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_WEAK_CIPHER", SMTP_EV_WEAK_CIPHER, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_STARTTLS_OK", SMTP_EV_STARTTLS_OK, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_INVALID_PEER_CERTIFICATE", SMTP_EV_INVALID_PEER_CERTIFICATE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_NO_PEER_CERTIFICATE", SMTP_EV_NO_PEER_CERTIFICATE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_WRONG_PEER_CERTIFICATE", SMTP_EV_WRONG_PEER_CERTIFICATE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_NO_CLIENT_CERTIFICATE", SMTP_EV_NO_CLIENT_CERTIFICATE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_UNUSABLE_CLIENT_CERTIFICATE", SMTP_EV_UNUSABLE_CLIENT_CERTIFICATE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_EV_UNUSABLE_CA_LIST", SMTP_EV_UNUSABLE_CA_LIST, CONST_CS | CONST_PERSISTENT);

    /* For the $writing in the callback in $session->set_monitorcb(): */
    REGISTER_LONG_CONSTANT("SMTP_CB_HEADERS", SMTP_CB_HEADERS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_CB_READING", SMTP_CB_READING, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_CB_WRITING", SMTP_CB_WRITING, CONST_CS | CONST_PERSISTENT);

    /* For the smtp_errno() and smtp_strerror(): */
    REGISTER_LONG_CONSTANT("SMTP_ERR_DROPPED_CONNECTION", SMTP_ERR_DROPPED_CONNECTION, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_ADDRFAMILY", SMTP_ERR_EAI_ADDRFAMILY, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_AGAIN", SMTP_ERR_EAI_AGAIN, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_BADFLAGS", SMTP_ERR_EAI_BADFLAGS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_FAIL", SMTP_ERR_EAI_FAIL, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_FAMILY", SMTP_ERR_EAI_FAMILY, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_MEMORY", SMTP_ERR_EAI_MEMORY, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_NODATA", SMTP_ERR_EAI_NODATA, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_NONAME", SMTP_ERR_EAI_NONAME, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_SERVICE", SMTP_ERR_EAI_SERVICE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EAI_SOCKTYPE", SMTP_ERR_EAI_SOCKTYPE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_EXTENSION_NOT_AVAILABLE", SMTP_ERR_EXTENSION_NOT_AVAILABLE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_HOST_NOT_FOUND", SMTP_ERR_HOST_NOT_FOUND, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_INVAL", SMTP_ERR_INVAL, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_INVALID_RESPONSE_STATUS", SMTP_ERR_INVALID_RESPONSE_STATUS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_INVALID_RESPONSE_SYNTAX", SMTP_ERR_INVALID_RESPONSE_SYNTAX, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_NOTHING_TO_DO", SMTP_ERR_NOTHING_TO_DO, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_NO_ADDRESS", SMTP_ERR_NO_ADDRESS, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_NO_RECOVERY", SMTP_ERR_NO_RECOVERY, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_STATUS_MISMATCH", SMTP_ERR_STATUS_MISMATCH, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_TRY_AGAIN", SMTP_ERR_TRY_AGAIN, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("SMTP_ERR_UNTERMINATED_RESPONSE", SMTP_ERR_UNTERMINATED_RESPONSE, CONST_CS | CONST_PERSISTENT);

    /* For $how in $session->starttls_enable(): */
    REGISTER_LONG_CONSTANT("STARTTLS_DISABLED", Starttls_DISABLED, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("STARTTLS_ENABLED", Starttls_ENABLED, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("STARTTLS_REQUIRED", Starttls_REQUIRED, CONST_CS | CONST_PERSISTENT);

    /* For the $which in $session->set_timeout(): */
    REGISTER_LONG_CONSTANT("TIMEOUT_GREETING", Timeout_GREETING, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("TIMEOUT_ENVELOPE", Timeout_ENVELOPE, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("TIMEOUT_DATA", Timeout_DATA, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("TIMEOUT_TRANSFER", Timeout_TRANSFER, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("TIMEOUT_DATA2", Timeout_DATA2, CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("TIMEOUT_OVERRIDE_RFC2822_MINIMUM", Timeout_OVERRIDE_RFC2822_MINIMUM, CONST_CS | CONST_PERSISTENT);
    
    return SUCCESS;
}


PHP_MSHUTDOWN_FUNCTION(esmtp)
{
    if (ESMTP_G(starttls_data)) {
        zval_ptr_dtor(&ESMTP_G(starttls_data));
        ESMTP_G(starttls_data) = NULL;
    }
    if (ESMTP_G(eventcb_data)) {
        zval_ptr_dtor(&ESMTP_G(eventcb_data));
        ESMTP_G(eventcb_data) = NULL;
    }
    if (ESMTP_G(messagecb_data)) {
        zval_ptr_dtor(&ESMTP_G(messagecb_data));
        ESMTP_G(messagecb_data) = NULL;
    }
    if (ESMTP_G(authinteractcb_data)) {
        zval_ptr_dtor(&ESMTP_G(authinteractcb_data));
        ESMTP_G(authinteractcb_data) = NULL;
    }

    return SUCCESS;
}



PHP_RINIT_FUNCTION(esmtp)
{
    return SUCCESS;
}



PHP_RSHUTDOWN_FUNCTION(esmtp)
{
    return SUCCESS;
}


PHP_MINFO_FUNCTION(esmtp)
{
    php_info_print_table_start();
    php_info_print_table_header(2, "esmtp support", "enabled");
    php_info_print_table_end();

}



#endif /* HAVE_LIBESMTP */

