/*
  +----------------------------------------------------------------------+
  | PHP Version 7                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 2006-2018 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Andrey Hristov <andrey@php.net>                             |
  +----------------------------------------------------------------------+
*/
#include "php_api.h"
#include "mysqlnd_api.h"
extern "C" {
#include <ext/json/php_json_parser.h>
}
#include "xmysqlnd.h"
#include "xmysqlnd_driver.h"
#include "xmysqlnd_session.h"
#include "xmysqlnd_schema.h"
#include "xmysqlnd_collection.h"
#include "xmysqlnd_stmt.h"
#include "xmysqlnd_stmt_result_meta.h"
#include "xmysqlnd_utils.h"
#include "mysqlx_exception.h"
#include "util/exceptions.h"
#include "xmysqlnd_extension_plugin.h"

namespace mysqlx {

namespace drv {

/* {{{ xmysqlnd_collection::xmysqlnd_collection */
xmysqlnd_collection::xmysqlnd_collection(
								xmysqlnd_schema * const cur_schema,
								const MYSQLND_CSTRING cur_collection_name,
								zend_bool is_persistent)
{
	DBG_ENTER("xmysqlnd_collection::xmysqlnd_collection");
	if (!(schema = cur_schema->get_reference())) {
		throw util::xdevapi_exception(util::xdevapi_exception::Code::schema_creation_failed);
	}
	persistent = is_persistent;
	collection_name = mnd_dup_cstring(cur_collection_name, persistent);
	DBG_INF_FMT("name=[%d]%*s", collection_name.l, collection_name.l, collection_name.s);

}
/* }}} */

struct st_collection_exists_in_database_var_binder_ctx
{
	const MYSQLND_CSTRING schema_name;
	const MYSQLND_CSTRING collection_name;
	unsigned int counter;
};


/* {{{ collection_op_var_binder */
static const enum_hnd_func_status
collection_op_var_binder(
	void * context,
	XMYSQLND_SESSION session,
	XMYSQLND_STMT_OP__EXECUTE * const stmt_execute)
{
	enum_hnd_func_status ret{HND_FAIL};
	st_collection_exists_in_database_var_binder_ctx* ctx = (st_collection_exists_in_database_var_binder_ctx*) context;
	const MYSQLND_CSTRING* param{nullptr};
	DBG_ENTER("collection_op_var_binder");
	switch (ctx->counter) {
		case 0:
			param = &ctx->schema_name;
			ret = HND_AGAIN;
			goto bind;
		case 1:{
			param = &ctx->collection_name;
			ret = HND_PASS;
bind:
			{
				enum_func_status result;
				zval zv;
				ZVAL_UNDEF(&zv);
				ZVAL_STRINGL(&zv, param->s, param->l);
				DBG_INF_FMT("[%d]=[%*s]", ctx->counter, param->l, param->s);
				result = xmysqlnd_stmt_execute__bind_one_param(stmt_execute, ctx->counter, &zv);

				zval_ptr_dtor(&zv);
				if (FAIL == result) {
					ret = HND_FAIL;
				}
			}
			break;
		}
		default:
			assert(!"should not happen");
			break;
	}
	++ctx->counter;
	DBG_RETURN(ret);
}
/* }}} */


struct collection_exists_in_database_ctx
{
	const MYSQLND_CSTRING expected_collection_name;
	zval* exists;
};


/* {{{ collection_xplugin_op_on_row */
static const enum_hnd_func_status
collection_xplugin_op_on_row(
	void * context,
	XMYSQLND_SESSION session,
	xmysqlnd_stmt * const /*stmt*/,
	const XMYSQLND_STMT_RESULT_META * const /*meta*/,
	const zval * const row,
	MYSQLND_STATS * const /*stats*/,
	MYSQLND_ERROR_INFO * const /*error_info*/)
{
	collection_exists_in_database_ctx* ctx = static_cast<collection_exists_in_database_ctx*>(context);
	DBG_ENTER("collection_xplugin_op_on_row");
	if (ctx && row) {
		const MYSQLND_CSTRING object_name = { Z_STRVAL(row[0]), Z_STRLEN(row[0]) };
		const MYSQLND_CSTRING object_type = { Z_STRVAL(row[1]), Z_STRLEN(row[1]) };

		if (equal_mysqlnd_cstr(object_name, ctx->expected_collection_name)
			&& is_collection_object_type(object_type))
		{
			ZVAL_TRUE(ctx->exists);
		}
		else
		{
			ZVAL_FALSE(ctx->exists);
		}
	}
	DBG_RETURN(HND_AGAIN);
}
/* }}} */


/* {{{ xmysqlnd_collection::exists_in_database */
enum_func_status
xmysqlnd_collection::exists_in_database(
	struct st_xmysqlnd_session_on_error_bind on_error,
	zval* exists)
{
	DBG_ENTER("xmysqlnd_collection::exists_in_database");
	ZVAL_FALSE(exists);

	enum_func_status ret;
	static const MYSQLND_CSTRING query = {"list_objects", sizeof("list_objects") - 1 };

	st_collection_exists_in_database_var_binder_ctx var_binder_ctx = {
		mnd_str2c(schema->get_name()),
		mnd_str2c(collection_name),
		0
	};
	const st_xmysqlnd_session_query_bind_variable_bind var_binder = { collection_op_var_binder, &var_binder_ctx };

	collection_exists_in_database_ctx on_row_ctx = {
		mnd_str2c(collection_name),
		exists
	};

	const st_xmysqlnd_session_on_row_bind on_row = { collection_xplugin_op_on_row, &on_row_ctx };

	ret = schema->get_session()->query_cb(namespace_xplugin,
							   query,
							   var_binder,
							   noop__on_result_start,
							   on_row,
							   noop__on_warning,
							   on_error,
							   noop__on_result_end,
							   noop__on_statement_ok);

	DBG_RETURN(ret);
}
/* }}} */


struct st_collection_sql_single_result_ctx
{
	zval* result;
};


/* {{{ collection_xplugin_op_on_row */
static const enum_hnd_func_status
collection_sql_single_result_op_on_row(
	void * context,
	XMYSQLND_SESSION session,
	xmysqlnd_stmt * const /*stmt*/,
	const XMYSQLND_STMT_RESULT_META * const /*meta*/,
	const zval * const row,
	MYSQLND_STATS * const /*stats*/,
	MYSQLND_ERROR_INFO * const /*error_info*/)
{
	st_collection_sql_single_result_ctx* ctx = (st_collection_sql_single_result_ctx*) context;
	DBG_ENTER("collection_xplugin_op_on_row");
	if (ctx && row) {
		ZVAL_COPY_VALUE(ctx->result, &row[0]);
	}
	DBG_RETURN(HND_AGAIN);
}
/* }}} */


/* {{{ xmysqlnd_collection::count */
enum_func_status
xmysqlnd_collection::count(
	struct st_xmysqlnd_session_on_error_bind on_error,
	zval* counter)
{
	DBG_ENTER("xmysqlnd_collection::count");
	ZVAL_LONG(counter, 0);

	enum_func_status ret;

	xmysqlnd_schema * schema = get_schema();
	auto session = schema->get_session();

	char* query_str{nullptr};
	mnd_sprintf(&query_str, 0, "SELECT COUNT(*) FROM %s.%s", schema->get_name().s, get_name().s);
	if (!query_str) {
		DBG_RETURN(FAIL);
	}
	const MYSQLND_CSTRING query = {query_str, strlen(query_str)};

 	st_collection_sql_single_result_ctx on_row_ctx = {
		counter
	};

	const st_xmysqlnd_session_on_row_bind on_row = { collection_sql_single_result_op_on_row, &on_row_ctx };

	ret = session->query_cb(namespace_sql,
							   query,
							   noop__var_binder,
							   noop__on_result_start,
							   on_row,
							   noop__on_warning,
							   on_error,
							   noop__on_result_end,
							   noop__on_statement_ok);
	mnd_sprintf_free(query_str);
	DBG_RETURN(ret);
}
/* }}} */

/* {{{ xmysqlnd_collection::add */
xmysqlnd_stmt *
xmysqlnd_collection::add(XMYSQLND_CRUD_COLLECTION_OP__ADD * crud_op)
{
	DBG_ENTER("xmysqlnd_collection::add");
	xmysqlnd_stmt* ret{nullptr};
	XMYSQLND_SESSION session;
	struct st_xmysqlnd_message_factory msg_factory;
	struct st_xmysqlnd_msg__collection_add collection_add;

	if( xmysqlnd_crud_collection_add__finalize_bind(crud_op) == PASS ) {
		session = get_schema()->get_session();

		msg_factory = xmysqlnd_get_message_factory(&session->data->io,
											session->data->stats,
											session->data->error_info);
		collection_add = msg_factory.get__collection_add(&msg_factory);
		enum_func_status request_ret = collection_add.send_request(&collection_add,
											xmysqlnd_crud_collection_add__get_protobuf_message(crud_op));
		if (PASS == request_ret) {
			xmysqlnd_stmt * stmt = session->create_statement_object(session);
			stmt->get_msg_stmt_exec() = msg_factory.get__sql_stmt_execute(&msg_factory);
			ret = stmt;
		}
	} else {
		devapi::RAISE_EXCEPTION(err_msg_add_doc);
	}

	DBG_RETURN(ret);
}
/* }}} */


/* {{{ xmysqlnd_collection::remove */
xmysqlnd_stmt *
xmysqlnd_collection::remove(XMYSQLND_CRUD_COLLECTION_OP__REMOVE * op)
{
	xmysqlnd_stmt* ret{nullptr};
	DBG_ENTER("xmysqlnd_collection::remove");
	if (!op || FAIL == xmysqlnd_crud_collection_remove__finalize_bind(op)) {
		DBG_RETURN(ret);
	}
	if (xmysqlnd_crud_collection_remove__is_initialized(op)) {
		auto session = get_schema()->get_session();
		const struct st_xmysqlnd_message_factory msg_factory = xmysqlnd_get_message_factory(&session->data->io,
																			session->data->stats, session->data->error_info);
		struct st_xmysqlnd_msg__collection_ud collection_ud = msg_factory.get__collection_ud(&msg_factory);
		if (PASS == collection_ud.send_delete_request(&collection_ud, xmysqlnd_crud_collection_remove__get_protobuf_message(op))) {
			xmysqlnd_stmt * stmt = session->create_statement_object(session);
			stmt->get_msg_stmt_exec() = msg_factory.get__sql_stmt_execute(&msg_factory);
			ret = stmt;
		}
		DBG_INF(ret != nullptr? "PASS":"FAIL");
	}

	DBG_RETURN(ret);
}
/* }}} */


/* {{{ xmysqlnd_collection::modify */
xmysqlnd_stmt *
xmysqlnd_collection::modify(XMYSQLND_CRUD_COLLECTION_OP__MODIFY * op)
{
	xmysqlnd_stmt* ret{nullptr};
	DBG_ENTER("xmysqlnd_collection::modify");
	if (!op || FAIL == xmysqlnd_crud_collection_modify__finalize_bind(op)) {
		DBG_RETURN(ret);
	}
	if (xmysqlnd_crud_collection_modify__is_initialized(op)) {
		auto session = get_schema()->get_session();
		const struct st_xmysqlnd_message_factory msg_factory = xmysqlnd_get_message_factory(&session->data->io, session->data->stats, session->data->error_info);
		struct st_xmysqlnd_msg__collection_ud collection_ud = msg_factory.get__collection_ud(&msg_factory);
		if (PASS == collection_ud.send_update_request(&collection_ud, xmysqlnd_crud_collection_modify__get_protobuf_message(op))) {
			xmysqlnd_stmt * stmt = session->create_statement_object(session);
			stmt->get_msg_stmt_exec() = msg_factory.get__sql_stmt_execute(&msg_factory);
			ret = stmt;
		}
		DBG_INF(ret != nullptr? "PASS":"FAIL");
	}

	DBG_RETURN(ret);
}
/* }}} */


/* {{{ xmysqlnd_collection::find */
xmysqlnd_stmt*
xmysqlnd_collection::find(XMYSQLND_CRUD_COLLECTION_OP__FIND * op)
{
	xmysqlnd_stmt* stmt{nullptr};
	DBG_ENTER("xmysqlnd_collection::find");
	if (!op || FAIL == xmysqlnd_crud_collection_find__finalize_bind(op)) {
		DBG_RETURN(stmt);
	}
	if (xmysqlnd_crud_collection_find__is_initialized(op)) {
		auto session = get_schema()->get_session();
		stmt = session->create_statement_object(session);
		if (FAIL == stmt->send_raw_message(stmt, xmysqlnd_crud_collection_find__get_protobuf_message(op), session->data->stats, session->data->error_info)) {
			xmysqlnd_stmt_free(stmt, session->data->stats, session->data->error_info);
			stmt = nullptr;
		}
	}
	DBG_RETURN(stmt);
}
/* }}} */


/* {{{ xmysqlnd_collection::get_reference */
xmysqlnd_collection *
xmysqlnd_collection::get_reference()
{
	DBG_ENTER("xmysqlnd_collection::get_reference");
	++refcount;
	DBG_INF_FMT("new_refcount=%u", refcount);
	DBG_RETURN(this);
}
/* }}} */


/* {{{ xmysqlnd_collection::free_reference */
enum_func_status
xmysqlnd_collection::free_reference(MYSQLND_STATS * stats, MYSQLND_ERROR_INFO * error_info)
{
	enum_func_status ret{PASS};
	DBG_ENTER("xmysqlnd_collection::free_reference");
	DBG_INF_FMT("old_refcount=%u", refcount);
	if (!(--refcount)) {
		cleanup(stats, error_info);
	}
	DBG_RETURN(ret);
}
/* }}} */


/* {{{ xmysqlnd_collection::free_contents */
void
xmysqlnd_collection::free_contents()
{
	DBG_ENTER("xmysqlnd_collection::free_contents");
	if (collection_name.s) {
		mnd_efree(get_name().s);
		collection_name.s = nullptr;
	}
	DBG_VOID_RETURN;
}
/* }}} */


/* {{{ xmysqlnd_collection::cleanup */
void
xmysqlnd_collection::cleanup(MYSQLND_STATS * stats, MYSQLND_ERROR_INFO * error_info)
{
	DBG_ENTER("xmysqlnd_collection::cleanup");
	free_contents();
	xmysqlnd_schema_free(schema, stats, error_info);

	DBG_VOID_RETURN;
}
/* }}} */


/* {{{ xmysqlnd_collection_create */
PHP_MYSQL_XDEVAPI_API xmysqlnd_collection *
xmysqlnd_collection_create(xmysqlnd_schema * schema,
								const MYSQLND_CSTRING collection_name,
								const zend_bool persistent,
								const MYSQLND_CLASS_METHODS_TYPE(xmysqlnd_object_factory) * const object_factory,
								MYSQLND_STATS * const stats,
								MYSQLND_ERROR_INFO * const error_info)
{
	xmysqlnd_collection* ret{nullptr};
	DBG_ENTER("xmysqlnd_collection_create");
	if (collection_name.s && collection_name.l) {
		ret = object_factory->get_collection(object_factory, schema, collection_name, persistent, stats, error_info);
		if (ret) {
			ret = ret->get_reference();
		}
	}
	DBG_RETURN(ret);
}
/* }}} */


/* {{{ xmysqlnd_collection_free */
PHP_MYSQL_XDEVAPI_API void
xmysqlnd_collection_free(xmysqlnd_collection * const collection, MYSQLND_STATS * stats, MYSQLND_ERROR_INFO * error_info)
{
	DBG_ENTER("xmysqlnd_collection_free");
	DBG_INF_FMT("collection=%p",
				collection);
	if (collection) {
		collection->free_reference(stats, error_info);
	}
	DBG_VOID_RETURN;
}
/* }}} */

} // namespace drv

} // namespace mysqlx

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
