=head1 NAME

iPE::AnnotationPlugin - Base class for annotation format plugin readers.

=head1 DESCRIPTION

The role of the AnnotationPlugin is to read in annotation files, and use the corresponding iPE::FeatureMap to create preliminary iPE::Transcripts.  These are constructed by iPE::Annotation, and are finalized by iPE::Annotation.

The AnnotationPlugin must account for whether or not partial transcripts (transcripts which do not have both a start and stop codon annotated) will be used for estimation.  It is the job of the plugin to check this variable and if it indicates that partial transcripts are unwanted, they must be thrown out by the plugin.  This should be defined in L<iPE::Options> as allowPartialTranscripts, and can be accessed through the L<iPE> module with the g_options function.

=cut

package iPE::AnnotationPlugin;
use iPE;
use strict;

=head1 MEMBERS

=over 8

=item filename ()

This returns the file to be parsed for annotations.  Use this to open and read the file.

=cut
sub filename { shift->{filename_} }

=item featureMap ()

This returns the feature map with which to translate the annotations to state sequences.

=cut
sub featureMap { shift->{featureMap_} }

=item transcripts ()

This is the member which should be populated with all the transcripts found in in the file.  This is an array reference and can be accessed via the {transcripts_} hash key of the calling object.

=cut
sub transcripts { shift->{transcripts_} }

=back

=head1 FUNCTIONS

=over 8

=item new (memberHash)

The constructor populates all the variables except the transcripts variable.  In order to do additional construction in your class, you must call this superclass constructor in your constructor.  This is probably not necessary since you can do this in the populate () function, which is only called once after the constructor is called.

The memberHash hash reference requires the following keys to be defined:

=over 8

=item file

The name of the file to be read in (fully qualified pathname if necessary) for annotation.

=item featureMap

A reference to a featureMap object (does not have to be owned solely by this instantiation) which the plugin can refer to in order to translate features into state sequences.

=back

=cut

sub new {
    my ($class, $m) = @_;
    my $this = bless {}, $class;

    die "$class must be instantiated with a file and featureMap key defined.\n"
        if (not defined $m->{filename} ||
            not defined $m->{featureMap});

    # test that the file is readable.
    open FH, "<$m->{filename}" 
        or die "Could not open file $m->{filename} for reading.\n";
    close FH;

    $this->{filename_} = $m->{filename};
    $this->{featureMap_} = $m->{featureMap};
    $this->{transcripts_} = [];

    return $this;
}

=item populate (length)

This function is unimplemented in the base class and must be implemented in any subclass of this class.  It is here where the work of adding L<iPE::Annotation::Transcript>s to the {transcripts_} array reference of the calling object must be done.  

In order to create transcripts, iterate over every transcript of the annotation, and then every feature of the transcript.  For each feature, call the method featureMapping () on the featureMap object.  This will return FeatureMappings which pertain to the feature with the passed name.  Then iterate over every featureMapping, calling addToTranscript () on that object, passing in an iPE::Annotation::Transcript object.  This method will automatically add a feature to the transcript for you.

When finished, finalize the transcript using the finalize () method in iPE::Annotation::Transcript.  Here you will pass the length parameter, which represents the underlying sequence length, to check to see that the transcript does not overstep the outer bounds of the transcript.

=cut
sub populate {
    my ($this) = @_;
    die ref($this)." has no populate method.\n";
}

=back

=head1 SEE ALSO

L<iPE::Annotation::Transcript>, L<iPE::FeatureMap>, L<iPE::FeatureMapping> L<iPE::Annotation>, L<iPE::Options>, L<iPE>

=head1 AUTHOR

Bob Zimmermann (rpz@cse.wustl.edu)

=cut

1;

