/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cstring>
#include <iostream>
#include <boost/test/unit_test.hpp>
#include <stdexcept>

#include "../ScopedMap.hh"

using namespace peekabot;

class ArbitraryObject
{
public:
    ArbitraryObject() : m_value(1) {}
private:
    int m_value;
};


BOOST_AUTO_TEST_SUITE( ScopedMapTests );

BOOST_AUTO_TEST_CASE( pushing_variables )
{
    ScopedMap m;

    // Try pushing a variable and verify its existance
    std::string str1("variable1");
    m.push_variable<int>(str1, 1);
    BOOST_CHECK( m.exists<int>(str1) );

    // Try pushing a new value for the same variable. Should still exist.
    m.push_variable<int>(str1, 2);
    BOOST_CHECK( m.exists<int>(str1) );    

    // Try pushing an arbitrary object.
    std::string str2("variable2");
    ArbitraryObject a;
    m.push_variable<ArbitraryObject>(str2, a);
    BOOST_CHECK( m.exists<ArbitraryObject>(str2) );

    // Try pushing a variable with the same name as variable1, but with
    // different type. This should throw.
    BOOST_CHECK_THROW( m.push_variable<std::string>(str1, str2),
                       boost::bad_any_cast );

}

BOOST_AUTO_TEST_CASE ( popping_variables )
{
    ScopedMap m;

    // Try pushing a variable and then popping it.
    std::string str1("variable1");
    m.push_variable<int>(str1, 1);
    m.pop_variable<int>(str1);
    BOOST_CHECK( !m.exists<int>(str1) );

    // Try pushing a couple of values for the same variable and them popping
    // them.
    std::string str2("variable2");
    m.push_variable<float>(str2, 1.0);
    m.push_variable<float>(str2, 2.0);
    m.push_variable<float>(str2, 3.0);
    m.push_variable<float>(str2, 4.0);
    BOOST_CHECK( m.exists<float>(str2) );
    m.pop_variable<float>(str2);
    BOOST_CHECK( m.exists<float>(str2) );
    m.pop_variable<float>(str2);
    BOOST_CHECK( m.exists<float>(str2) );
    m.pop_variable<float>(str2);
    BOOST_CHECK( m.exists<float>(str2) );
    // This should remove the last value for "variable2"
    m.pop_variable<float>(str2);
    BOOST_CHECK( !m.exists<float>(str2) ); 

    // Try pushing a value for a variable, and then popping it as a different
    // type.
    std::string str3("variable3");
    m.push_variable<std::string>(str3, "String variable");
    BOOST_CHECK_THROW( m.pop_variable<int>(str3), boost::bad_any_cast);

    // Try popping a variable that never existed to begin with.
    std::string str4("variable4");
    BOOST_CHECK_THROW( m.pop_variable<char>(str4), std::runtime_error ); 

}

BOOST_AUTO_TEST_CASE ( getting_variables )
{
    ScopedMap m;

    // Push a variable, fetch its value and see that it's correct.
    std::string str1("variable1");
    m.push_variable<int>(str1, 1465);
    BOOST_CHECK( m.get_variable<int>(str1) == 1465 );

    // Try pushing and popping a couple of times, all the while checking to
    // see that the current value is correct.
    std::string str2("variable2");
    m.push_variable<double>(str2, 3.25);
    BOOST_CHECK( m.get_variable<double>(str2) == 3.25 );
    m.push_variable<double>(str2, 5.5);
    BOOST_CHECK( m.get_variable<double>(str2) == 5.5 );
    m.pop_variable<double>(str2);
    BOOST_CHECK( m.get_variable<double>(str2) == 3.25 );
    m.push_variable<double>(str2, 12.125);
    BOOST_CHECK( m.get_variable<double>(str2) == 12.125 );
    m.pop_variable<double>(str2);
    BOOST_CHECK( m.get_variable<double>(str2) == 3.25 );
    m.pop_variable<double>(str2);
    BOOST_CHECK( !m.exists<float>(str2) );

    // Try getting pushing "variable3" as an int, and then getting as a float
    std::string str3("variable3");
    m.push_variable<int>(str3, 4);
    BOOST_CHECK_THROW( m.get_variable<float>(str3), boost::bad_any_cast );

    // Try getting a variable that was never pushed.
    std::string str4("variable4");
    BOOST_CHECK_THROW( m.get_variable<char>(str4), std::runtime_error );
}


BOOST_AUTO_TEST_SUITE_END();
