/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cstring>
#include <boost/test/unit_test.hpp>

#include "../ChunkedBuffer.hh"

using namespace peekabot;





BOOST_AUTO_TEST_SUITE( ChunkedBufferTests );



BOOST_AUTO_TEST_CASE( write_and_read )
{
    size_t chunk_size[] = { 1, 2, 8, 16, 64, 256, 1024 };
    
    for( int i = 0; i < 7; ++i )
    {
        ChunkedBuffer buffer(chunk_size[i]);

        const char data[] = ("hello world! this is a somewhat "
                             "long message. yeah!");
        char readbuf[1024];

        // write some data and read it back
        buffer.write(data, sizeof(data));
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 1024), sizeof(data) );
        // make sure the bytes wrote and the same we got back
        BOOST_CHECK( strcmp(data, readbuf) == 0 );
        // the buffer should be empty now - is it?
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 1024), (unsigned)0 );

        // write some data and read back max 10 bytes of it...
        buffer.write(data, sizeof(data));
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 10), (unsigned)10 );
        // make sure the bytes wrote and the same we got back
        BOOST_CHECK( strncmp(data, readbuf, 10) == 0 );
        // read the remaining bytes
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 1024), sizeof(data)-10 );
        // the buffer should be empty now - is it?
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 1024), (unsigned)0 );

        // write data, read "hello world! ", then read the rest
        buffer.write(data, sizeof(data));
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 13), 13u );
        BOOST_CHECK( memcmp(readbuf, data, 13) == 0 );
        BOOST_CHECK_EQUAL( buffer.read(readbuf, 1024), sizeof(data)-13 );
        BOOST_CHECK( strcmp(readbuf, data+13) == 0 );
    }
}


BOOST_AUTO_TEST_CASE( write_and_read_byte )
{
    ChunkedBuffer buffer(2);

    // write 5 bytes, or 2.5 chunks
    buffer.write_byte(0);
    buffer.write_byte(1);
    buffer.write_byte(2);
    buffer.write_byte(3);
    buffer.write_byte(4);
    // ...and read 'em back
    for( uint8_t i = 0; i < 5; i++ )
        BOOST_CHECK_EQUAL( buffer.read_byte(), (unsigned)i );
}


BOOST_AUTO_TEST_CASE( get_size )
{
    ChunkedBuffer buffer(10);
    char data[] = "0123456789abcdefghijklmnopq";
    char readbuf[16];

    BOOST_CHECK_EQUAL( buffer.get_size(), (unsigned)0 );
    buffer.write_byte(0);
    BOOST_CHECK_EQUAL( buffer.get_size(), (unsigned)1 );
    buffer.write(data, sizeof(data));
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data)+1 );
    BOOST_CHECK_EQUAL( buffer.read(readbuf, 16), (unsigned)16 );
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data)+1-16 );
}


BOOST_AUTO_TEST_CASE( is_empty )
{
    ChunkedBuffer buffer(10);

    BOOST_CHECK( buffer.is_empty() == true );
    buffer.write_byte(0);
    BOOST_CHECK( buffer.is_empty() == false );
    buffer.read_byte();
    BOOST_CHECK( buffer.is_empty() == true );
}


BOOST_AUTO_TEST_CASE( clear )
{
    ChunkedBuffer buffer(2);

    // write 5 bytes, or 2.5 chunks
    buffer.write_byte(0);
    buffer.write_byte(1);
    buffer.write_byte(2);
    buffer.write_byte(3);
    buffer.write_byte(4);
    // clear the buffer!
    buffer.clear();
    BOOST_CHECK( buffer.is_empty() );

    // test to write a byte and then read it back
    buffer.write_byte(4);
    BOOST_CHECK_EQUAL( buffer.get_size(), 1u );
    BOOST_CHECK_EQUAL( buffer.read_byte(), 4u );
}


BOOST_AUTO_TEST_CASE( one_byte_chunk_size )
{
    ChunkedBuffer buffer(1);

    char data[] = "0123456789abcdefghijklmnopq";
    char readbuf[128];
    BOOST_REQUIRE( sizeof(data) <= 128 );

    buffer.write_byte(100);
    buffer.write(data, sizeof(data));
    BOOST_CHECK_EQUAL( buffer.read_byte(), (unsigned)100 );
    BOOST_CHECK_EQUAL( buffer.read(readbuf, 128), sizeof(data) );
    BOOST_CHECK( strcmp(data, readbuf) == 0 );
}


// Buffer size > max pool memory -- check for potential memory leaks
BOOST_AUTO_TEST_CASE( large_buffer )
{
    ChunkedBuffer buffer(10);

    char data[] = "0123456789abcdefghijklmnopqABCDEFGHIJKLMNO9876543210qwerty";
    char readbuf[128];
    BOOST_REQUIRE( sizeof(data) <= 128 );
    
    buffer.write(data, sizeof(data));
    BOOST_CHECK_EQUAL( buffer.read(readbuf, 128), sizeof(data) );
    BOOST_CHECK( buffer.is_empty() );

    buffer.write(data, sizeof(data));
    BOOST_CHECK_EQUAL( buffer.read(readbuf, 128), sizeof(data) );
    BOOST_CHECK( buffer.is_empty() );

    // leave the buffer in a non-empty state to check lure out potential 
    // memory leaks
    buffer.write(data, sizeof(data));
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data) );
}


BOOST_AUTO_TEST_CASE( overwrite )
{
    struct TestInstance
    {
        char input[512];
        char expected[512];
        size_t pos;
        char overwrt[512];
    } data[11];
    
    strcpy(data[0].input,    "abcDEADBEEFdef!");
    strcpy(data[0].expected, "abcMEATGRINDER!");
    strcpy(data[0].overwrt,  "MEATGRINDER");
    data[0].pos = 3;

    strcpy(data[1].input,    "DEADBEEF---FRESHMEAT---DEADBEEF");
    strcpy(data[1].expected, "APALIKE!---FRESHMEAT---DEADBEEF");
    strcpy(data[1].overwrt,  "APALIKE!");
    data[1].pos = 0;

    strcpy(data[2].input,    "DEADBEEF---FRESHMEAT---DEADBEEF");
    strcpy(data[2].expected, "DEADBEEF---FRESHMEAT---APALIKE!");
    strcpy(data[2].overwrt,  "APALIKE!");
    data[2].pos = 23;

    strcpy(data[3].input,    "a");
    strcpy(data[3].expected, "A");
    strcpy(data[3].overwrt,  "A");
    data[3].pos = 0;

    strcpy(data[4].input,    "0123456789012345678901234567890123456789");
    strcpy(data[4].expected, "*123456789012345678901234567890123456789");
    strcpy(data[4].overwrt,  "*");
    data[4].pos = 0;

    strcpy(data[5].input,    "0123456789012345678901234567890123456789");
    strcpy(data[5].expected, "0*23456789012345678901234567890123456789");
    strcpy(data[5].overwrt,  "*");
    data[5].pos = 1;

    strcpy(data[6].input,    "0123456789012345678901234567890123456789");
    strcpy(data[6].expected, "01234567890*2345678901234567890123456789");
    strcpy(data[6].overwrt,  "*");
    data[6].pos = 11;

    strcpy(data[7].input,    "0123456789012345678901234567890123456789");
    strcpy(data[7].expected, "012345678901234567890123456789012345678*");
    strcpy(data[7].overwrt,  "*");
    data[7].pos = 39;

    strcpy(data[8].input,    "0123456789012345678901234567890123456789");
    strcpy(data[8].expected, "01234567890123456789012345678901234567*9");
    strcpy(data[8].overwrt,  "*");
    data[8].pos = 38;

    strcpy(data[9].input,    "0123456789012345678901234567890123456789");
    strcpy(data[9].expected, "012345678901234567890123456789**********");
    strcpy(data[9].overwrt,  "**********");
    data[9].pos = 30;

    strcpy(data[10].input,    "0123456789012345678901234567890123456789");
    strcpy(data[10].expected, "0123456789*******************90123456789");
    strcpy(data[10].overwrt,  "*******************");
    data[10].pos = 10;


    char readbuf[1024];
    // Declare the different chunk sizes to test with
    size_t sizes[] = { 1, 2, 3, 4, 5, 8, 16, 32, 128, 256, 333, 1024 };

    for( size_t j = 0; j < sizeof(sizes)/sizeof(size_t); j++ )
    {
        for( int i = 0; i < 11; i++ )
        {
            ChunkedBuffer buf1(sizes[j]);
            ChunkedBuffer buf2(sizes[j]);
            ChunkedBuffer buf3(sizes[j]);

            //
            // We always write and subsequently read the null terminator
            // to and from the buffer respectively, since it's needed for 
            // all string operations.
            //
            buf2.write(data[i].input, strlen(data[i].input)+1);

            buf1.write("skrp", 5);
            buf1.write(data[i].input, strlen(data[i].input)+1);
            buf1.read(readbuf, 5);

            buf3.write("xyz", 3);
            buf3.write(data[i].input, strlen(data[i].input)+1);
            buf3.write("XYZ", 3);

            BOOST_CHECK( 
                !buf1.overwrite(
                    data[i].overwrt, 
                    strlen(data[i].overwrt), 
                    data[i].pos) );

            BOOST_CHECK(                 
                !buf2.overwrite(
                    data[i].overwrt, 
                    strlen(data[i].overwrt), 
                    data[i].pos) );

            BOOST_CHECK( 
                !buf3.overwrite(
                    data[i].overwrt, 
                    strlen(data[i].overwrt), 
                    data[i].pos+3) );

            // Get rid of the leading crap
            buf3.read(readbuf, 3);

            BOOST_CHECK_EQUAL( 
                buf1.read(
                    readbuf,
                    strlen(data[i].expected)+1), 
                strlen(data[i].expected)+1 );
            BOOST_CHECK( !strcmp(readbuf, data[i].expected) );

            BOOST_CHECK_EQUAL( 
                buf2.read(
                    readbuf,
                    strlen(data[i].expected)+1), 
                strlen(data[i].expected)+1 );
            BOOST_CHECK( !strcmp(readbuf, data[i].expected) );

            BOOST_CHECK_EQUAL( 
                buf3.read(
                    readbuf,
                    strlen(data[i].expected)+1), 
                strlen(data[i].expected)+1 );
            BOOST_CHECK( !strcmp(readbuf, data[i].expected) );
        }


        //
        // Try to do an overwrite past the end of a buffer, and 
        // check that it fails!
        //
        TestInstance faildata;
        
        strcpy(faildata.input,    "0123456789012345678901234567890123456789");
        strcpy(faildata.expected, "012345678901234567890123456789012*******");
        strcpy(faildata.overwrt,  "**********");
        faildata.pos = 33;
        
        ChunkedBuffer buf(sizes[j]);
        
        buf.write(faildata.input, strlen(faildata.input));
        
        BOOST_CHECK( 
            buf.overwrite(
                faildata.overwrt, strlen(faildata.overwrt), faildata.pos) );
        
        BOOST_CHECK_EQUAL( 
            buf.read(
                readbuf,
                strlen(faildata.expected)), 
            strlen(faildata.expected) );
        readbuf[strlen(faildata.expected)] = '\0';
        BOOST_CHECK( !strcmp(readbuf, faildata.expected) );




        //
        // Check zero byte overwrites...
        //
        buf.clear();
        buf.write("deadbeef", 8+1);
        BOOST_CHECK( !buf.overwrite( 0, 0, 3 ) );
        BOOST_CHECK_EQUAL( buf.read(readbuf, 8+1), (size_t)8+1 );
        BOOST_CHECK( !strcmp(readbuf, "deadbeef") );
    }
}


BOOST_AUTO_TEST_CASE( peek_multi_byte )
{
    ChunkedBuffer buffer(10);

    const char data[] = "hello world! this is a somewhat long message. yeah!";
    char readbuf[1024];

    // write some data to the buffer
    buffer.write(data, sizeof(data));

    // Save the original size of the buffer
    size_t orig_size = buffer.get_size();
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data) );

    // peek 5 bytes. Expected: "hello", return=5
    BOOST_CHECK_EQUAL( buffer.peek(readbuf, 5), 5u );
    BOOST_CHECK( !strncmp(readbuf, data, 5) );
    BOOST_CHECK_EQUAL( buffer.get_size(), orig_size );
    
    // peek 12 (> 10) bytes. Expected: "hello world!", return=12
    BOOST_CHECK_EQUAL( buffer.peek(readbuf, 12), 12u );
    BOOST_CHECK( !strncmp(readbuf, data, 12) );
    BOOST_CHECK_EQUAL( buffer.get_size(), orig_size );

    // peek 2*strlen(data) bytes. Expected: data, return=strlen(data)
    BOOST_CHECK_EQUAL( buffer.peek(readbuf, 2*sizeof(data)), sizeof(data) );
    BOOST_CHECK( !strcmp(readbuf, data) );
    BOOST_CHECK_EQUAL( buffer.get_size(), orig_size );
}


BOOST_AUTO_TEST_CASE( peek_single_byte )
{
    ChunkedBuffer buffer(2);

    const char data[] = "abc";
    uint8_t x;

    // write some data to the buffer
    buffer.write(data, sizeof(data));

    // Save the original size of the buffer
    size_t orig_size = buffer.get_size();
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data) );

    BOOST_CHECK_EQUAL( buffer.peek(x), false );
    BOOST_CHECK_EQUAL( x, static_cast<uint8_t>('a') );
    BOOST_CHECK_EQUAL( buffer.get_size(), orig_size );
    
    // redo the same operation - results should be identical
    BOOST_CHECK_EQUAL( buffer.peek(x), false );
    BOOST_CHECK_EQUAL( x, static_cast<uint8_t>('a') );
    BOOST_CHECK_EQUAL( buffer.get_size(), orig_size );

    // peek past the end of the buffer - should fail
    buffer.clear();

    BOOST_CHECK_EQUAL( buffer.get_size(), 0u );
    BOOST_CHECK_EQUAL( buffer.peek(x), true );
    BOOST_CHECK_EQUAL( buffer.get_size(), 0u );
}


BOOST_AUTO_TEST_CASE( discard )
{
    ChunkedBuffer buffer(8);

    const char data[] = "hello world, foo bar";
    char readbuf[64];

    // Test empty-buffer discard behaviour
    BOOST_CHECK_EQUAL( buffer.discard(10), 0u );
    BOOST_CHECK_EQUAL( buffer.get_size(), 0u );

    // write some data to the buffer
    buffer.write(data, sizeof(data));
    size_t n_discard = strlen("hello world, ");

    // Discard the "hello world, "-portion of the buffer
    BOOST_CHECK_EQUAL( buffer.discard(n_discard), n_discard );
    BOOST_CHECK_EQUAL( buffer.get_size(), sizeof(data)-n_discard );
    // check the remaining buffer contents
    buffer.peek(readbuf, sizeof(data)-n_discard);
    BOOST_CHECK( !strcmp(readbuf, &data[n_discard]) );

    // Discard the rest of the buffer
    BOOST_CHECK_EQUAL( buffer.discard(sizeof(data)-n_discard), 
                       sizeof(data)-n_discard );
    BOOST_CHECK_EQUAL( buffer.get_size(), 0u );

    // Re-visit empty-buffer discard behaviour
    BOOST_CHECK_EQUAL( buffer.discard(10), 0u );
    BOOST_CHECK_EQUAL( buffer.get_size(), 0u );
}


BOOST_AUTO_TEST_CASE( copy_ctor )
{
    const char data1[] = "hello";
    const char data2[] = "hello world, foo bar, moin loin groin";
    char readbuf[64];

    size_t chunk_size[] = { 1, 2, 8, 16, 64, 256, 1024 };

    for( int i = 0; i < 7; ++i )
    {
        ChunkedBuffer orig(chunk_size[i]);

        // Test copying of empty buffers
        ChunkedBuffer copy1(orig);
        BOOST_CHECK_EQUAL( copy1.get_size(), orig.get_size() );
    
        // Test copying of a non-empty single chunk buffer
        orig.write(data1, sizeof(data1));
    
        ChunkedBuffer copy2(orig);
        BOOST_CHECK_EQUAL( copy2.get_size(), orig.get_size() );
        BOOST_CHECK_EQUAL( copy2.read(readbuf, 64), orig.get_size() );
        BOOST_CHECK( !strncmp(readbuf, data1, 64) );


        // Test copying of a non-empty multiple chunk buffer
        orig.clear();
        orig.write(data2, sizeof(data2));
    
        ChunkedBuffer copy3(orig);
        BOOST_CHECK_EQUAL( copy3.get_size(), orig.get_size() );
        BOOST_CHECK_EQUAL( copy3.read(readbuf, 64), orig.get_size() );
        BOOST_CHECK( !strncmp(readbuf, data2, 64) );
    }
}



BOOST_AUTO_TEST_SUITE_END();
