/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_SERIALIZATION_SERIALIZATION_INTERFACE_HH_INCLUDED
#define PEEKABOT_SERIALIZATION_SERIALIZATION_INTERFACE_HH_INCLUDED


#include <boost/type_traits.hpp>
#include <boost/mpl/and.hpp>
#include <boost/mpl/not.hpp>
#include <boost/mpl/or.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/cstdint.hpp>

#include "SerializableFactory.hh"
#include "SerializationBuffer.hh"
#include "Access.hh"


namespace peekabot
{
    namespace serialization
    {
        class SerializationInterface;

        /**
         * \brief An interface for writing binary data in a format compatible with
         * DeserializationBuffer. It's independent of the underlying data source.
         *
         * \sa DeserializationInterface
         */
        class SerializationInterface
        {
            // Meta function for checking for save() methods
            typedef char (&no_tag)[1];
            typedef char (&yes_tag)[2];

            template<typename T, void (T::*)(SerializationInterface &) const>
            struct ptmf_save_helper {};

            template<typename T> static no_tag has_member_save_helper(...);

            template<typename T> static yes_tag has_member_save_helper(
                ptmf_save_helper<T, &T::save>* p);

            template<typename T> struct has_member_save
            {
                BOOST_STATIC_CONSTANT(
                    bool, value = sizeof(
                        has_member_save_helper<T>(0)) == sizeof(yes_tag));

                typedef boost::mpl::bool_<value> type;
            };

        public:
            SerializationInterface(SerializationBuffer &buf) : m_buf(buf) {}

            SerializationInterface(const SerializationInterface &other)
                : m_buf(other.m_buf) {}

            inline void save_binary(const void *x, size_t n)
            {
                m_buf.write(x, n);
            }


            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                void>::type
            save_array(const T *t, std::size_t n)
            {
                save_binary(t, sizeof(T)*n);
            }


            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::not_<
                    boost::mpl::and_<
                        boost::is_arithmetic<T>,
                        boost::mpl::not_< boost::is_same<T, bool> >
                        >
                    >::value,
                void>::type
            save_array(const T *t, std::size_t n)
            {
                for( std::size_t i = 0; i < n; ++i )
                    *this << t[i];
            }

            // (Non-polymorphic) serialization of types with a save() method
            template<class T> inline
            typename boost::enable_if_c<
                has_member_save<T>::value,
                SerializationInterface &>::type
            operator<<(const T &t)
            {
                *this << boost::uint8_t(
                    typename detail::VersionTag<T>::value());
                t.save(*this);
                return *this;
            }

            // Serialization of pointers to polymorphic types, which we expect
            // to be registered
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_polymorphic<T>::value,
                SerializationInterface &>::type
            operator<<(const T *t)
            {
                const serialization::SerializableInfoBase &info =
                    TheSerializableFactory::instance().get_serializable_info(t);
                *this << info.m_id << boost::uint8_t(info.version());
                info.save(*this, t);
                return *this;
            }

            // Enum
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_enum<T>::value,
                SerializationInterface &>::type
            operator<<(const T t)
            {
                int32_t tmp = (int32_t)t;
                *this << tmp;
                return *this;
            }

            // bool
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_same<bool, T>::value,
                SerializationInterface &>::type
            operator<<(const T t)
            {
                uint8_t tmp = (uint8_t)t;
                *this << tmp;
                return *this;
            }

            // Non-bool arithmetic types
            // E.g. int, long, float, double, char, etc.
            template<class T> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                SerializationInterface &>::type
            operator<<(const T &t)
            {
                save_binary(&t, sizeof(T));
                return *this;
            }

            // Array of non-bool arithmetic types
            template<class T, size_t n> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                SerializationInterface &>::type
            operator<<(const T (&t)[n])
            {
                // These types are all PODs, and an array of PODs is a POD, and
                // thus its memory representation is contiguous => we can do an
                // efficient save!
                save_binary(t, sizeof(T)*n);
                return *this;
            }

            // Array of non-arithmetic types and bools
            // E.g. bool, enum, pointers, user-defined types, non-PODs
            template<class T, size_t n> inline
            typename boost::enable_if_c<
                boost::mpl::or_<
                    boost::mpl::not_< boost::is_arithmetic<T> >,
                    boost::is_same<T, bool>
                    >::value,
                SerializationInterface &>::type
            operator<<(const T (&t)[n])
            {
                for( size_t i = 0; i < n; ++i )
                    *this << t[i];
                return *this;
            }

            // Everything else...
            // (This allows non-intrusive serialization)
            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::not_<
                    boost::mpl::or_<
                        has_member_save<T>,
                        boost::is_polymorphic<T>,
                        boost::is_arithmetic<T>,
                        boost::is_enum<T>,
                        boost::is_pointer<T> >
                    >::value,
                SerializationInterface &>::type
            operator<<(const T &t)
            {
                *this << boost::uint8_t(
                    typename detail::VersionTag<T>::value());
                save(*this, t);
                return *this;
            }

        private:
            SerializationBuffer &m_buf;
        };
    }


    using serialization::SerializationInterface;

    inline SerializationInterface &operator<<(
        SerializationInterface &ar, const std::string &str)
    {
        boost::uint32_t len = (boost::uint32_t)str.length();
        ar << len;
        ar.save_binary(str.c_str(), len);
        return ar;
    }
}


#endif // PEEKABOT_SERIALIZATION_SERIALIZATION_INTERFACE_HH_INCLUDED
