/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_SERIALIZATION_DESERIALIZATION_INTERFACE_HH_INCLUDED
#define PEEKABOT_SERIALIZATION_DESERIALIZATION_INTERFACE_HH_INCLUDED


#include <boost/type_traits.hpp>
#include <boost/mpl/and.hpp>
#include <boost/mpl/not.hpp>
#include <boost/mpl/or.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/cstdint.hpp>

#include "../Bytesex.hh"
#include "SerializableFactory.hh"
#include "DeserializationBuffer.hh"
#include "Access.hh"


namespace peekabot
{
    namespace serialization
    {
        class DeserializationInterface;

        /**
         * \brief An interface for reading binary data independent of their
         * byte order. It's independent of the underlying data source.
         *
         * \sa SerializationInterface
         */
        class DeserializationInterface
        {
            // Meta function for checking for load() methods
            typedef char (&no_tag)[1];
            typedef char (&yes_tag)[2];

            template<typename T, void (T::*)(DeserializationInterface &)>
            struct ptmf_load_helper {};

            template<typename T> static no_tag has_member_load_helper(...);

            template<typename T> static yes_tag has_member_load_helper(
                ptmf_load_helper<T, &T::load>* p);

            template<typename T> struct has_member_load
            {
                BOOST_STATIC_CONSTANT(
                    bool, value = sizeof(
                        has_member_load_helper<T>(0)) == sizeof(yes_tag));

                typedef boost::mpl::bool_<value> type;
            };

        public:
            DeserializationInterface(
                DeserializationBuffer &buf, bool buffer_is_big_endian)
                : m_buf(buf), m_buffer_is_big_endian(buffer_is_big_endian) {}

            DeserializationInterface(const DeserializationInterface &other)
                : m_buf(other.m_buf),
                  m_buffer_is_big_endian(other.m_buffer_is_big_endian) {}

            inline bool is_big_endian() const throw()
            {
                return m_buffer_is_big_endian;
            }

            inline void load_binary(void *x, size_t n)
            {
                m_buf.read(x, n);
            }

            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                void>::type
            load_array(T *t, std::size_t n)
            {
                // An array of PODs is a POD, and thus its memory representation
                // is contiguous => we can do an efficient load!
                load_binary(t, sizeof(T)*n);
                if( sizeof(T) > 1 &&
                    m_buffer_is_big_endian != (PEEKABOT_BYTE_ORDER == PEEKABOT_BIG_ENDIAN) )
                    switch_byte_order(t, n);
            }

            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::not_<
                    boost::mpl::and_<
                        boost::is_arithmetic<T>,
                        boost::mpl::not_< boost::is_same<T, bool> >
                        >
                    >::value,
                void>::type
            load_array(T *t, std::size_t n)
            {
                for( std::size_t i = 0; i < n; ++i )
                    *this >> t[i];
            }

            // (Non-polymorphic) deserialization of types with a load() method
            template<class T> inline
            typename boost::enable_if_c<
                has_member_load<T>::value,
                DeserializationInterface &>::type
            operator>>(T &t)
            {
                boost::uint8_t version;
                *this >> version;
                t.load(*this, version);
                return *this;
            }

            // Deserialization of pointers to polymorphic types, which we
            // expect to be registered
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_polymorphic<T>::value,
                DeserializationInterface &>::type
            operator>>(T * &t)
            {
                IdType id;
                boost::uint8_t version;
                *this >> id >> version;
                const SerializableInfoBase &info =
                    TheSerializableFactory::instance().get_serializable_info(id);
                t = reinterpret_cast<T *>(info.create());
                info.load(*this, t, version);
                return *this;
            }

            // Enum
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_enum<T>::value,
                DeserializationInterface &>::type
            operator>>(T &t)
            {
                int32_t tmp;
                *this >> tmp;
                t = (T)tmp;
                return *this;
            }

            // bool
            template<class T> inline
            typename boost::enable_if_c<
                boost::is_same<bool, T>::value,
                DeserializationInterface &>::type
            operator>>(T &t)
            {
                uint8_t tmp;
                *this >> tmp;
                t = (T)tmp;
                return *this;
            }

            // Non-bool arithmetic types
            // E.g. int, long, float, double, char, etc.
            template<class T> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                DeserializationInterface &>::type
            operator>>(T &t)
            {
                load_binary(&t, sizeof(T));
                if( sizeof(T) > 1 &&
                    m_buffer_is_big_endian != (PEEKABOT_BYTE_ORDER == PEEKABOT_BIG_ENDIAN) )
                    switch_byte_order(&t, 1);
                return *this;
            }

            // Array of non-bool arithmetic types
            template<class T, size_t n> inline
            typename boost::enable_if_c<
                boost::mpl::and_<
                    boost::is_arithmetic<T>,
                    boost::mpl::not_< boost::is_same<T, bool> >
                    >::value,
                DeserializationInterface &>::type
            operator>>(T (&t)[n])
            {
                // These types are all PODs, and an array of PODs is a POD, and
                // thus its memory representation is contiguous => we can do an
                // efficient load!
                load_binary(t, sizeof(T)*n);
                if( sizeof(T) > 1 &&
                    m_buffer_is_big_endian != (PEEKABOT_BYTE_ORDER == PEEKABOT_BIG_ENDIAN) )
                    switch_byte_order(t, n);
                return *this;
            }

            // Array of non-arithmetic types and bools
            // E.g. bool, enum, pointers, user-defined types, non-PODs
            template<class T, size_t n> inline
            typename boost::enable_if_c<
                boost::mpl::or_<
                    boost::mpl::not_< boost::is_arithmetic<T> >,
                    boost::is_same<T, bool>
                    >::value,
                DeserializationInterface &>::type
            operator>>(T (&t)[n])
            {
                for( size_t i = 0; i < n; ++i )
                    *this >> t[i];
                return *this;
            }

            // Everything else...
            // (This allows non-intrusive serialization)
            template<typename T> inline
            typename boost::enable_if_c<
                boost::mpl::not_<
                    boost::mpl::or_<
                        has_member_load<T>,
                        boost::is_polymorphic<T>,
                        boost::is_arithmetic<T>,
                        boost::is_enum<T>,
                        boost::is_pointer<T> >
                    >::value,
                DeserializationInterface &>::type
            operator>>(T &t)
            {
                boost::uint8_t version;
                *this >> version;
                load(*this, t, version);
                return *this;
            }

        private:
            DeserializationBuffer &m_buf;

            bool m_buffer_is_big_endian;
        };
    }


    using serialization::DeserializationInterface;

    inline DeserializationInterface &operator>>(
        DeserializationInterface &ar, std::string &str)
    {
        boost::uint32_t len;
        ar >> len;
        if( len > 0 )
        {
            char *buf = new char[len+1];
            ar.load_binary(buf, len);
            buf[len] = 0;
            str = buf;
            delete[] buf;
        }
        else
            str = "";
        return ar;
    }
}


#endif // PEEKABOT_SERIALIZATION_DESERIALIZATION_INTERFACE_HH_INCLUDED
