/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_TRI_MESH_ENTITY_HH_INCLUDED
#define PEEKABOT_RENDERER_TRI_MESH_ENTITY_HH_INCLUDED


#include "../CullableEntity.hh"
#include "../VertexBased.hh"

#include <cstddef>
#include <vector>
#include <boost/cstdint.hpp>
#include <Eigen/Core>


namespace peekabot
{
    namespace renderer
    {
        class TriMesh;

        class TriMeshEntity :
            public CullableEntity,
            public VertexBased
        {
        public:
            typedef std::vector<boost::uint32_t> Indices;


            TriMeshEntity();

            TriMeshEntity(const TriMeshEntity &other);

            virtual ~TriMeshEntity();

            virtual TriMeshEntity *clone() const;

            virtual void get_renderables(PrepareRenderContext &context) const;

            void set_indices(const Indices &indices);

            void add_indices(const Indices &new_indices);

            void get_indices(Indices &indices) const;

            void toggle_double_sided(bool double_sided);

            bool is_double_sided() const { return m_double_sided; }

            virtual void set_vertices(
                const Vertices &vertices);

            virtual void add_vertices(
                const Vertices &new_vertices,
                const Vertices &all_vertices);

            virtual void get_vertices(
                Vertices &vertices) const;

            virtual void set_colored_vertices(
                const Vertices &vertices,
                const VertexColors &colors);

            virtual void add_colored_vertices(
                const Vertices &new_vertices,
                const Vertices &all_vertices,
                const VertexColors &new_colors,
                const VertexColors &all_colors);

            std::size_t vertex_count() const;

            std::size_t tri_count() const;

        private:
            static void update_normals(
                const boost::uint32_t *indices, std::size_t n_indices,
                const Eigen::Vector3f *vertices, std::size_t n_vertices,
                Eigen::Vector3f *ff_normals,
                Eigen::Vector3f *bf_normals);

            /**
             * \brief Set the entity's vertices without clearing the color
             * pointers.
             */
            void set_vertices_helper(const Vertices &vertices);

            void add_vertices_helper(
                const Vertices &new_vertices, const Vertices &all_vertices);

        private:
            /**
             * \brief Set to true if and only if the tri-mesh should be
             * rendered as double sided.
             *
             * When this property is set to true, triangles of the opposite
             * direction will be automatically generated and draw for all added
             * triangles.
             */
            bool m_double_sided;

            /**
             * \brief The front-face mesh.
             */
            boost::shared_ptr<TriMesh> m_ff_mesh;

            /**
             * \brief The back-face mesh.
             */
            boost::shared_ptr<TriMesh> m_bf_mesh;

            /**
             * \brief The largest index in the index buffer, used to determine
             * when it's safe to draw the index buffer (when there are
             * m_max_index-1 vertices).
             */
            boost::int64_t m_max_index;
        };
    }
}


#endif // PEEKABOT_RENDERER_TRI_MESH_ENTITY_HH_INCLUDED
