/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_2D_HH_INCLUDED
#define PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_2D_HH_INCLUDED


#include <set>
#include <map>
#include <vector>
#include <stdexcept>
#include <Eigen/Core>
#include <boost/cstdint.hpp>

#include "../../Types.hh"
#include "../CullableEntity.hh"


namespace peekabot
{
    namespace renderer
    {
        class Camera;
        class Frustum;


        class OccupancyGrid2D : public CullableEntity
        {
        public:
            OccupancyGrid2D(float cell_size, size_t max_cache_size);

            ~OccupancyGrid2D() throw();

            void set_belief(const Eigen::Vector2f &x, float belief)
                throw(std::domain_error);

            void set_cells(
                const std::vector<std::pair<Eigen::Vector2f, float> > &cells)
                throw(std::domain_error);

            float get_belief(const Eigen::Vector2f &x)
                throw(std::runtime_error);

            void clear_cell(const Eigen::Vector2f &x) throw();

            void clear() throw();

            virtual void get_renderables(PrepareRenderContext &context) const;

            void set_unoccupied_color(const RGBColor &color);

            void set_occupied_color(const RGBColor &color);

        private:
            class Batch
            {
            public:
                Batch(PrepareRenderContext &context, size_t batch_size);

                ~Batch();

                void add_quad(
                    double x_c, double y_c,
                    double size,
                    boost::uint8_t r, boost::uint8_t g, boost::uint8_t b);

            private:
                void flush();

                void allocate_buffers();

            private:
                PrepareRenderContext &m_context;
                const size_t m_batch_size;
                size_t m_used;
                float *m_vertices;
                boost::uint8_t *m_colors;
            };

            class Node
            {
            public:
                Node(Node *parent);

                ~Node();

                inline bool is_leaf() const throw()
                {
                    return m_children == 0;
                }

                uint8_t get_belief(
                    const Eigen::Vector2f &x,
                    const Eigen::Vector2f &center,
                    float node_size) throw();

                void set_belief(
                    const Eigen::Vector2f &x,
                    uint8_t belief,
                    const float cell_size,
                    const Eigen::Vector2f &center,
                    float node_size) throw(std::domain_error);

                inline Node *get_parent() throw()
                {
                    return m_parent;
                }

                inline const Node *get_parent() const throw()
                {
                    return m_parent;
                }

                /**
                 * \brief Get a child quadrant's position relative to its
                 * parent.
                 *
                 * \pre \f$ 0 \leq i < 4 \f$
                 *
                 * \param quadrant The quadrant for which to get the offset
                 * from its parent node.
                 * \return A vector describing the offset of node i from its parent.
                 *
                 * \sa \c m_children (for quadrant indexing).
                 */
                inline Eigen::Vector2f quadrant_offset(
                    int i, float node_size) const throw()
                {
                    return Eigen::Vector2f(
                        node_size/4*(2*(i&1)-1),
                        node_size/4*((i&2)-1) );
                }

                /**
                 * \brief Check if a coordinate is enclosed in the
                 * quad-tree node.
                 *
                 * \return \c true if the node encloses the given coordinate,
                 *         \c false otherwise.
                 */
                bool encloses(
                    const Eigen::Vector2f &x,
                    const Eigen::Vector2f &center,
                    float node_size) const throw();

                /**
                 * \brief Check if a coordinate is enclosed the node's
                 *        <em>i</em>:th quadrant.
                 *
                 * \pre The coordinate is enclosed in the node, i.e.
                 * <tt>encloses(x) == true</tt>.
                 *
                 * \param i The quadrant to check against, in the range 0
                 *          to 3.
                 * \return \c true if the coordinate is enclosed by the
                 *         node's <em>i</em>:th quadrant, \c false
                 *         otherwise.
                 *
                 * \remarks Note that this doesn't check if the coordinare
                 * is enclosed by a child node, or even demands that any
                 * child nodes exist. \sa encloses(float, float),
                 * m_children (for quadrant indexing).
                 */
                inline bool enclosed_in_quadrant(
                    const Eigen::Vector2f &x, int i,
                    const Eigen::Vector2f &center) const throw()
                {
                    return ( ( i&1 ? x(0) >= center(0) : x(0) < center(0) ) &&
                             ( i&2 ? x(1) >= center(1) : x(1) < center(1) ) );
                }

                void render_no_cull(
                    Batch &batch,
                    const RGBColor &color_base,
                    const RGBColor &color_slope,
                    const Eigen::Vector2f &center,
                    float node_size) throw();

                void render(
                    Batch &batch,
                    const Camera &camera,
                    const RGBColor &color_base,
                    const RGBColor &color_slope,
                    const Eigen::Vector2f &center,
                    float node_size,
                    const Frustum &frustum,
                    int plane_mask) throw();

            private:
                /**
                 * \brief Optimize the storage by merging neighbouring nodes
                 * if possible.
                 *
                 * \pre \a node is a leaf node.
                 */
                static void optimize(Node *node);

            public:
                Node *m_parent;

                /**
                 * \brief Pointers to the child nodes of the node.
                 *
                 * \invariant m_children == 0 \f$\Leftrightarrow\f$ the
                 * node is a leaf.
                 */
                Node **m_children;

                uint8_t m_belief;
            };

        private:
            Node *grow_to_accomodate(const Eigen::Vector2f &x) throw();

            void calculate_bvs() throw();

        private:


            /**
             * \brief The root node of the quad-tree.
             * \invariant Always non-null (except in the constructor).
             * \remarks May change over time (if the tree is enlarged).
             */
            Node *m_root_node;

            float m_cell_size;

            float m_root_cell_size;

            Eigen::Vector2f m_root_cell_center;

            RGBColor m_unoccupied_color;

            RGBColor m_occupied_color;

            size_t m_max_cache_size;
        };
    }
}


#endif // PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_2D_HH_INCLUDED
