/*
 * Copyright Staffan Gimåker 2007-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cassert>
#include <cmath>

#include "MeshBased.hh"
#include "Camera.hh"
#include "../PrepareRenderContext.hh"


using namespace peekabot;
using namespace peekabot::renderer;


MeshBased::MeshBased(uint8_t lod_count, 
                     const BoundingSphere &bsphere, 
                     float hysteresis) throw()
    : CullableEntity(bsphere),
      m_n_lods(lod_count),
      m_curr_lod(0),
      m_hysteresis((uint8_t)roundf(100-hysteresis*100)),
      m_lods(new LOD[m_n_lods])
{
}

MeshBased::~MeshBased()
{
    delete[] m_lods;
}

MeshBased::MeshBased(const MeshBased &x) throw()
    : CullableEntity(x.get_bounding_sphere_lc()),
      m_n_lods(x.m_n_lods),
      m_curr_lod(0),
      m_hysteresis(x.m_hysteresis),
      m_lods(new LOD[m_n_lods])
{
    for( int i = 0; i < m_n_lods; ++i )
        set_lod(i, x.m_lods[i].m_threshold, x.m_lods[i].m_mesh);
}

MeshBased *MeshBased::clone() const throw()
{
    return new MeshBased(*this);
}

void MeshBased::set_lod(uint8_t lod_no, uint16_t coverage, const Mesh *mesh) throw()
{
    assert( lod_no < m_n_lods );
    
    if( m_lods[lod_no].m_mesh != 0 )
        delete m_lods[lod_no].m_mesh;
    
    m_lods[lod_no].m_threshold = coverage;
    m_lods[lod_no].m_mesh = mesh->clone();
}

void MeshBased::get_renderables(PrepareRenderContext &context) const
{  
    // Vertical screen coverage, in pixels
    float cov;
    float r = get_bounding_sphere_wc().get_radius();

    const Camera *camera = context.get_camera();
    
    if( camera->is_orthographic() )
    {
        // In this case, the coverage depends only on zoom distance and the
        // object's size - and the viewport height, of course.
        //
        // Note that it *does not* depend on viewer-object distance!
        
        float d = camera->get_zoom_distance();
        float half_h = d*tanf(camera->get_fov_rad()/2);

        cov = camera->get_viewport_height() * (r / half_h);
    }
    else
    {
        // Calculate pp the vector between the viewer and the object position,
        // projected on the view vectorx
        Eigen::Vector3f pp(get_bounding_sphere_wc().get_pos() -
                           camera->get_viewer_mtow().translation());
        Eigen::Vector3f vv(camera->get_transformation().linear().col(0));
        pp = pp.dot(vv) * vv;

        // The x-distance between the camera view point and the object
        float dist = pp.norm();

        cov = camera->get_viewport_height() * (
            r * camera->get_near_plane()/dist / 
            (camera->get_near_plane()*tanf(camera->get_fov()/2/180*M_PI)));
    }
    
    if( cov < m_lods[m_curr_lod].m_threshold*m_hysteresis/100 )
        while( cov < m_lods[m_curr_lod].m_threshold*m_hysteresis/100 && m_curr_lod > 0 )
            --m_curr_lod;
    else if( m_curr_lod < m_n_lods-1 && cov >= m_lods[m_curr_lod+1].m_threshold )
        while( m_curr_lod < m_n_lods-1 && cov >= m_lods[m_curr_lod+1].m_threshold )
            ++m_curr_lod;
    
    assert( m_curr_lod < m_n_lods );
    assert( m_lods[m_curr_lod].m_mesh != 0 );
    
    context.prepare(m_lods[m_curr_lod].m_mesh);
}

