/*
 * Copyright Staffan Gimåker 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_TEXTURE_HH_INCLUDED
#define PEEKABOT_RENDERER_TEXTURE_HH_INCLUDED


#include <GL/glew.h>
#include <boost/utility.hpp>

#include "../Types.hh"


namespace peekabot
{
    namespace renderer
    {
        class Texture : public boost::noncopyable
        {
            /**
             * \brief The texture name.
             *
             * As given by and fed back into OpenGL to reference the texture.
             *
             * \invariant <tt>m_tex_name != 0</tt>.
             *
             * \see Man pages for \c glGenTextures on the subject of texture names.
             */
            GLuint m_tex_name;
                
            /**
             * \brief Keeps track of the currently bound texture.
             *
             * The initial value is 0, a reserved texture name denoting that 
             * no texture is bound.
             */
            static GLuint ms_bound_tex;

        public:
            Texture();

            Texture(int32_t w, int32_t h, GLenum format, void *data);

            virtual ~Texture();

            void set_data(int32_t w, int32_t h, GLenum format, void *data);

            /**
             * \brief Bind the texture.
             *
             * Binding the texture will cause it to be used when rendering
             * subsequent geometry.
             *
             * If the texture is invalid, this operation will silently fail.
             *
             * \e Calls: \c glBindTexture
             *
             * \pre \c is_valid() is \c true.
             *
             * \remarks Note that only one texture can be bound at a time, 
             * binding multiple textures subsequently will automatically unbind all
             * but the last bound texture.
             */
            void bind() const throw();


            /**
             * \brief Unbind a bound texture.
             *
             * Calling this will result in that subsequently drawn geometry will not
             * be affected by the previously bound texture.
             * Does nothing if the texture is unbound.
             *
             * \e Calls: \c glBindTexture, \c is_bound()
             */
            void unbind() const throw();

            /**
             * \brief Checks if the texture is currently bound.
             *
             * \return \c true if the texture is currently bound, \c false 
             * otherwise.
             */
            bool is_bound() const throw();
        };
    }
}


#endif // PEEKABOT_RENDERER_TEXTURE_HH_INCLUDED
