/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Tessellator.hh"
#include "../MessageHub.hh"

#include <cassert>


using namespace peekabot;
using namespace peekabot::renderer;


Tessellator::Tessellator(
    const EmitFaceCallback &emit_callback,
    const CombineCallback &combine_callback)
    : m_tess(gluNewTess()),
      m_idx(0),
      m_emit_callback(emit_callback),
      m_combine_callback(combine_callback)
{
    gluTessProperty(m_tess, GLU_TESS_BOUNDARY_ONLY, GL_FALSE);
    gluTessProperty(m_tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ODD);

    gluTessCallback(
        m_tess, GLU_TESS_BEGIN_DATA,
        reinterpret_cast<GluTessCallbackType>(
            &Tessellator::begin_callback));

    gluTessCallback(
        m_tess, GLU_TESS_END_DATA,
        reinterpret_cast<GluTessCallbackType>(
            &Tessellator::end_callback));

    gluTessCallback(
        m_tess, GLU_TESS_VERTEX_DATA,
        reinterpret_cast<GluTessCallbackType>(
            &Tessellator::tess_callback));

    gluTessCallback(
        m_tess, GLU_TESS_COMBINE_DATA,
        reinterpret_cast<GluTessCallbackType>(
            &Tessellator::combine_callback));

    gluTessCallback(
        m_tess, GLU_TESS_ERROR,
        reinterpret_cast<GluTessCallbackType>(
            &Tessellator::error_callback));
}


Tessellator::~Tessellator()
{
    gluDeleteTess(m_tess);

    for( size_t i = 0; i < m_verts.size(); ++i )
        delete m_verts[i];
    m_verts.clear();
}


void Tessellator::begin_polygon()
{
    m_contour_count = 0;
    gluTessBeginPolygon(m_tess, this);
}


void Tessellator::end_polygon()
{
    if( m_contour_count > 0 )
        end_contour();

    gluTessEndPolygon(m_tess);

    for( size_t i = 0; i < m_verts.size(); ++i )
        delete m_verts[i];
    m_verts.clear();
}


void Tessellator::begin_contour()
{
    if( m_contour_count > 0 )
        end_contour();
    ++m_contour_count;

    gluTessBeginContour(m_tess);
}


void Tessellator::end_contour()
{
    --m_contour_count;
    gluTessEndContour(m_tess);
}


void Tessellator::vertex(float x, float y, float z)
{
    GLdouble coords[] = { x, y, z };

    Vertex *v = new Vertex;
    v->vertex = Eigen::Vector3f(x,y,z);
    v->index = m_idx++;
    m_verts.push_back(v);

    assert( v );
    gluTessVertex(m_tess, coords, (void *)v);
}


void Tessellator::triangulate_fan(const std::vector<Vertex *> &v)
{
    assert( v.size() > 2 );
    size_t n = v.size()-1;
    for( size_t i = 1; i < n; ++i )
    {
        if( is_ccw(v[0], v[i], v[i+1]) )
        {
            m_emit_callback(
                v[0]->index, v[i]->index, v[i+1]->index,
                v[0]->vertex, v[i]->vertex, v[i+1]->vertex);
        }
        else
        {
            m_emit_callback(
                v[0]->index, v[i+1]->index, v[i]->index,
                v[0]->vertex, v[i+1]->vertex, v[i]->vertex);
        }
    }
}

void Tessellator::triangulate_strip(const std::vector<Vertex *> &v)
{
    assert( v.size() > 2 );
    size_t n = v.size()-1;
    for( size_t i = 1; i < n; ++i )
    {
        if( is_ccw(v[i-1], v[i], v[i+1]) )
        {
            m_emit_callback(
                v[i-1]->index, v[i]->index, v[i+1]->index,
                v[i-1]->vertex, v[i]->vertex, v[i+1]->vertex);
        }
        else
        {
            m_emit_callback(
                v[i-1]->index, v[i+1]->index, v[i]->index,
                v[i-1]->vertex, v[i+1]->vertex, v[i]->vertex);
        }
    }
}

void Tessellator::triangulate_triangles(const std::vector<Vertex *> &v)
{
    assert( v.size() > 2 );
    assert( v.size() % 3 == 0 );
    for( size_t i = 0; i < v.size(); i += 3 )
    {
        if( is_ccw(v[i+0], v[i+1], v[i+2]) )
        {
            m_emit_callback(
                v[i]->index, v[i+1]->index, v[i+2]->index,
                v[i]->vertex, v[i+1]->vertex, v[i+2]->vertex);
        }
        else
        {
            m_emit_callback(
                v[i]->index, v[i+2]->index, v[i+1]->index,
                v[i]->vertex, v[i+2]->vertex, v[i+1]->vertex);
        }
    }
}


void Tessellator::tess_callback(Vertex *vertex_data, void *polygon_data)
{
    Tessellator *tess = (Tessellator *)polygon_data;
    tess->m_q.push_back(vertex_data);
}


void Tessellator::combine_callback(
    GLdouble coords[3],
    Vertex *vertex_data[4],
    GLfloat weight[4],
    void **outdata,
    void *polygon_data)
{
    Tessellator *tess = (Tessellator *)polygon_data;

    Vertex *v = new Vertex;
    v->vertex = Eigen::Vector3f(coords[0], coords[1], coords[2]);

    // Even though that [1] states that:
    //
    //   All vertex pointers are valid even when some of the weights are 0.
    //   coords gives the location of the new vertex.
    //
    // This is apparently not always the case for all GLU implementations. So
    // we need to work around this somehow.
    //
    //  [1]: http://www.opengl.org/sdk/docs/man/xhtml/gluTessCallback.xml

    assert( vertex_data[0] );
    assert( vertex_data[1] );

    std::size_t idx1 = vertex_data[0]->index;
    std::size_t idx2 = vertex_data[1]->index;
    std::size_t idx3, idx4;

    if( vertex_data[2] )
    {
        idx3 = vertex_data[2]->index;
    }
    else
    {
        idx3 = idx1;
        assert( weight[2] == 0 );
    }

    if( vertex_data[3] )
    {
        idx4 = vertex_data[3]->index;
    }
    else
    {
        idx4 = idx1;
        assert( weight[3] == 0 );
    }

    v->index = tess->m_combine_callback(
        Eigen::Vector3f(coords[0], coords[1], coords[2]),
        idx1, idx2, idx3, idx4,
        weight);

    tess->m_verts.push_back(v);

    *outdata = (void *)v;
}


void Tessellator::begin_callback(GLenum type, void *polygon_data)
{
    Tessellator &tess = *(Tessellator *)polygon_data;
    tess.m_type = type;
}


void Tessellator::end_callback(void *polygon_data)
{
    Tessellator &tess = *(Tessellator *)polygon_data;

    if( tess.m_type == GL_TRIANGLES )
        tess.triangulate_triangles(tess.m_q);
    else if( tess.m_type == GL_TRIANGLE_FAN )
        tess.triangulate_fan(tess.m_q);
    else if( tess.m_type == GL_TRIANGLE_STRIP )
        tess.triangulate_strip(tess.m_q);
    else
        assert( false );

    tess.m_q.clear();
}


void Tessellator::error_callback(GLenum _errno)
{
    TheMessageHub::instance().publish(
        ERROR_MESSAGE,
        "Tessellation error: %s (%s:%u)",
        gluErrorString(_errno), __FILE__, __LINE__);
}
