/*
 * Copyright Staffan Gimåker 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_STATE_HH_INCLUDED
#define PEEKABOT_RENDERER_STATE_HH_INCLUDED


#include <set>
#include <map>
#include <stack>
#include <boost/shared_ptr.hpp>
#include <exception>

#include "Statelet.hh"



namespace peekabot
{
namespace renderer
{


    // Possible extensions: Template states
    class State
    {
    public:
        typedef std::set<StateletType> OverrideSet;

        typedef std::pair<State, OverrideSet> Baseline;


        class StateletNotSet : public std::exception {};


        State() throw();

        State(const State &state) throw();

        State &operator=(const State &state) throw();


        //State clone() const throw();

        
        template<typename T> void set(boost::shared_ptr<T> s) throw()
        {
            if( !s )
            {
                m_statelets.erase(T::get_type());
            }
            else
            {
                m_statelets[s->_get_type()] = s;
            }
        }

        // note: takes ownership!
        void set(Statelet *s) throw();

        template<typename T> boost::shared_ptr<T> get() throw()
        {
            if( !is_set<T>() )
                set(new T);

            return boost::dynamic_pointer_cast<T>(m_statelets[T::get_type()]);
        }

        template<typename T> const boost::shared_ptr<T> get() const throw(StateletNotSet)
        {
            if( !is_set<T>() )
                throw StateletNotSet();

            return boost::dynamic_pointer_cast<T>(
                m_statelets.find(T::get_type())->second);

        }

        bool is_set(StateletType type) const throw()
        {
            Statelets::const_iterator it = m_statelets.find(type);
            return ( it != m_statelets.end() && it->second );
        }

        template<typename T> bool is_set() const throw()
        {
            return is_set(T::get_type());
        }

        template<typename T> void unset() throw()
        {
            unset(T::get_type());
        }

        void unset(StateletType type) throw()
        {
            m_statelets.erase(type);
        }


        void apply(const State *template_state = 0) const throw();


        //boost::shared_ptr<Statelet> &operator[](StateletType type) throw();

        // We want to sort on state merged with the BASELINE state... hmm...
        /*static bool less_than(
            const State &a, const State &s, 
            const std::vector<StateletType> &sort_order) throw()
        {
            const State &baseline = State::get_baseline();

            for( size_t i = 0; i < sort_order.size(); i++ )
            {
                StateletType type = sort_order[i];

                const boost::shared_ptr<Statelet> *s1 = &a.m_statelets[type];
                const boost::shared_ptr<Statelet> *s2 = &s.m_statelets[type];

                if( !(*s1) )
                    s1 = &baseline.m_statelets[type];
                if( !(*s2) )
                    s2 = &baseline.m_statelets[type];

                if( *s1 != *s2 && *s1 != 0 && *s2 != 0 && (*s1)->less_than(*s2) )
                        return true;
            }

            return false;
        }*/



        static void set_baseline(const State &state, const OverrideSet &os = State::get_override_set()) throw();

        static void push_baseline() throw();

        static void pop_baseline() throw();

        static const std::pair<State, OverrideSet> &get_baseline() throw();

        static const State &get_baseline_state() throw();

        static const OverrideSet &get_override_set() throw();

        /**
         * \brief Unconditionally apply the current baseline state and update 
         * the baseline delta.
         */
        static void apply_baseline() throw();           


    private:
        typedef std::stack<Baseline> BaselineStack;
        typedef std::set<StateletType> BaselineDelta;

        typedef std::map<StateletType, boost::shared_ptr<Statelet> > Statelets;


        Statelets m_statelets;


       


        static Baseline ms_baseline;

        static BaselineStack ms_baseline_stack;


        static State ms_current_state;

        static BaselineDelta ms_baseline_delta;
    };


    template<> void State::set<Statelet>(boost::shared_ptr<Statelet> s) throw();

}
}

#endif // PEEKABOT_RENDERER_STATE_HH_INCLUDED
