/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_RENDERER_RESOURCE_MANAGER_HH
#define __PEEKABOT_RENDERER_RESOURCE_MANAGER_HH


#include <map>
#include <set>
#include <string>
#include <stdexcept>
#include <iostream>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>

#include "AbstractProvider.hh"


namespace peekabot
{
    namespace renderer
    {

        template<class T, class U = std::string>
        class ResourceManager
        {
        public:
            typedef T ResourceType;
            typedef U ResourceIdentifier;
            typedef AbstractProvider<ResourceType, ResourceIdentifier> Provider;

        private:
            typedef std::map<ResourceIdentifier, boost::weak_ptr<ResourceType> > ResourceMap;
            typedef std::set<Provider *> Providers;

            ResourceMap m_resources;

            Providers m_providers;
        private:
            void purge() throw()
            {
                for( typename ResourceMap::iterator it = m_resources.begin();
                     it != m_resources.end(); )
                {
                    if( it->second.expired() )
                        m_resources.erase(it++);
                    else
                        ++it;
                }
            }

        public:
            virtual ~ResourceManager()
            {
                // Delete all providers
                for( typename Providers::iterator it = m_providers.begin(); 
                     it != m_providers.end(); ++it )
                    delete *it;
                m_providers.clear();
            }

            bool is_cached(const ResourceIdentifier &id) const throw()
            {
                typename ResourceMap::const_iterator it = m_resources.find(id);

                return it != m_resources.end() && !it->second.expired();
            }

            boost::shared_ptr<ResourceType> get(const ResourceIdentifier &id)
                throw(std::runtime_error)
            {
                // Purge stale, deleted resources
                purge();

                // Is it already loaded?
                typename ResourceMap::iterator it = m_resources.find(id);
                if( it != m_resources.end() )
                {
                    boost::shared_ptr<ResourceType> ptr = it->second.lock();
                    if( ptr )
                        return ptr;
                }

                // Look for a provider that can provide us with the 
                // wanted resource
                if( m_providers.empty() )
                    throw std::runtime_error("No resource providers available");

                std::string err_msg = "No matching provider found";

                for( typename Providers::iterator it = m_providers.begin(); 
                     it != m_providers.end(); ++it )
                {
                    try
                    {
                        if( (*it)->matches(id) )
                        {
                            boost::shared_ptr<ResourceType> res = (*it)->get(id);
                            m_resources[id] = res;
                            return res;
                        }
                    }
                    catch(std::exception &e)
                    {
                        err_msg = ("Exception caught while getting resource "
                                   "from provider: ");
                        err_msg += e.what();
                    }
                    catch(...)
                    {
                        err_msg = ("Unknown exception caught while getting "
                                   "resource from provider");
                    }
                }

                throw std::runtime_error(err_msg);
            }

            void add_provider(Provider *provider) throw()
            {
                m_providers.insert(provider);
            }
        };

    }
}


#endif // __PEEKABOT_RENDERER_RESOURCE_MANAGER_HH
