/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_RENDER_TRAVERSER_HH_INCLUDED
#define PEEKABOT_RENDERER_RENDER_TRAVERSER_HH_INCLUDED


#include <set>
#include <vector>
#include <boost/cstdint.hpp>

#include "State.hh"
#include "SceneTraverser.hh"
#include "PrepareRenderContext.hh"
#include "RenderContext.hh"


namespace peekabot
{
    namespace renderer
    {
        class Camera;
        class Renderable;
        class Entity;
        class ObjectEntity;

        class RenderTraverser : public PrepareRenderContext,
                                public SceneTraverser
        {
            class Batch
            {
                struct Item
                {
                    Item(const Renderable *r,
                         const State *template_state,
                         const State *override_state)
                        : m_renderable(r),
                          m_template_state(template_state),
                          m_override_state(override_state) {}

                    const Renderable *m_renderable;
                    const State      *m_template_state;
                    const State      *m_override_state;

                private:
                    Item();
                };

            public:
                ~Batch();

                inline void add(
                    const Renderable *r,
                    const State *template_state)
                {
                    m_items.push_back(Item(r, template_state, 0));
                }

                inline void add(
                    const Renderable *r,
                    const State *template_state,
                    const State &override_state)
                {
                    m_items.push_back(
                        Item(r, template_state, new State(override_state)));
                }

                void render(RenderContext &context, bool front_to_back);

            private:
                typedef std::vector<Item> Items;
                Items m_items;
            };

        public:
            RenderTraverser(
                const Camera &camera,
                const bool *visible_layers);

            virtual ~RenderTraverser();


            virtual void traverse(const Entity *entity);

            virtual void prepare(const Renderable *r, bool assume_ownership);


            void render_opaque() throw();

            void render_translucent() throw();

            /**
             * \brief Render the "selected effect" outlines of the objects
             * rendered in a previous call to render_pvs().
             *
             * \pre render_pvs() has been executed, on the same object, with
             * success prior to calling this method.
             */
            void render_outlines(
                const RGBColor &selected_color,
                const RGBColor &ancestor_selected_color) throw();

            void render_always_on_top() throw();

            void render_tripods() throw();

        private:
            void handle_renderable(
                const Renderable *renderable,
                const ObjectEntity *object);

            void render_batch(Batch &batch, bool front_to_back) throw();

            void make_tripod(const ObjectEntity *object);

        private:
            const bool *m_visible_layers;

            const ObjectEntity *m_entity_tmp;

        private:
            Batch               m_opaque;
            Batch               m_translucent;
            Batch               m_opaque_on_top;
            Batch               m_translucent_on_top;
            Batch               m_selected;
            Batch               m_ancestor_selected;

            std::set<const Renderable *> m_delete_set;

            RenderContext m_context;

            std::vector<float> m_tripod_verts;
            std::vector<boost::uint8_t> m_tripod_colors;
        };
    }
}


#endif // PEEKABOT_RENDERER_RENDER_TRAVERSER_HH_INCLUDED
