/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "PngLoader.hh"

#include <GL/glew.h>
#include <boost/xpressive/xpressive.hpp>
#include <boost/scoped_array.hpp>
#include <gtkmm.h>
#include <stdexcept>
#include <cassert>


using namespace peekabot;
using namespace peekabot::renderer;


PngLoader::Informer PngLoader::ms_informer;


boost::shared_ptr<Texture>
PngLoader::get(const std::string &id)
{
    Glib::RefPtr<Gdk::Pixbuf> img = Gdk::Pixbuf::create_from_file(id);

    if( img->get_n_channels() < 3 )
        throw std::runtime_error(
            "Only PNG files with 3 or 4 channels are supported");

    // TODO: We don't really need to make this extra copy, we could just
    // use glPixelStorei(GL_PACK_ALIGNMENT, alignment) to tell inform OpenGL
    // about the correct rowstride...

    int w = img->get_width();
    int h = img->get_height();
    int rowstride = img->get_rowstride();
    int pixelstride = img->get_n_channels();
    assert( pixelstride >= 3 && pixelstride <= 4 );
    boost::uint8_t *pixels = img->get_pixels();

    boost::scoped_array<uint8_t> data(new uint8_t[3*w*h]);

    for( int j = 0; j < h; ++j )
    {
        for( int i = 0; i < w; ++i )
        {
            // PNG uses RGBA (I hope :)
            data[3*(w*j+i)+0] = pixels[rowstride*j + i*pixelstride + 0];
            data[3*(w*j+i)+1] = pixels[rowstride*j + i*pixelstride + 1];
            data[3*(w*j+i)+2] = pixels[rowstride*j + i*pixelstride + 2];
        }
    }

    return boost::shared_ptr<Texture>(
        new Texture(w, h, GL_RGB, data.get()));
}

bool PngLoader::matches(const std::string &id) throw()
{
    static const boost::xpressive::sregex match_exp =
        boost::xpressive::sregex::compile(".*\\.png");
    return boost::xpressive::regex_match(id, match_exp);
}
