/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Font.hh"
#include "Tessellator.hh"

#include <ft2build.h>
#include FT_OUTLINE_H

#include <cassert>
#include <stdexcept>
#include <iostream>
#include <boost/bind.hpp>
#include <boost/ref.hpp>


using namespace peekabot;
using namespace peekabot::renderer;


unsigned int Font::ms_ft_use_count = 0;
FT_Library Font::ms_ft_library;


namespace
{
    // Simple linear interpolation between two points a and b.
    //   t=0 => a is returned
    //   t=1 => b is returned
    inline Eigen::Vector2f lerp(
        const Eigen::Vector2f &a,
        const Eigen::Vector2f &b,
        float t)
    {
        // a + (b-a)*t = a + bt - at = a(1-t) + bt
        return (1-t)*a + t*b;
    }

    float point_line_dist_2d_sq(
        const Eigen::Vector2f &P,
        const Eigen::Vector2f &Q, // Point on the line
        const Eigen::Vector2f &v) // unit vector parallell to the line
    {
        // dist(P,L) = norm(P-Q)^2 - ((P-Q) dot v)^2
        //           = { pq = P-Q } = pq_0^2+pq_1^2 - (pq_0*v_0 + pq_1*v_1)^2
        Eigen::Vector2f PQ = P-Q;
        float PQv = PQ.dot(v);
        return PQ.dot(PQ) - PQv*PQv;
    }


    struct DecompositionData
    {
        Tessellator *tess;
        const Font *font;
    };


    //
    // Note: we can do better than using the distance from the control point to the line
    // as a stop heuristic for the subdivision, since this is not a time critical step.
    //
    // For quadratic curves, we can use a binary search to find the largest
    // distance in log-time. Cubic curves can have two maxima, so the technique is
    // not applicable there.
    //

    /*
     * \brief Subdivide the given bézier curve (a,b,c) until it's approximated
     * within the given tolerance.
     *
     * Uses de Casteljau's algorithm to recursively subdivide the bezier curve
     * into smaller bezier curves. When the tolerance is met, a curve is 
     * approximated by a straight line between the end points (a and c).
     *
     * The distance between the line (a,c) and the control b is the metric used
     * to determine when subdivision stops.
     *
     * \param a The starting point of the curve.
     * \param b The control point.
     * \param a The end point of the curve.
     * \param tolerance The tolerance in m to which the curve should be 
     * approximated.
     */
    void quadratic_bezier_subdivide_recursive(
        const Eigen::Vector2f &a,
        const Eigen::Vector2f &b,
        const Eigen::Vector2f &c,
        const float tolerance,
        Tessellator *tess)
    {
        Eigen::Vector2f v = a-c;
        v.normalize();
        float dist_sq = point_line_dist_2d_sq(b, a, v);

        if( dist_sq > tolerance*tolerance )
        {
            //
            // Use de Casteljau's algorithm to split the bezier curve into two 
            // smaller bezier curves: (a, ab, abbc) and (abbc, bc, c)
            //

            // ab, bc = Control points for the new, smaller curves
            Eigen::Vector2f ab = lerp(a, b, 0.5);
            Eigen::Vector2f bc = lerp(b, c, 0.5);
            // abbc = The bezier curve evaluated at t=0.5
            Eigen::Vector2f abbc = lerp(ab, bc, 0.5);

            quadratic_bezier_subdivide_recursive(a, ab, abbc, tolerance, tess);
            quadratic_bezier_subdivide_recursive(abbc, bc, c, tolerance, tess);
        }
        else
            tess->vertex(c(0), c(1), 0);
    }


    void cubic_bezier_subdivide_recursive(
        const Eigen::Vector2f &a,
        const Eigen::Vector2f &b,
        const Eigen::Vector2f &c,
        const Eigen::Vector2f &d,
        const float tolerance,
        Tessellator *tess)
    {
        Eigen::Vector2f v = a-d;
        v.normalize();

        float dist = (
            sqrt(point_line_dist_2d_sq(b, a, v)) +
            sqrt(point_line_dist_2d_sq(c, a, v)) ) / 2;
    
        if( dist > tolerance )
        {
            Eigen::Vector2f ab = lerp(a, b, 0.5);
            Eigen::Vector2f bc = lerp(b, c, 0.5);
            Eigen::Vector2f cd = lerp(c, d, 0.5);
            Eigen::Vector2f abbc = lerp(ab, bc, 0.5);
            Eigen::Vector2f bccd = lerp(bc, cd, 0.5);
            Eigen::Vector2f mid = lerp(abbc, bccd, 0.5);

            cubic_bezier_subdivide_recursive(a, ab, abbc, mid, tolerance, tess); // ????
            cubic_bezier_subdivide_recursive(mid, bccd, cd, d, tolerance, tess); // ????
        }
        else
            tess->vertex(d(0), d(1), 0);
    }






    int move_to_cb(const FT_Vector *to, void *user)
    {
        //std::cout << "Move to!" << std::endl;

        DecompositionData *dd = (DecompositionData *)user;
        Tessellator *tess = dd->tess;

        // The move_to command is emitted by FT_Decompose_Outline() when 
        // starting a new contour
        tess->begin_contour();
        tess->vertex(
            dd->font->fu26_6_to_m(to->x), dd->font->fu26_6_to_m(to->y), 0);

        return 0;
    }

    int line_to_cb(const FT_Vector *to, void *user)
    {
        //std::cout << "Line to!" << std::endl;

        DecompositionData *dd = (DecompositionData *)user;
        Tessellator *tess = dd->tess;

        tess->vertex(
            dd->font->fu26_6_to_m(to->x), dd->font->fu26_6_to_m(to->y), 0);

        return 0;
    }

    int conic_to_cb(
        const FT_Vector *control,
        const FT_Vector *to,
        void *user)
    {
        //std::cout << "Conic to!" << std::endl;

        DecompositionData *dd = (DecompositionData *)user;
        Tessellator *tess = dd->tess;

        quadratic_bezier_subdivide_recursive(
            Eigen::Vector2f(tess->last_vertex()(0), tess->last_vertex()(1)), 
            dd->font->fu26_6_to_m(Eigen::Vector2f((float)control->x, (float)control->y)),
            dd->font->fu26_6_to_m(Eigen::Vector2f((float)to->x, (float)to->y)),
            5e-5f, tess);

        return 0;
    }

    int cubic_to_cb(
        const FT_Vector *control1,
        const FT_Vector *control2,
        const FT_Vector *to,
        void *user)
    {
        //std::cout << "Cubic to!" << std::endl;

        DecompositionData *dd = (DecompositionData *)user;
        Tessellator *tess = dd->tess;

        cubic_bezier_subdivide_recursive(
            Eigen::Vector2f(tess->last_vertex()(0), tess->last_vertex()(1)),
            dd->font->fu26_6_to_m(Eigen::Vector2f((float)control1->x, (float)control1->y)),
            dd->font->fu26_6_to_m(Eigen::Vector2f((float)control2->x, (float)control2->y)),
            dd->font->fu26_6_to_m(Eigen::Vector2f((float)to->x, (float)to->y)),
            5e-5f, tess);

        return 0;
    }
}


Font::Font(const std::string &filename)
{
    if( ms_ft_use_count == 0 )
    {
        // Initialize the FreeType font library
        FT_Error error = FT_Init_FreeType( &ms_ft_library );

        if( error )
            throw std::runtime_error(
                "Failed to initialize FreeType font library");
    }

    ++ms_ft_use_count;

    // TODO: this is not exception safe! If the code below throws, the FT 
    // library will never be de-initialized!

    FT_Error error = FT_New_Face(ms_ft_library, filename.c_str(), 0, &m_face);
    
    if( error == FT_Err_Unknown_File_Format )
    {
        throw std::runtime_error(
            "The font file could be opened and read, but it appears ... that its font format is unsupported");
    }
    else if( error )
    {
        throw std::runtime_error(
            "another error code means that the font file could not ... be opened or read, or simply that it is broken...");
    }

    FT_Select_Charmap(m_face, FT_ENCODING_UNICODE);
}


Font::~Font()
{
    FT_Done_Face(m_face);

    if( --ms_ft_use_count == 0 )
    {
        FT_Done_FreeType(ms_ft_library);
    }
}


void Font::render_string(const std::string &str, float offset)
{
    bool use_kerning = FT_HAS_KERNING(m_face); 
    FT_UInt previous = 0;

    Eigen::Vector2f pen_pos(0,0);

    glTranslatef(offset, 0, 0);

    for( size_t i = 0; i < str.length(); ++i )
    {
        // convert character code to glyph index
        FT_UInt glyph_index = FT_Get_Char_Index(m_face, str[i]);
        // retrieve kerning distance and move pen position
        if( use_kerning && previous && glyph_index )
        {
            FT_Vector delta;
            FT_Get_Kerning( 
                m_face, previous, glyph_index, FT_KERNING_UNSCALED, &delta ); 
            pen_pos += Eigen::Vector2f( fu26_6_to_m(delta.x), fu26_6_to_m(delta.y) );

            glTranslatef(fu26_6_to_m(delta.x), fu26_6_to_m(delta.y), 0);

            //std::cout << "  kern delta=" << delta.x << ", " << delta.y << std::endl;
        }

        const GlyphData &gd = get_glyph_data(glyph_index);

        glBegin(GL_TRIANGLES);
        glNormal3f(0, 0, 1);
        for( size_t i = 0; i < gd.m_vertices.size(); i += 3 )
        {
            glVertex3f( gd.m_vertices[i+0], 
                        gd.m_vertices[i+1], 
                        gd.m_vertices[i+2] );
        }
        glEnd();

        glTranslatef(gd.m_hori_advance, 0, 0);

        pen_pos += Eigen::Vector2f(gd.m_hori_advance, 0.0f);

        // record current glyph index
        previous = glyph_index; 
    }
}


std::pair<float, float> Font::measure_string(
    const std::string &str) const throw()
{
    bool use_kerning = FT_HAS_KERNING(m_face); 
    FT_UInt previous = 0;

    std::pair<float, float> bb;
    bb.first = bb.second = 0;

    for( size_t i = 0; i < str.length(); ++i )
    {
        // convert character code to glyph index
        FT_UInt glyph_index = FT_Get_Char_Index(m_face, str[i]);
        // retrieve kerning distance and move pen position
        if( use_kerning && previous && glyph_index )
        {
            FT_Vector delta;
            FT_Get_Kerning( m_face, previous, glyph_index, FT_KERNING_UNSCALED, &delta ); 
            bb.first += fu26_6_to_m(delta.x);
        }

        const GlyphData &gd = get_glyph_data(glyph_index);

        bb.first += gd.m_hori_advance;
        bb.second = std::max(gd.m_height, bb.second);
            
        // record current glyph index
        previous = glyph_index;
    }

    return bb;
}


void Font::load_glyph(GlyphType glyph_index) const
{
    if( FT_Load_Glyph(m_face, glyph_index, FT_LOAD_NO_SCALE) )
        throw std::runtime_error("FT_Load_Glyph failed");

    FT_Outline_Funcs outline_funcs;

    // Initialize the data for FreeType to parse the outline
    outline_funcs.shift = 0;
    outline_funcs.delta = 0;
    outline_funcs.move_to = &move_to_cb;
    outline_funcs.line_to = &line_to_cb;
    outline_funcs.conic_to = &conic_to_cb;
    outline_funcs.cubic_to = &cubic_to_cb;

    /*if( inContext->enableState.glObjects )
    {
        inData->tolerance *= face->units_per_EM;
    }*/

    //std::cout << "units_per_EM: " << face->units_per_EM << std::endl;

    GlyphData gd;
    gd.m_hori_bearing_x = fu26_6_to_m(m_face->glyph->metrics.horiBearingX);
    gd.m_hori_bearing_y = fu26_6_to_m(m_face->glyph->metrics.horiBearingY);
    gd.m_hori_advance   = fu26_6_to_m(m_face->glyph->metrics.horiAdvance);
    gd.m_width          = fu26_6_to_m(m_face->glyph->metrics.width);
    gd.m_height         = fu26_6_to_m(m_face->glyph->metrics.height);

    Tessellator tess(
        boost::bind(
            &Font::emit_face_callback, boost::ref(gd.m_vertices), _4, _5, _6),
        &Font::combine_callback);
    tess.begin_polygon();

    DecompositionData dd;
    dd.tess = &tess;
    dd.font = this;

    if( FT_Outline_Decompose(&m_face->glyph->outline, &outline_funcs, &dd) )
        throw std::runtime_error("FT_Outline_Decompose failed!");

    tess.end_polygon();

    m_glyphs.insert( std::make_pair(glyph_index, gd) );
}

const Font::GlyphData &Font::get_glyph_data(GlyphType glyph) const
{
    // Retrieve the glyph geometry and metrics - extract it from 
    // the font file when it's not already loaded
    if( !has_glyph(glyph) )
        load_glyph(glyph);

    GlyphMap::const_iterator it = m_glyphs.find(glyph);
    assert( it != m_glyphs.end() );

    return it->second;
}


// Tessellation callback - called by Tessellator for each
// triangle output
void Font::emit_face_callback(
    std::vector<float> &verts,
    const Eigen::Vector3f &v1,
    const Eigen::Vector3f &v2,
    const Eigen::Vector3f &v3)
{
    verts.push_back(v1(0));
    verts.push_back(v1(1));
    verts.push_back(v1(2));

    verts.push_back(v2(0));
    verts.push_back(v2(1));
    verts.push_back(v2(2));

    verts.push_back(v3(0));
    verts.push_back(v3(1));
    verts.push_back(v3(2));
}


std::size_t Font::combine_callback(
    const Eigen::Vector3f &v,
    std::size_t idx1, std::size_t idx2,
    std::size_t idx3, std::size_t idx4,
    const float w[4])
{
    // We don't uses indices so it's safe to return a bogus value here
    return 0;
}
