/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ArrayBuffer.hh"
#include "Device.hh"

#include <GL/glew.h>
#include <boost/scoped_array.hpp>
#include <boost/cstdint.hpp>
#include <stdexcept>
#include <cassert>
#include <cstring>


using namespace peekabot;
using namespace peekabot::renderer;


ArrayBuffer::VBOHandle ArrayBuffer::ms_bound_index_buffer = 0;
ArrayBuffer::VBOHandle ArrayBuffer::ms_bound_vertex_buffer = 0;


ArrayBuffer::ArrayBuffer(bool force_no_vbos, bool index_buffer)
    : m_data(0),
      m_size(0),
      m_is_index_buffer(index_buffer),
      m_uses_vbos(
          !force_no_vbos &&
          (/*!TheConfiguration::instance().get_option<bool>(
            "renderer.disable_vbos", false) &&*/
           Device::has_vbo_support()) ),
      m_is_mapped(false),
      m_usage(STATIC_DRAW)
{
    if( m_uses_vbos )
    {
        glGenBuffersARB(1, &m_vbo_handle);
        CHECK_GL_ERRORS();
    }
}


ArrayBuffer::ArrayBuffer(const ArrayBuffer &other)
    : m_size(other.m_size),
      m_is_index_buffer(other.m_is_index_buffer),
      m_uses_vbos(other.m_uses_vbos),
      m_is_mapped(false),
      m_usage(other.m_usage)
{
    if( m_uses_vbos )
    {
        glGenBuffersARB(1, &m_vbo_handle);
        CHECK_GL_ERRORS();
    }

    if( m_size > 0 )
    {
        // *** Snip from the VBO extension docs: ***
        //
        // Pointer values returned
        // by MapBuffer may not be passed as parameter values to GL commands.
        // For example, they may not be used to specify array pointers, or to
        // specify or query pixel or texture image data; such actions produce
        // undefined results, although implementations may not check for such
        // behavior for performance reasons.
        //
        // Alas, mapping the buffer and copying it into a our new buffer is illegal
        // and we are forced to make an intermediary copy of the data. Sad but true.
        boost::scoped_array<boost::uint8_t> data(new boost::uint8_t[m_size]);
        other.get_data(data.get(), m_size, 0);
        set_data(data.get(), m_size);
    }
}


ArrayBuffer::~ArrayBuffer()
{
    if( m_uses_vbos )
    {
        if( is_mapped() )
            unmap();

        if( is_bound() )
            unbind();

        glDeleteBuffersARB(1, &m_vbo_handle);
        CHECK_GL_ERRORS();
    }
    else
    {
        if( m_data )
            delete[] m_data;
    }
}


void ArrayBuffer::set_usage_hint(UsageHint usage)
{
    m_usage = usage;
}


void ArrayBuffer::set_data(const void *data, std::size_t n)
{
    m_size = n;

    if( m_uses_vbos )
    {
        if( !is_bound() )
            bind();

        if( n > 0 )
        {
            glBufferDataARB(
                get_gl_buffer_type(),
                (GLsizeiptr)n, data,
                get_gl_usage_hint());

            if( glGetError() == GL_OUT_OF_MEMORY )
                throw std::runtime_error(
                    "ArrayBuffer::set_data() failed: out of memory");

            CHECK_GL_ERRORS();
        }
    }
    else
    {
        if( m_data )
        {
            delete[] m_data;
            m_data = 0;
        }

        if( n > 0 )
        {
            m_data = new boost::uint8_t[n];
            assert( m_data );
            if( data )
                std::memcpy(m_data, data, n);
        }
    }
}


void ArrayBuffer::set_sub_data(const void *data, std::size_t n, std::size_t off)
{
    assert( m_size >= n+off );

    if( m_uses_vbos )
    {
        if( !is_bound() )
            bind();

        glBufferSubDataARB(
            get_gl_buffer_type(),
            (GLintptr)off, (GLsizeiptr)n, data);

        CHECK_GL_ERRORS();
    }
    else
    {
        assert( m_data );
        std::memcpy(m_data+off, data, n);
    }
}


void ArrayBuffer::get_data(void *data, std::size_t n, std::size_t off) const
{
    if( m_uses_vbos )
    {
        if( is_mapped() )
            unmap();

        if( !is_bound() )
            bind();

        glGetBufferSubDataARB(
            get_gl_buffer_type(),
            (GLintptr)off, (GLsizeiptr)n, data);

        CHECK_GL_ERRORS();
    }
    else
    {
        std::memcpy(data, m_data, n);
    }
}


void ArrayBuffer::resize(std::size_t n)
{
    set_data(0, n);
}


void ArrayBuffer::grow_back(std::size_t n)
{
    if( !empty() )
    {
        std::size_t old_size = size();
        boost::scoped_array<boost::uint8_t> old(new boost::uint8_t[old_size]);
        get_data(old.get(), old_size, 0);

        resize(n+old_size);
        set_sub_data(old.get(), old_size, 0);

        assert( size() == old_size + n );
    }
    else
    {
        resize(n);
    }
}


void ArrayBuffer::grow_front(std::size_t n)
{
    if( !empty() )
    {
        std::size_t old_size = size();
        boost::scoped_array<boost::uint8_t> old(new boost::uint8_t[old_size]);
        get_data(old.get(), old_size, 0);

        resize(n+old_size);
        set_sub_data(old.get(), old_size, old_size);
    }
    else
    {
        resize(n);
    }
}


void *ArrayBuffer::map(bool write_only)
{
    if( is_mapped() )
        throw std::runtime_error("Buffer already mapped");

    if( m_uses_vbos )
    {
        return map_vbo(write_only);
    }
    else
    {
        m_is_mapped = true;
        return m_data;
    }
}


const void *ArrayBuffer::map() const
{
    if( is_mapped() )
        throw std::runtime_error("Buffer already mapped");

    if( m_uses_vbos )
    {
        return map_vbo();
    }
    else
    {
        m_is_mapped = true;
        return m_data;
    }
}


void ArrayBuffer::unmap() const
{
    if( !is_mapped() )
        throw std::runtime_error("Buffer not mapped");

    if( m_uses_vbos )
    {
        unmap_vbo();
    }
    else
    {
        m_is_mapped = false;
    }
}


void *ArrayBuffer::map_vbo(bool write_only)
{
    assert( m_uses_vbos );

    if( !is_bound() )
        bind();

    // Mapping a zero-size VBO is pretty much a no-op, and can trigger bugs in
    // some drivers (Mesa/Intel), so we take care to treat it a little
    // specially.
    if( m_size == 0 )
    {
        m_is_mapped = true;
        return 0;
    }
    else
    {
        void *ret = glMapBufferARB(
            get_gl_buffer_type(),
            write_only ? GL_WRITE_ONLY_ARB : GL_READ_WRITE_ARB);
        CHECK_GL_ERRORS();

        if( ret == 0 )
            throw std::runtime_error("Failed to map VBO");

        m_is_mapped = true;

        return ret;
    }
}


const void *ArrayBuffer::map_vbo() const
{
    assert( m_uses_vbos );

    if( !is_bound() )
        bind();

    // Mapping a zero-size VBO is pretty much a no-op, and can trigger bugs in
    // some drivers (Mesa/Intel), so we take care to treat it a little
    // specially.
    if( m_size == 0 )
    {
        m_is_mapped = true;
        return 0;
    }
    else
    {
        void *ret = glMapBufferARB(
            get_gl_buffer_type(), GL_READ_ONLY);
        CHECK_GL_ERRORS();

        if( ret == 0 )
            throw std::runtime_error("Failed to map VBO");

        m_is_mapped = true;

        return ret;
    }
}


void ArrayBuffer::unmap_vbo() const
{
    assert( m_uses_vbos );

    if( !is_bound() )
        bind();

    // Mapping a zero-size VBO is pretty much a no-op, and can trigger bugs in
    // some drivers (Mesa/Intel), so we take care to treat it a little
    // specially.
    if( m_size == 0 )
    {
        m_is_mapped = false;
        return;
    }
    else
    {
        bool corrupted = !glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
        CHECK_GL_ERRORS();
        m_is_mapped = false;

        if( corrupted )
            throw std::runtime_error("Buffer corrupted");
    }
}


void ArrayBuffer::bind() const
{
    assert( m_uses_vbos );

    if( m_is_index_buffer )
        ms_bound_index_buffer = m_vbo_handle;
    else
        ms_bound_vertex_buffer = m_vbo_handle;

    glBindBufferARB(get_gl_buffer_type(), m_vbo_handle);
    CHECK_GL_ERRORS();
}


void ArrayBuffer::unbind() const
{
    assert( m_uses_vbos );

    if( m_is_index_buffer )
        ms_bound_index_buffer = 0;
    else
        ms_bound_vertex_buffer = 0;

    glBindBufferARB(get_gl_buffer_type(), 0);
    CHECK_GL_ERRORS();
}


GLenum ArrayBuffer::get_gl_usage_hint() const
{
    if( m_usage == STATIC_DRAW )
        return GL_STATIC_DRAW_ARB;
    else if( m_usage == DYNAMIC_DRAW )
        return GL_DYNAMIC_DRAW_ARB;
    else
        return GL_STATIC_DRAW_ARB;
}


void ArrayBuffer::clear_index_buffer_vbo_binding() const
{
    if( !Device::has_vbo_support() )
        return;

    if( ms_bound_index_buffer != 0 )
    {
        ms_bound_index_buffer = 0;
        glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0);
        CHECK_GL_ERRORS();
    }
}


void ArrayBuffer::clear_vertex_buffer_vbo_binding() const
{
    if( !Device::has_vbo_support() )
        return;

    if( ms_bound_vertex_buffer != 0 )
    {
        ms_bound_vertex_buffer = 0;
        glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
        CHECK_GL_ERRORS();
    }
}
