/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "WorkQueue.hh"

#include <stdexcept>
#include <cmath>


using namespace peekabot;
using namespace peekabot::gui;


namespace
{
    inline void noop() {}
}


WorkQueue::WorkQueue()
    : m_posts_idx(0),
      m_is_dispatching(0),
      m_process_interval(boost::posix_time::milliseconds(1000/30)),
      m_pre_process_fun(&noop),
      m_post_process_fun(&noop)
{
}


WorkQueue::~WorkQueue()
{
    m_dispatcher_conn.disconnect();
}


void WorkQueue::post(const boost::function<void ()> &handler)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    m_posts[m_posts_idx].push(handler);
    if( !m_is_dispatching )
    {
        m_is_dispatching = true;
        m_dispatcher.emit();
    }
}


void WorkQueue::set_processing_freq(float freq)
{
    if( freq <= 0 )
        throw std::runtime_error(
            "The processing frequency must be greater than zero");

    m_process_interval = boost::posix_time::milliseconds(roundf(1000.0f/freq));
}


float WorkQueue::get_processing_freq() const
{
    return 1000.0f/m_process_interval.total_milliseconds();
}


void WorkQueue::set_pre_processing_callback(
    const boost::function<void ()> &fun)
{
    m_pre_process_fun = fun;
}


void WorkQueue::set_post_processing_callback(
    const boost::function<void ()> &fun)
{
    m_post_process_fun = fun;
}


void WorkQueue::on_dispatch()
{
    boost::posix_time::ptime now =
        boost::posix_time::microsec_clock::local_time();

    if( now - m_last_process < m_process_interval )
    {
        Glib::signal_timeout().connect_once(
            sigc::mem_fun(*this, &WorkQueue::process),
            (m_process_interval - (now - m_last_process)).total_milliseconds());
    }
    else
    {
        process();
    }
}


void WorkQueue::process()
{
    m_last_process = boost::posix_time::microsec_clock::local_time();

    std::size_t idx;
    // Handle posts
    {
        boost::recursive_mutex::scoped_lock lock(m_mutex);
        idx = m_posts_idx;
        m_posts_idx = !m_posts_idx;
        m_is_dispatching = false;
    }

    // Execute the posts
    m_pre_process_fun();
    while( !m_posts[idx].empty() )
    {
        m_posts[idx].front()();
        m_posts[idx].pop();
    }
    m_post_process_fun();
}


void WorkQueue::start()
{
    m_last_process =
        boost::posix_time::microsec_clock::local_time() - m_process_interval;

    m_dispatcher_conn = m_dispatcher.connect(
        sigc::mem_fun(*this, &WorkQueue::on_dispatch));

    if( m_is_dispatching )
        on_dispatch();
}
