/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_GUI_SCENE_VIEW_FRAME_HH_INCLUDED
#define PEEKABOT_GUI_SCENE_VIEW_FRAME_HH_INCLUDED


#include "../Types.hh"
#include "Frame.hh"

#include <set>
#include <GL/glew.h>
#include <gtkmm.h>
#include <gtkglmm.h>
#include <boost/cstdint.hpp>
#include <Eigen/Geometry>
#include <boost/signals2.hpp>


namespace peekabot
{
    class ServerData;
    class SceneObject;
    class CameraObject;

    namespace gui
    {
        class Gui;
        class FrameLayoutController;

        class SceneViewFrame : public Frame
        {
        public:
            SceneViewFrame(FrameLayoutController &layout);

            virtual ~SceneViewFrame();

            void redraw_scene_view();

            ObjectID get_camera() const;

            ObjectID pick(double x, double y);

        private:
            void setup_gl_area();

            void setup_cam_ctrls();

            void setup_layer_ctrls();

            void setup_screenshot_ctrls();


            bool on_gl_area_expose_event(GdkEventExpose *event);

            bool on_gl_area_mouse_motion(GdkEventMotion *event);

            bool on_gl_area_scroll(GdkEventScroll *event);

            bool on_gl_area_button_release(GdkEventButton *event);

            bool on_gl_area_button_press(GdkEventButton *event);

            bool on_gl_area_key_release(GdkEventKey *event);

            bool on_gl_area_enter_notify(GdkEventCrossing *event);

            void on_ortho_toggled();

            void on_reset_camera();

            void on_object_attached(
                ObjectID id, const std::string &name, ObjectType type);

            void on_object_detached(ObjectID id);

            void on_object_renamed(ObjectID id, const std::string &name);

            void on_camera_changed();

            void on_screenshot();

            void on_quickshot();

            std::string generate_snapshot_filename(
                const std::string &prefix,
                const std::string &postfix) const;

            void on_layer_toggled(int layer_no);

            void on_center_on_selection();

            void on_selection_changed();

            void toggle_toolbar_controls(bool sensitive);

            void on_top_down(Eigen::Vector3f v);

            void on_align_camera(std::size_t dim, int sign);

            /// \name Methods executed in the server thread
            /// @{

            void toggle_ortho(ServerData &sd, ObjectID cam_id, bool ortho);

            void set_camera_transform(
                ServerData &sd, ObjectID cam_id,
                Eigen::Transform3f trans);

            void connect_on_camera_view_mode_set(ServerData &sd, ObjectID cam_id);

            void on_camera_view_mode_set(CameraObject *cam);

            void center_on_selection_handler(
                ServerData &sd, ObjectID cam_id,
                std::set<ObjectID> selection);

            void ss_top_down(
                ServerData &sd, ObjectID cam_id, Eigen::Vector3f v);

            void ss_align_camera(
                ServerData &sd, ObjectID cam_id, std::size_t dim, int sign);

            /// @}

        private:
            class CameraCols : public Gtk::TreeModel::ColumnRecord
            {
            public:
                CameraCols()
                {
                    add(m_id);
                    add(m_name);
                }

                Gtk::TreeModelColumn<ObjectID> m_id;
                Gtk::TreeModelColumn<std::string> m_name;
            };

            Gui &m_gui;

            Gtk::ToolItem m_tool_cameras;

            Gtk::ToggleToolButton m_tool_ortho;

            Gtk::MenuToolButton m_tool_cam_ctrls;

            Gtk::Menu m_menu_cam_ctrls;

            Gtk::MenuItem m_center_on_selection_item;

            Gtk::MenuToolButton m_tool_screenshot_ctrls;

            Gtk::Menu m_menu_screenshot_ctrls;

            Gtk::GL::DrawingArea m_gl_area;

            Gtk::ToolItem m_tool_expander;

            CameraCols m_camera_cols;

            Glib::RefPtr<Gtk::ListStore> m_cameras_model;

            Gtk::ComboBox m_cameras;

            gdouble m_pointer_x, m_pointer_y;

            ObjectID m_cam_id;

            bool m_layers[10];

            boost::signals2::connection m_view_mode_set_conn;

            Gtk::ToolItem m_tool_layers;

            Gtk::Table m_layer_table;

            Gtk::ToggleButton m_layer_btn[10];

            double m_button_press_x[3];
            double m_button_press_y[3];
        };
    }
}


#endif // PEEKABOT_GUI_SCENE_VIEW_FRAME_HH_INCLUDED
