/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "PbarPlayerController.hh"
#include "MainWindowController.hh"
#include "Gui.hh"
#include "../Server.hh"
#include "../ServerExecutionContext.hh"
#include "../PbarPlayer.hh"
#include "../Action.hh"
#include "../ActionSource.hh"

#include <sstream>
#include <cassert>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


// This class hold state related to the client
class PbarPlayerController::ActionSourceData : public ActionSource
{
public:
    ActionSourceData(const Config &config)
        : ActionSource(config) {}

    virtual void send_response(const boost::shared_ptr<Action> &) {}
};


PbarPlayerController::PbarPlayerController(Gui &gui)
    : m_gui(gui),
      m_player(0),
      m_action_source_data(0)
{
    setup_widgets();
    setup_actions();
}


void PbarPlayerController::setup_actions()
{
    m_open_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        m_gui.get_builder()->get_object("open_pbar_action"));
    m_open_action->signal_activate().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_open_pbar));

    m_close_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        m_gui.get_builder()->get_object("close_pbar_action"));
    m_close_action->signal_activate().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_close));

    m_reload_action = Glib::RefPtr<Gtk::Action>::cast_dynamic(
        m_gui.get_builder()->get_object("reload_pbar_action"));
    m_reload_action->signal_activate().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_reload));

    m_toggle_toolbar_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        m_gui.get_builder()->get_object("toggle_pbar_toolbar_action"));
    m_toggle_toolbar_action->signal_toggled().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_toggle_toolbar));

    m_play_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        m_gui.get_builder()->get_object("play_pbar_action"));
    m_play_conn = m_play_action->signal_toggled().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_play_toggled));
}


void PbarPlayerController::setup_widgets()
{
    m_gui.get_builder()->get_widget("playback_toolbar", m_toolbar);

    m_gui.get_builder()->get_widget("open_pbar_dialog", m_open_pbar_dialog);
    m_open_pbar_dialog->signal_response().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_open_pbar_response));

    m_gui.get_builder()->get_widget("pbar_step_spinbutton", m_step_spinbutton);
    m_step_spinbutton->set_range(1, 1000);
    m_step_spinbutton->set_value(1);
    m_step_spinbutton->set_increments(1, 10);
    m_step_spinbutton->signal_icon_release().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_step_clicked));

    m_gui.get_builder()->get_widget(
        "pbar_timestep_spinbutton", m_timestep_spinbutton);
    m_timestep_spinbutton->set_range(0, 3600);
    m_timestep_spinbutton->set_increments(0.1, 1);
    m_timestep_spinbutton->set_value(1);
    m_timestep_spinbutton->signal_icon_release().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_timestep_clicked));

    m_gui.get_builder()->get_widget(
        "pbar_speed_spinbutton", m_speed_spinbutton);
    m_speed_spinbutton->set_range(0.01, 1000);
    m_speed_spinbutton->set_value(1.0);
    m_speed_spinbutton->set_increments(0.1, 1);
    m_speed_spinbutton->signal_changed().connect(
        sigc::mem_fun(*this, &PbarPlayerController::on_playback_speed_changed));

    m_gui.get_builder()->get_widget("pbar_timeline_hscale", m_timeline);
}


void PbarPlayerController::on_open_pbar()
{
    m_open_pbar_dialog->show();
}


void PbarPlayerController::on_open_pbar_response(int response)
{
    m_open_pbar_dialog->hide();
    if( response == Gtk::RESPONSE_OK )
    {
        assert( !m_player );

        m_filename = m_open_pbar_dialog->get_filename();
        load_recording(m_filename);
    }
}


void PbarPlayerController::on_toggle_toolbar()
{
    m_toolbar->set_visible(m_toggle_toolbar_action->get_active());
}


void PbarPlayerController::on_play_toggled()
{
    bool play = m_play_action->get_active();

    m_step_spinbutton->set_sensitive(!play);
    m_timestep_spinbutton->set_sensitive(!play);

    if( play )
    {
        m_started = true;
        m_player->play();
    }
    else
    {
        m_player->pause();
    }
}


void PbarPlayerController::on_playback_speed_changed()
{
    m_player->set_playback_speed(m_speed_spinbutton->get_value());
}


void PbarPlayerController::dispatch_action(boost::shared_ptr<Action> action)
{
    m_gui.server_post(
        boost::bind(&PbarPlayerController::execute_action, this, _1, action));
}


void PbarPlayerController::execute_action(
    ServerData &sd, boost::shared_ptr<Action> action)
{
    ServerExecutionContext context(
        m_gui.get_config(), *sd.m_scene, *m_action_source_data);

    try
    {
        action->execute(&context);
    }
    catch(std::exception &e)
    {
        /*TheMessageHub::instance().publish(
            ERROR_MESSAGE, e.what());*/
    }
    catch(...)
    {
        /*TheMessageHub::instance().publish(
            ERROR_MESSAGE,
            "Caught an unknown exception during action execution");*/
    }

    m_gui.post(
        boost::bind(&PbarPlayerController::on_time_advanced, this,
                    m_player->elapsed().total_milliseconds()/1000.0));

    if( m_player->is_finished() )
        m_gui.post(boost::bind(&PbarPlayerController::on_finished, this));
}


void PbarPlayerController::on_time_advanced(double t)
{
    if( m_started )
        m_timeline->set_value(t);
}


void PbarPlayerController::on_finished()
{
    m_play_action->set_sensitive(false);
    m_step_spinbutton->set_sensitive(false);
    m_timestep_spinbutton->set_sensitive(false);
    m_play_conn.block();
    m_play_action->set_active(false);
    m_play_conn.unblock();
}


void PbarPlayerController::on_reload()
{
    unload_recording();
    load_recording(m_filename);
}


void PbarPlayerController::on_close()
{
    unload_recording();
}


void PbarPlayerController::load_recording(const std::string &filename)
{
    assert( !m_player );
    assert( !m_action_source_data );

    try
    {
        m_player = new PbarPlayer(
            filename,
            boost::bind(&PbarPlayerController::dispatch_action, this, _1));
    }
    catch(std::exception &e)
    {
        delete m_player;
        m_player = 0;
        m_gui.get_main_window_controller().display_message(
            Gtk::MESSAGE_ERROR, e.what());
        return;
    }

    m_action_source_data = new ActionSourceData(m_gui.get_config());
    m_player->set_playback_speed(m_speed_spinbutton->get_value());

    // Activate replay controls
    m_play_action->set_sensitive(true);
    m_step_spinbutton->set_sensitive(true);
    m_timestep_spinbutton->set_sensitive(true);
    m_close_action->set_sensitive(true);
    m_reload_action->set_sensitive(true);
    // Deactivate the open action
    m_open_action->set_sensitive(false);

    m_timeline->set_range(
        0, m_player->duration().total_milliseconds()/1000.0);
    m_timeline->set_value(0);

    m_started = false;

    // Post some statistics in the status bar
    std::stringstream ss;
    ss << "Recording with " << m_player->action_count()
       << " actions and " << m_player->duration().total_seconds()
       << " seconds duration loaded";
    m_gui.get_main_window_controller().set_status_message(ss.str());
}


void PbarPlayerController::unload_recording()
{
    assert( m_player );
    assert( m_action_source_data );

    delete m_player;
    m_player = 0;
    delete m_action_source_data;
    m_action_source_data = 0;

    // Reset the timeline
    m_timeline->set_value(0);
    // Deactivate and make sure the play button is not toggled
    m_play_action->set_sensitive(false);
    m_play_conn.block();
    m_play_action->set_active(false);
    m_play_conn.unblock();
    // Deactivate step widgets
    m_step_spinbutton->set_sensitive(false);
    m_timestep_spinbutton->set_sensitive(false);
    // Deactivate close/reload buttons
    m_close_action->set_sensitive(false);
    m_reload_action->set_sensitive(false);
    // Activate the open action
    m_open_action->set_sensitive(true);
}


void PbarPlayerController::on_step_clicked(
    Gtk::EntryIconPosition, const GdkEventButton *)
{
    m_started = true;
    m_player->step(m_step_spinbutton->get_value_as_int());
}


void PbarPlayerController::on_timestep_clicked(
    Gtk::EntryIconPosition, const GdkEventButton *)
{
    m_started = true;
    m_player->step(
        boost::posix_time::milliseconds(
            1000*m_timestep_spinbutton->get_value()));
}
