/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "LogViewController.hh"
#include "Gui.hh"
#include "../MessageHub.hh"

#include <sstream>
#include <boost/bind.hpp>


using namespace peekabot;
using namespace peekabot::gui;


LogViewController::LogViewController(Gui &gui)
    : m_gui(gui)
{
    m_gui.get_builder()->get_widget("log_textview", m_view);
    m_buffer = m_view->get_buffer();
    m_eof_mark = m_buffer->create_mark(m_buffer->end());

    m_gui.get_builder()->get_widget("log_scrolledwindow", m_log_window);
    m_gui.get_builder()->get_widget("log_paned", m_log_paned);

    m_toggle_log_action = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(
        m_gui.get_builder()->get_object("toggle_log_action"));
    m_toggle_log_action->signal_toggled().connect(
        sigc::mem_fun(*this, &LogViewController::on_toggle_log));

    TheMessageHub::instance().subscribe(INFO_MESSAGE, this);
    TheMessageHub::instance().subscribe(WARNING_MESSAGE, this);
    TheMessageHub::instance().subscribe(ERROR_MESSAGE, this);
}


LogViewController::~LogViewController()
{
    TheMessageHub::instance().unsubscribe(INFO_MESSAGE, this);
    TheMessageHub::instance().unsubscribe(WARNING_MESSAGE, this);
    TheMessageHub::instance().unsubscribe(ERROR_MESSAGE, this);
}


void LogViewController::handle_message(const Message &message) throw()
{
    m_gui.post(
        boost::bind(
            &LogViewController::handle_message_handler, this, message));
}


void LogViewController::handle_message_handler(Message message)
{
    std::stringstream ss;

    if( m_buffer->begin() != m_buffer->end() )
        ss << "\n";

    if( message.get_type() == INFO_MESSAGE )
        ss << "INFO: ";
    else if( message.get_type() == WARNING_MESSAGE )
        ss << "WARNING: ";
    else if( message.get_type() == ERROR_MESSAGE )
        ss << "ERROR: ";

    ss << message.get_message();

    Gtk::TextBuffer::iterator it = m_buffer->insert(m_buffer->end(), ss.str());
    m_buffer->move_mark(m_eof_mark, it);
    m_view->scroll_to(m_eof_mark);

    // Prune the buffer to 1000 lines
    const int MAX_LOG_LINES = 1000;
    if( m_buffer->get_line_count() > MAX_LOG_LINES )
    {
        Gtk::TextBuffer::iterator beg_it = m_buffer->begin();
        Gtk::TextBuffer::iterator prune_it =
            m_buffer->get_iter_at_line(
                m_buffer->get_line_count()-MAX_LOG_LINES);
        m_buffer->erase(beg_it, prune_it);
    }

    // If there are new errors, indicate that in the toolbar by
    // changing the icon
    if( message.get_type() == ERROR_MESSAGE &&
        !m_toggle_log_action->get_active() &&
        m_toggle_log_action->get_icon_name() != "peekabot-log-error" )
    {
        m_toggle_log_action->set_icon_name("peekabot-log-error");
        m_toggle_log_action->set_tooltip(
            "Show/hide the log (there are new errors)");
    }
}


void LogViewController::on_toggle_log()
{
    if( m_toggle_log_action->get_active() )
    {
        m_log_paned->set_position(m_log_paned->get_height()-150);

        if( m_toggle_log_action->get_icon_name() != "peekabot-log" )
        {
            m_toggle_log_action->set_icon_name("peekabot-log");
            m_toggle_log_action->set_tooltip("Show/hide the log");
        }
    }

    m_log_window->set_visible(m_toggle_log_action->get_active());
}
