/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "VertexBasedProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/Assign.hh"
#include "../../actions/SetVertices.hh"
#include "../../actions/SetColoredVertices.hh"
#include "../../actions/SetProp.hh"

#include <vector>
#include <Eigen/Core>


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ VertexSet implementation --------------------
//

struct VertexSet::Impl
{
    inline void add(float x, float y, float z)
    {
        m_verts.push_back(Eigen::Vector3f(x, y, z));
    }

    inline void clear()
    {
        m_verts.clear();
    }

    inline std::size_t size() const
    {
        return m_verts.size();
    }

    inline bool empty() const
    {
        return m_verts.empty();
    }

    std::vector<Eigen::Vector3f> m_verts;
};


VertexSet::VertexSet()
    : m_impl(new VertexSet::Impl)
{
}

VertexSet::VertexSet(const VertexSet &other)
    : m_impl(new Impl(*other.m_impl))
{
}

VertexSet::~VertexSet()
{
}

VertexSet &VertexSet::operator=(const VertexSet &other)
{
    m_impl.reset(new Impl(*other.m_impl));
    return *this;
}

void VertexSet::add(float x, float y, float z)
{
    m_impl->add(x, y, z);
}

void VertexSet::add_vertex(float x, float y, float z)
{
    add(x, y, z);
}

void VertexSet::clear()
{
    m_impl->clear();
}

std::size_t VertexSet::size() const
{
    return m_impl->size();
}

bool VertexSet::empty() const
{
    return m_impl->empty();
}

//
// ------------------ ColoredVertexSet implementation --------------------
//

struct ColoredVertexSet::Impl
{
    inline void add(
        float x, float y, float z,
        float r, float g, float b)
    {
        m_verts.push_back(Eigen::Vector3f(x, y, z));
        m_colors.push_back(float_to_uint8(r));
        m_colors.push_back(float_to_uint8(g));
        m_colors.push_back(float_to_uint8(b));
    }

    inline boost::uint8_t float_to_uint8(float x) const
    {
        int tmp = 255*x;
        return tmp > 255 ? 255 : (tmp < 0 ? 0 : tmp);
    }

    inline void clear()
    {
        m_verts.clear();
        m_colors.clear();
    }

    inline std::size_t size() const
    {
        return m_verts.size();
    }

    inline bool empty() const
    {
        return m_verts.empty();
    }

    std::vector<Eigen::Vector3f> m_verts;
    std::vector<boost::uint8_t> m_colors;
};


ColoredVertexSet::ColoredVertexSet()
    : m_impl(new ColoredVertexSet::Impl)
{
}

ColoredVertexSet::ColoredVertexSet(const ColoredVertexSet &other)
    : m_impl(new Impl(*other.m_impl))
{
}

ColoredVertexSet::~ColoredVertexSet()
{
}

ColoredVertexSet &ColoredVertexSet::operator=(const ColoredVertexSet &other)
{
    m_impl.reset(new Impl(*other.m_impl));
    return *this;
}

void ColoredVertexSet::add(
    float x, float y, float z,
    float r, float g, float b)
{
    m_impl->add(x, y, z, r, g, b);
}

void ColoredVertexSet::clear()
{
    m_impl->clear();
}

std::size_t ColoredVertexSet::size() const
{
    return m_impl->size();
}

bool ColoredVertexSet::empty() const
{
    return m_impl->empty();
}

//
// ------------------ VertexBasedProxyBase implementation -------------------
//

VertexBasedProxyBase::VertexBasedProxyBase()
{
}


VertexBasedProxyBase::VertexBasedProxyBase(const VertexBasedProxyBase &p)
    : ObjectProxyBase(p)
{
}

//

DelayedDispatch VertexBasedProxyBase::set_vertices(
    const VertexSet &vertices)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetVertices(get_object_id(), vertices.m_impl->m_verts));
}


DelayedDispatch VertexBasedProxyBase::add_vertices(
    const VertexSet &vertices)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetVertices(get_object_id(), vertices.m_impl->m_verts, true));
}


DelayedDispatch VertexBasedProxyBase::set_vertices(
    const ColoredVertexSet &vertices)
{
    assert( vertices.m_impl->m_verts.size() ==
            vertices.m_impl->m_colors.size()/3 );

    return DelayedDispatch(
        get_client_impl(),
        new SetColoredVertices(
            get_object_id(),
            vertices.m_impl->m_verts,
            vertices.m_impl->m_colors));
}


DelayedDispatch VertexBasedProxyBase::add_vertices(
    const ColoredVertexSet &vertices)
{
    assert( vertices.m_impl->m_verts.size() ==
            vertices.m_impl->m_colors.size()/3 );

    return DelayedDispatch(
        get_client_impl(),
        new SetColoredVertices(
            get_object_id(),
            vertices.m_impl->m_verts,
            vertices.m_impl->m_colors, true));
}


DelayedDispatch VertexBasedProxyBase::add_vertex(float x, float y, float z)
{
    VertexSet vs;
    vs.add(x,y,z);
    return add_vertices(vs);
}


DelayedDispatch VertexBasedProxyBase::clear_vertices()
{
    VertexSet empty_vs;
    return set_vertices(empty_vs);
}


DelayedDispatch VertexBasedProxyBase::set_max_vertices(boost::uint32_t n)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), MAX_VERTICES_PROP, n));
}


DelayedDispatch VertexBasedProxyBase::set_vertex_overflow_policy(
    VertexOverflowPolicy overflow_policy)
{

    return DelayedDispatch(
        get_client_impl(),
        new SetProp(
            get_object_id(), VERTEX_OVERFLOW_POLICY_PROP, overflow_policy));
}


//
// ------------------ VertexBasedProxy implementation -------------------
//


VertexBasedProxy::VertexBasedProxy()
{
}

VertexBasedProxy::VertexBasedProxy(const VertexBasedProxyBase &p)
    : ObjectProxyBase(p),
      VertexBasedProxyBase(p)
{
}


VertexBasedProxy &VertexBasedProxy::operator=(
    const VertexBasedProxy &p)
{
    return *this = (VertexBasedProxyBase &)p;
}


VertexBasedProxy &VertexBasedProxy::operator=(
    const VertexBasedProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}

DelayedDispatch VertexBasedProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(get_object_id(p)),
                   get_object_id(), VERTEX_BASED_CLASS));
}

DelayedDispatch VertexBasedProxy::assign(
    PeekabotClient &client,
    const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), VERTEX_BASED_CLASS));
}

DelayedDispatch VertexBasedProxy::assign(
    const ObjectProxyBase &parent, 
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), VERTEX_BASED_CLASS));
}
