/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "OccupancyGrid3DProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/AddObject.hh"
#include "../../actions/Assign.hh"
#include "../../actions/SetOccupancyGrid3DCells.hh"
#include "../../actions/SetProp.hh"
#include "../../actions/MiniBundle.hh"

#include <vector>
#include <Eigen/Core>


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ OccupancySet3D implementation -------------------
//

struct OccupancySet3D::Impl
{
    inline void set_cell(float x, float y, float z, float belief)
    {
        m_cells.push_back(std::make_pair(Eigen::Vector3f(x, y, z), belief));
    }

    inline void clear()
    {
        m_cells.clear();
    }

    inline std::size_t size() const
    {
        return m_cells.size();
    }

    inline bool empty() const
    {
        return m_cells.empty();
    }

    std::vector<std::pair<Eigen::Vector3f, float> > m_cells;
};


OccupancySet3D::OccupancySet3D()
    : m_impl(new OccupancySet3D::Impl)
{
}

OccupancySet3D::OccupancySet3D(const OccupancySet3D &other)
    : m_impl(new Impl(*other.m_impl))
{
}

OccupancySet3D::~OccupancySet3D()
{
}

OccupancySet3D &OccupancySet3D::operator=(const OccupancySet3D &other)
{
    m_impl.reset(new Impl(*other.m_impl));
    return *this;
}

void OccupancySet3D::set_cell(float x, float y, float z, float belief)
{
    m_impl->set_cell(x, y, z, belief);
}

void OccupancySet3D::clear()
{
    m_impl->clear();
}

std::size_t OccupancySet3D::size() const
{
    return m_impl->size();
}

bool OccupancySet3D::empty() const
{
    return m_impl->empty();
}


//
// ------------------ OccupancyGrid3DProxyBase implementation -------------------
//


OccupancyGrid3DProxyBase::OccupancyGrid3DProxyBase()
{
}


OccupancyGrid3DProxyBase::OccupancyGrid3DProxyBase(OccupancyGrid3DProxyBase &p)
    : ObjectProxyBase(p)
{
}

//

DelayedDispatch OccupancyGrid3DProxyBase::set_cells(
    const OccupancySet3D &cells)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetOccupancyGrid3DCells(get_object_id(), cells.m_impl->m_cells));
}


DelayedDispatch OccupancyGrid3DProxyBase::enable_color_mapping(
    float z_min, float z_max)
{
    MiniBundle *b = new MiniBundle();
    b->add_action(
        new SetProp(get_object_id(), OG3D_COLOR_MAPPING_Z_MIN_PROP, z_min));
    b->add_action(
        new SetProp(get_object_id(), OG3D_COLOR_MAPPING_Z_MAX_PROP, z_max));
    b->add_action(
        new SetProp(get_object_id(), OG3D_COLOR_MAPPING_ENABLED_PROP, true));

    return DelayedDispatch(get_client_impl(), b);
}


DelayedDispatch OccupancyGrid3DProxyBase::disable_color_mapping()
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), OG3D_COLOR_MAPPING_ENABLED_PROP, false));
}


//
// ------------------ OccupancyGrid3DProxy implementation -------------------
//


OccupancyGrid3DProxy::OccupancyGrid3DProxy()
{
}


OccupancyGrid3DProxy::OccupancyGrid3DProxy(OccupancyGrid3DProxy &p)
    : OccupancyGrid3DProxyBase(p)
{
}


OccupancyGrid3DProxy &OccupancyGrid3DProxy::operator=(
    const OccupancyGrid3DProxy &p)
{
    return *this = (OccupancyGrid3DProxyBase &)p;
}


OccupancyGrid3DProxy &OccupancyGrid3DProxy::operator=(const
    OccupancyGrid3DProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}


DelayedDispatch OccupancyGrid3DProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(
            PathIdentifier(get_object_id(p)), get_object_id(), OG3D_OBJECT));
}


DelayedDispatch OccupancyGrid3DProxy::assign(
    PeekabotClient &client,
    const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), OG3D_OBJECT));
}


DelayedDispatch OccupancyGrid3DProxy::assign(
    const ObjectProxyBase &parent,
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(), new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), OG3D_OBJECT));
}


DelayedDispatch OccupancyGrid3DProxy::add(
    PeekabotClient &client,
    const std::string &path,
    float cell_xy_size, float cell_z_size,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_xy_size));
    args.push_back(Any(cell_z_size));

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(
            PathIdentifier(path), conflict_policy,
            get_object_id(), OG3D_OBJECT, args));
}


DelayedDispatch OccupancyGrid3DProxy::add(
    const ObjectProxyBase &parent,
    const std::string &name,
    float cell_xy_size, float cell_z_size,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    AddObject::Args args;
    args.push_back(Any(cell_xy_size));
    args.push_back(Any(cell_z_size));

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(
            PathIdentifier(get_object_id(parent), name),
            conflict_policy, get_object_id(), OG3D_OBJECT, args));
}
