/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "CameraProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/AddObject.hh"
#include "../../actions/Assign.hh"
#include "../../actions/SetProp.hh"


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ CameraProxyBase implementation -------------------
//


CameraProxyBase::CameraProxyBase()
{
}


CameraProxyBase::CameraProxyBase(const CameraProxyBase &p)
    : ObjectProxyBase(p)
{
}

//

DelayedDispatch CameraProxyBase::set_fov(float fov_deg)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), CAMERA_FOV_PROP, fov_deg));
}


DelayedDispatch CameraProxyBase::set_zoom_distance(float zoom_distance)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), CAMERA_ZOOM_DISTANCE_PROP, zoom_distance));
}


DelayedDispatch CameraProxyBase::set_orthographic(bool orthographic)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), CAMERA_ORTHO_PROP, orthographic));
}


//
// ------------------ CameraProxy implementation -------------------
//


CameraProxy::CameraProxy()
{
}


CameraProxy::CameraProxy(const CameraProxyBase &p)
    : CameraProxyBase(p)
{
}


CameraProxy &CameraProxy::operator=(const CameraProxy &p)
{
    return *this = (CameraProxyBase &)p;
}


CameraProxy &CameraProxy::operator=(const CameraProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}


DelayedDispatch CameraProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(get_object_id(p)),
                   get_object_id(), CAMERA_OBJECT));
}


DelayedDispatch CameraProxy::assign(
    PeekabotClient &client,
    const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), CAMERA_OBJECT));
}


DelayedDispatch CameraProxy::assign(
    const ObjectProxyBase &parent, 
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(), new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), CAMERA_OBJECT));
}


DelayedDispatch CameraProxy::add(
    PeekabotClient &client,
    const std::string &path,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(path),
                      conflict_policy, get_object_id(), CAMERA_OBJECT));
}


DelayedDispatch CameraProxy::add(
    const ObjectProxyBase &parent, 
    const std::string &name,
    NameConflictPolicy conflict_policy)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new AddObject(PathIdentifier(get_object_id(parent), name),
                      conflict_policy, get_object_id(), CAMERA_OBJECT));
}
