/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_CLIENT_SERVER_CONNECTION_HH_INCLUDED
#define PEEKABOT_CLIENT_SERVER_CONNECTION_HH_INCLUDED


#include <string>
#include <queue>
#include <stdexcept>
#include <boost/thread/condition.hpp>
#include <boost/thread/thread.hpp>
#include <boost/thread/recursive_mutex.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/utility.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

#include "../Types.hh"
#include "../Sockets.hh"
#include "Status.hh"
#include "Result.hh"
#include "Transport.hh"


namespace peekabot
{
    class Action;

    namespace client
    {
        class ClientImpl;

        /**
         * \internal
         *
         * \brief Models the physical connection to the peekabot server.
         *
         * Encapsulates functionality for establishing a connection,
         * transmitting and receiving data (actions) and the low level parts
         * of the protocol (authentication, packet headers, etc.).
         *
         * \section Thread safety
         *
         * All public methods are guaranteed to be thread-safe.
         */
        class ServerConnection : public Transport,
                                 public boost::noncopyable
        {
        public:
            ServerConnection(boost::shared_ptr<ClientImpl> client);

            virtual ~ServerConnection();

            /**
             * \brief Establish a link to the given host.
             *
             * This method blocks.
             *
             * If the connection fails, or the authentication fails an
             * exception is thrown to indicate the error.
             */
            void connect(const std::string &hostname,
                         unsigned int port,
                         bool low_latency_mode);

            /**
             * \brief Shut down the connection and discard any unsent data.
             */
            void disconnect();

            /**
             * \brief Send all queued outbound actions.
             *
             * The method will send all unsent actions if and only
             * if the connection is connected and remains connected during the
             * call to flush().
             *
             * If the connection is in its disconnected state, this method is
             * a no-op.
             */
            virtual void flush();

            /**
             * \brief Returns \c true if the connection is connected, and
             * \c false if the connection is in its disconnected state or '
             * disconnecting.
             */
            bool is_connected() const;

            /**
             * \brief Queue the given action for transmission.
             *
             * This method is asynchronous, delivery is not guaranteed since the
             * connection may be lost before the action can be sent.
             */
            virtual void dispatch_action(boost::shared_ptr<Action> action);

            /**
             * \brief Returns the uptime of the connection.
             *
             * \return The duration for which the connection has been up.
             */
            boost::posix_time::time_duration get_uptime() const throw();

        private:
            void _connect(
                const std::string &hostname,
                unsigned int port,
                bool low_latency_mode);

            void perform_authentication(bool low_latency_mode)
                throw(std::exception);


            /**
             * \brief Runs the transmit thread.
             */
            void tx_thread();

            /**
             * \brief Runs the receive thread.
             */
            void rx_thread();


            void serialize_and_send(boost::shared_ptr<Action> action);

            void blocking_send(const void *buf, size_t n);

            size_t timed_send(const void *buf, size_t n, uint32_t timeout_ms);

            size_t timed_recv(void *buf, size_t n, uint32_t timeout_ms);

            void timed_connect(
                const std::string &hostname, int port, size_t timeout_ms);

            void discard_unsent() throw();

        private:
            boost::thread * volatile m_tx_thread;
            boost::thread * volatile m_rx_thread;

            volatile bool m_stop_signal;

            boost::recursive_mutex m_sockfd_mutex;

            sockets::SocketType m_sockfd;

            bool m_server_is_big_endian;

            boost::posix_time::ptime m_up_since;

            mutable boost::recursive_mutex m_outbound_mutex;
            std::queue<boost::shared_ptr<Action> > m_outbound;
            boost::condition m_outbound_push_cond;
            boost::condition m_outbound_pop_cond;
        };
    }
}


#endif // PEEKABOT_CLIENT_SERVER_CONNECTION_HH_INCLUDED
