/*
 * Copyright Staffan Gimåker 2006-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "PeekabotClient.hh"
#include "ClientImpl.hh"
#include "RecordingImpl.hh"
#include "../actions/NoOp.hh"
#include "../actions/UploadFile.hh"
#include "../actions/RemoveFile.hh"
#include "../Version.hh"


using namespace peekabot;
using namespace peekabot::client;


PeekabotClient::PeekabotClient()
    : m_impl(new ClientImpl())
{
}


PeekabotClient::PeekabotClient(const PeekabotClient &client)
    : m_impl(client.m_impl)
{
}


PeekabotClient::PeekabotClient(boost::shared_ptr<ClientImpl> impl)
    : m_impl(impl)
{
}


PeekabotClient::~PeekabotClient()
{
}


void PeekabotClient::connect(const std::string &hostname, unsigned int port)
{
    m_impl->connect_master(hostname, port);
}


void PeekabotClient::disconnect()
{
    if( is_connected() )
    {
        if( is_bundling() )
            end_bundle();

        m_impl->disconnect_master();
    }
}

void PeekabotClient::client_version(int &major, int &minor, int &rev) const
{
    major = PEEKABOT_VERSION_MAJOR;
    minor = PEEKABOT_VERSION_MINOR;
    rev   = PEEKABOT_VERSION_REVISION;
}


const std::string &PeekabotClient::client_version_str() const
{
    return PEEKABOT_VERSION_STRING;
}


bool PeekabotClient::is_connected() const
{
    return m_impl->is_master_connected();
}


void PeekabotClient::begin_bundle()
{
    m_impl->begin_bundle();
}


DelayedDispatch PeekabotClient::end_bundle()
{
    return DelayedDispatch(m_impl, m_impl->end_bundle(), true);
}


bool PeekabotClient::is_bundling() const
{
    return m_impl->is_bundling();
}


void PeekabotClient::flush()
{
    m_impl->flush_master();
}


DelayedDispatch PeekabotClient::noop()
{
    return DelayedDispatch(m_impl, new NoOp());
}


void PeekabotClient::sync()
{
    m_impl->sync_master();
}


void PeekabotClient::start_recording(const std::string &filename)
{
    m_impl->start_recording(filename);
}


void PeekabotClient::stop_recording()
{
    m_impl->stop_recording();
}


bool PeekabotClient::is_recording() const
{
    return m_impl->is_recording();
}


Recording PeekabotClient::open_recording(const std::string &filename)
{
    return Recording(
        boost::shared_ptr<RecordingImpl>(
            new RecordingImpl(*m_impl, filename)));
}


DelayedDispatch PeekabotClient::upload_file(const std::string &filename)
{
    return DelayedDispatch(m_impl, new UploadFile(filename));
}


DelayedDispatch PeekabotClient::remove_file(const std::string &filename)
{
    return DelayedDispatch(m_impl, new RemoveFile(filename));
}


boost::shared_ptr<ClientImpl> PeekabotClient::get_impl()
{
    return m_impl;
}


const boost::shared_ptr<ClientImpl> PeekabotClient::get_impl() const
{
    return m_impl;
}
