/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <boost/math/fpclassify.hpp>
#include <Eigen/Core>
#include <Eigen/LU>

#include "Translate.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


Translate::Translate()
{
}


Translate::Translate(
    ObjectID target,
    float x, float y, float z,
    CoordinateSystem coord_sys)
    : m_target(target),
      m_x(x), m_y(y), m_z(z),
      m_coord_sys(coord_sys)
{
}


Action *Translate::clone() const
{
    return new Translate(m_target, m_x, m_y, m_z, m_coord_sys);
}


void Translate::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error("Target object not found.");

    // Check for infinity and NaN
    if( !boost::math::isfinite(m_x) ||
        !boost::math::isfinite(m_y) ||
        !boost::math::isfinite(m_z) )
        throw std::logic_error(
            "translate failed: distances cannot be infinity or NaN");

    Eigen::Vector4f pos_p(
        ptr->get_transformation().matrix().col(3));
    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
            pos_p +=
                m_x*ptr->get_transformation().matrix().col(0) +
                m_y*ptr->get_transformation().matrix().col(1) +
                m_z*ptr->get_transformation().matrix().col(2);
            break;

        case PARENT_COORDINATES:
            pos_p += Eigen::Vector4f(m_x, m_y, m_z, 0);
            break;

        case WORLD_COORDINATES:
        {
            Eigen::Transform3f axes = Eigen::Transform3f(
                ptr->get_parent_mtow().inverse(Eigen::Isometry));
            pos_p +=
                m_x*axes.matrix().col(0) +
                m_y*axes.matrix().col(1) +
                m_z*axes.matrix().col(2);
            break;
        }

        default:
            throw std::runtime_error(
                "translate failed: unsupported coordinate system");
            break;
    }

    ptr->set_position(Eigen::Vector3f(pos_p(0), pos_p(1), pos_p(2)));
#endif
}


void Translate::save(SerializationInterface &ar) const
{
    ar << m_target << m_x << m_y << m_z << m_coord_sys;
}

void Translate::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_x >> m_y >> m_z >> m_coord_sys;
}
