/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/math/fpclassify.hpp>
#include <Eigen/LU>

#include "SetRotation.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


SetRotation::SetRotation(
    ObjectID target,
    float yaw,
    float pitch,
    float roll,
    CoordinateSystem coord_sys) throw()
    : m_target(target),
      m_yaw(yaw),
      m_pitch(pitch),
      m_roll(roll),
      m_coord_sys(coord_sys)
{
}


SetRotation::SetRotation() throw()
{
}


SetRotation::SetRotation(const SetRotation &action) throw()
    : m_target(action.m_target),
      m_yaw(action.m_yaw),
      m_pitch(action.m_pitch),
      m_roll(action.m_roll),
      m_coord_sys(action.m_coord_sys)
{
}


SetRotation::~SetRotation() throw()
{
}


Action *SetRotation::clone() const
{
    return new SetRotation(*this);
}


void SetRotation::execute(
    ServerExecutionContext *context)
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Target object not found");

    // Check for infinity and NaN
    if( !boost::math::isfinite(m_yaw) ||
        !boost::math::isfinite(m_pitch) ||
        !boost::math::isfinite(m_roll) )
        throw std::logic_error(
            "Failed to set rotations: angles contain infinity or NaN");

    Eigen::Transform3f rot;
    Eigen::Vector4f pos = ptr->get_transformation().matrix().col(3);

    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
            rot =
                Eigen::Translation3f(pos(0), pos(1), pos(2)) *
                Eigen::AngleAxisf(m_roll, Eigen::Vector3f::UnitX()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ()) *
                Eigen::Translation3f(-pos(0), -pos(1), -pos(2)) *
                ptr->get_transformation();
            break;

        case PARENT_COORDINATES:
            rot =
                Eigen::Translation3f(pos(0), pos(1), pos(2)) *
                Eigen::AngleAxisf(m_roll, Eigen::Vector3f::UnitX()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ());
            break;

        case WORLD_COORDINATES:
            // Rotate, then translate and at last convert to parent coordinates
            rot =
                Eigen::Transform3f(
                    ptr->get_parent_mtow().inverse(Eigen::Isometry)) *
                Eigen::Translation3f(pos(0), pos(1), pos(2)) *
                Eigen::AngleAxisf(m_roll, Eigen::Vector3f::UnitX()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ());
            break;

        default:
            throw std::runtime_error(
                "Failed to set rotation: unsupported coordinate system");
            break;
    }

    ptr->set_transformation(rot);
#endif
}


void SetRotation::save(SerializationInterface &ar) const
{
    ar << m_target
       << m_yaw << m_roll << m_pitch
       << m_coord_sys;
}


void SetRotation::load(DeserializationInterface &ar)
{
    ar >> m_target
       >> m_yaw >> m_roll >> m_pitch
       >> m_coord_sys;
}
