/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/math/fpclassify.hpp>
#include <Eigen/LU>

#include "SetPose.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


SetPose::SetPose(
    ObjectID target,
    float x, float y, float z,
    float yaw, float roll, float pitch,
    CoordinateSystem coord_sys) throw()
    : m_target(target),
      m_x(x), m_y(y), m_z(z),
      m_yaw(yaw), m_roll(roll), m_pitch(pitch),
      m_coord_sys(coord_sys)
{
}


SetPose::SetPose() throw()
{
}


SetPose::SetPose(const SetPose &action) throw()
    : m_target(action.m_target),
      m_x(action.m_x), m_y(action.m_y), m_z(action.m_z),
      m_yaw(action.m_yaw), m_roll(action.m_roll), m_pitch(action.m_pitch),
      m_coord_sys(action.m_coord_sys)
{
}


SetPose::~SetPose() throw()
{
}


Action *SetPose::clone() const
{
    return new SetPose(*this);
}


void SetPose::execute(
    ServerExecutionContext *context)
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = context->get_object(m_target);

    if( !ptr )
        throw std::runtime_error(
            "Failed to set pose: target object not found");

    // Check for infinity and NaN
    if( !boost::math::isfinite(m_x) ||
        !boost::math::isfinite(m_y) ||
        !boost::math::isfinite(m_z) )
        throw std::logic_error(
            "Failed to set pose: position cannot contain infinity or NaN");

    if( !boost::math::isfinite(m_yaw) ||
        !boost::math::isfinite(m_roll) ||
        !boost::math::isfinite(m_pitch) )
        throw std::logic_error(
            "Failed to set pose: angles contain infinity or NaN");


    Eigen::Transform3f pose;

    switch( m_coord_sys )
    {
        case LOCAL_COORDINATES:
            pose =
                Eigen::Translation3f(m_x, m_y, m_z) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitX()) *
                ptr->get_transformation();
            break;

        case PARENT_COORDINATES:
            pose =
                Eigen::Translation3f(m_x, m_y, m_z) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitX());
            break;

        case WORLD_COORDINATES:
            // Rotate, then translate and at last convert to parent coordinates
            pose =
                Eigen::Transform3f(
                    ptr->get_parent_mtow().inverse(Eigen::Isometry)) *
                Eigen::Translation3f(m_x, m_y, m_z) *
                Eigen::AngleAxisf(m_yaw, Eigen::Vector3f::UnitZ()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitY()) *
                Eigen::AngleAxisf(m_pitch, Eigen::Vector3f::UnitX());
            break;

        default:
            throw std::runtime_error(
                "Failed to set pose: unsupported coordinate system");
            break;
    }

    ptr->set_transformation(pose);
#endif
}


void SetPose::save(SerializationInterface &ar) const
{
    ar << m_target << m_x << m_y << m_z
       << m_yaw << m_roll << m_pitch
       << m_coord_sys;
}

void SetPose::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_x >> m_y >> m_z
       >> m_yaw >> m_roll >> m_pitch
       >> m_coord_sys;
}
